<?php

namespace Drupal\theme_change\Theme;

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Theme\ThemeNegotiatorInterface;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Path\PathMatcher;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\theme_change\Entity\ThemeChange;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class used for ThemeChangeswitcherNegotiator.
 */
class ThemeChangeswitcherNegotiator implements ThemeNegotiatorInterface {

  /**
   * The theme change handler.
   *
   * @var \Drupal\theme_change\Entity\ThemeChange
   */
  protected $themeChange;

  /**
   * The current path.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  protected $currentPath;

  /**
   * The current route.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $currentRoute;

  /**
   * The path matcher.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Constructs an ThemeChangeswitcherNegotiator object.
   */
  public function __construct(CurrentPathStack $current_path, PathMatcher $path_matcher, RouteMatchInterface $current_route_match, EntityTypeManagerInterface $entity_type_manager, AliasManagerInterface $aliasManager) {
    $this->currentPath = $current_path->getPath();
    $this->currentRoute = $current_route_match->getRouteName();
    $this->pathMatcher = $path_matcher;
    $this->entityTypeManager = $entity_type_manager;
    $this->aliasManager = $aliasManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('path.current'),
      $container->get('current_route_match'),
      $container->get('path.matcher'),
      $container->get('entity_type.manager'),
      $container->get('path_alias.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match) {
    if ($this->entityTypeManager->hasHandler('theme_change', 'list_builder')) {
      $storage = $this->entityTypeManager->getStorage('theme_change');
      $entities = $storage->loadMultiple();
      /** @var \Drupal\theme_change\Entity\ThemeChange $entity */
      foreach ($entities as $entity) {
        if ($this->check($entity)) {
          $this->themeChange = $entity;
          return TRUE;
        }
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function determineActiveTheme(RouteMatchInterface $route_match) {
    return $this->themeChange->getTheme();
  }

  /**
   * Implements to check theme change for path and route.
   */
  public function check(ThemeChange $entity) {
    if ($entity->getType() == 'path') {
      $current_path = $this->currentPath;
      $currentPathAlias = $this->aliasManager->getAliasByPath($current_path);
      $all_path = explode(',', $entity->getPath());
      foreach (array_map('trim', $all_path) as $path) {
        $path_match = $this->pathMatcher->matchPath($this->currentPath, $path);
        $path_alias_match = $this->pathMatcher->matchPath($currentPathAlias, $path);
        if ($path_match || $path_alias_match) {
          return TRUE;
        }
      }
    }
    elseif ($entity->getType() == 'route' && $this->currentRoute == $entity->getRoute()) {
      return TRUE;
    }
    return FALSE;
  }

}
