<?php

namespace Drupal\theme_file_editor\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Render\Markup;
use Drupal\Core\Theme\ThemeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\File\FileSystemInterface;

class ThemeFileEditorController extends ControllerBase
{

  /**
   * The theme handler service.
   *
   * @var \Drupal\Core\Theme\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Constructs a new ThemeFileEditorController.
   *
   * @param \Drupal\Core\Theme\ThemeHandlerInterface $theme_handler
   *   The theme handler service.
   */
  public function __construct(ThemeHandlerInterface $theme_handler)
  {
    $this->themeHandler = $theme_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('theme_handler')
    );
  }

  /**
   * Builds the page content with the Ace Editor and theme selection.
   *
   * @return array
   *   A render array for the theme file editor page.
   */
  public function edit(Request $request)
  {
    $theme_handler = \Drupal::service('theme_handler');
    $extension_list = \Drupal::service('extension.list.theme');
    $theme_options = [];
    $theme_files = [];

    // Get ALL themes (not just enabled ones)
    $all_themes = $theme_handler->rebuildThemeData();
    foreach ($all_themes as $theme_name => $theme_data) {
      // Skip hidden base themes
      if (!empty($theme_data->info['hidden'])) {
        continue;
      }

      $theme_path = $extension_list->getPath($theme_name);

      // Skip themes in core folder
      if (strpos($theme_path, 'core/themes') !== false) {
        continue;
      }

      $theme_options[$theme_name] = $theme_data->info['name'] ?? ucfirst($theme_name);
      $theme_files[$theme_name] = $this->getFilesInDirectory($theme_path);
    }

    $selected_theme = $request->get('theme') ?: array_key_first($theme_options);
    $selected_file = $request->get('file');

    $module_path = \Drupal::service('extension.list.module')->getPath('theme_file_editor');
    $ace_editor_url = base_path() . $module_path . '/ace-builds-master/src-min-noconflict/ace.js';
    $ace_base_url = base_path() . $module_path . '/ace-builds-master/src-min-noconflict';

    // Dropdown HTML for themes.
    $theme_dropdown = '<select id="theme-selector">';
    foreach ($theme_options as $theme_name => $theme_label) {
      $selected = $theme_name === $selected_theme ? ' selected' : '';
      $theme_dropdown .= '<option value="' . $theme_name . '"' . $selected . '>' . $theme_label . '</option>';
    }
    $theme_dropdown .= '</select>';

    $editor_markup = '
<div class="theme-editor-wrapper">
  <div class="theme-selection">
    <label for="theme-selector">Select Theme:</label>
    ' . $theme_dropdown . '
  </div>
  
  <div class="editor-content-wrapper">
    <div class="file-tree-panel">
      <div class="file-tree-header">Project Files</div>
      <div class="file-tree-container">
        ' . $this->renderFileTree($theme_files[$selected_theme], $selected_theme) . '
      </div>
    </div>
    
    <div class="editor-container">
      <div id="editor"></div>
      <button id="save-button">Save File</button>
    </div>
  </div>
</div>

<style>
.theme-editor-wrapper {
  display: flex;
  flex-direction: column;
  height: 80vh;
  gap: 15px;
}

.theme-selection {
  padding: 10px;
  background: #f5f5f5;
  border-radius: 4px;
}

.editor-content-wrapper {
  display: flex;
  flex: 1;
  min-height: 0;
  gap: 15px;
}

.file-tree-panel {
  width: 300px;
  display: flex;
  flex-direction: column;
  border: 1px solid #ddd;
  border-radius: 4px;
  overflow: hidden;
}

.file-tree-header {
  padding: 10px;
  background: #f5f5f5;
  border-bottom: 1px solid #ddd;
  font-weight: bold;
}

.file-tree-container {
  flex: 1;
  overflow-y: auto;
  padding: 10px;
}

.editor-container {
  flex: 1;
  display: flex;
  flex-direction: column;
  border: 1px solid #ddd;
  border-radius: 4px;
  overflow: hidden;
}

#editor {
  flex: 1;
  min-height: 0;
}

#save-button {
  margin: 10px;
  padding: 8px 16px;
  align-self: flex-start;
}

.file-tree {
  list-style: none;
  padding-left: 15px;
}

.file-tree li {
  margin: 5px 0;
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
}

.file-tree .folder {
  cursor: pointer;
}

.file-tree .folder-header {
  display: flex;
  align-items: center;
  gap: 5px;
}

.file-tree .file-link {
  color: #333;
  text-decoration: none;
}

.file-tree .file-link:hover {
  text-decoration: underline;
}

.file-tree .file-link.active {
  font-weight: bold;
  color: #0071b8;
}
</style>';

  // $editor_markup .= '<li><a href="#" class="file-link" data-file="' . $file . '" data-theme="' . $selected_theme . '">' . $file . '</a></li>';

    $build = [
      '#title' => $this->t('Edit Theme File'),
      '#markup' => Markup::create($editor_markup),
      '#attached' => [
        'library' => ['theme_file_editor/theme_file_editor'],
        'drupalSettings' => [
          'themeFileEditor' => [
            'editorID' => 'editor',
            'aceEditorUrl' => $ace_editor_url,
            'aceBaseUrl' => $ace_base_url,
            'themeOptions' => $theme_options,
            'selectedTheme' => $selected_theme,
            'selectedFile' => $selected_file,
            'themeFiles' => $theme_files,
          ],
        ],
      ],
    ];

    return $build;
  }
  private function renderFileTree(array $files, string $theme): string
  {
    $tree = [];
    foreach ($files as $filePath) {
      $parts = explode('/', str_replace('\\', '/', $filePath));
      $current = &$tree;
      foreach ($parts as $part) {
        if (!isset($current[$part])) {
          $current[$part] = [];
        }
        $current = &$current[$part];
      }
    }
    return $this->buildTreeHtml($tree, $theme);
  }

  private function buildTreeHtml(array $tree, string $theme, string $path = ''): string
  {
    $html = '<ul class="file-tree">';
    foreach ($tree as $name => $children) {
      $currentPath = $path ? $path . '/' . $name : $name;
      if (is_array($children) && !empty($children)) {
        // Folder with children - changed to not be expanded by default
        $html .= '<li class="folder">'; // Removed "expanded" class
        $html .= '<div class="folder-header">';
        $html .= '<i class="folder-icon fa fa-folder"></i>'; // Changed to regular folder icon
        $html .= '<span class="folder-name">' . htmlspecialchars($name) . '</span>';
        $html .= '</div>';
        $html .= $this->buildTreeHtml($children, $theme, $currentPath);
        $html .= '</li>';
      } else {
        // Single file
        $html .= '<li class="file">';
        $html .= '<a href="#" class="file-link" data-theme="' . htmlspecialchars($theme) . '" data-file="' . htmlspecialchars($currentPath) . '">';
        $html .= '<i class="file-icon fa fa-file"></i>';
        $html .= htmlspecialchars($name);
        $html .= '</a>';
        $html .= '</li>';
      }
    }
    $html .= '</ul>';
    return $html;
  }

  /**
   * Gets the file list for a theme in HTML format.
   */
  public function getFileList(Request $request)
  {
    $theme = $request->query->get('theme');
    if (!$theme) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Theme not provided.',
      ]);
    }

    $extension_list = \Drupal::service('extension.list.theme');
    $theme_path = $extension_list->getPath($theme);
    $files = $this->getFilesInDirectory($theme_path);

    $html = $this->renderFileTree($files, $theme);

    return new JsonResponse([
      'status' => 'success',
      'html' => $html,
    ]);
  }
  /**
   * Loads the content of a selected file.
   *
   * @param Request $request
   *   The current request object.
   *
   * @return JsonResponse
   *   A JSON response containing the file content or an error message.
   */
  public function loadFileContent(Request $request)
  {
    $theme = $request->query->get('theme');
    $file = $request->query->get('file');

    if (!$theme || !$file) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Theme or file not provided.',
      ]);
    }

    $extension_list = \Drupal::service('extension.list.theme');
    $theme_path = $extension_list->getPath($theme);
    $full_path = $theme_path . '/' . $file;

    // Normalize the path to handle different directory separators
    $full_path = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $full_path);

    if (!file_exists($full_path)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'File not found: ' . $full_path,
      ]);
    }

    if (!is_readable($full_path)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'File is not readable.',
      ]);
    }

    try {
      $content = file_get_contents($full_path);
      return new JsonResponse([
        'status' => 'success',
        'content' => $content,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Error reading file: ' . $e->getMessage(),
      ]);
    }
  }

  /**
   * Retrieves all files in the specified directory and its subdirectories.
   *
   * @param string $path
   *   The directory path to scan.
   * @param string $file
   *   The file to search for (optional).
   *
   * @return array
   *   An array of file names within the directory and its subdirectories.
   */
  private function getFilesInDirectory($path)
  {
    $files = [];
    if (file_exists($path) && is_dir($path)) {
      $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS),
        \RecursiveIteratorIterator::SELF_FIRST
      );

      $image_extensions = ['png', 'jpg', 'jpeg', 'gif', 'svg', 'webp', 'bmp', 'ico', 'tiff'];
      $media_extensions = ['mp3', 'mp4', 'mov', 'avi', 'wmv', 'flv', 'webm', 'ogg'];

      foreach ($iterator as $file) {
        if ($file->isFile()) {
          $extension = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));

          // Skip image and media files
          if (in_array($extension, array_merge($image_extensions, $media_extensions))) {
            continue;
          }

          $relativePath = str_replace(
            str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path) . DIRECTORY_SEPARATOR,
            '',
            str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $file->getPathname())
          );
          $files[] = $relativePath;
        }
      }
    }
    return $files;
  }


  /**
   * Saves the content of a selected file.
   */
  public function saveFileContent(Request $request)
  {
    try {
      $theme = $request->request->get('theme');
      $file = $request->request->get('file');
      $content = $request->request->get('content');

      // Validate all parameters exist
      if (empty($theme) || empty($file) || $content === null) {
        return new Response('', Response::HTTP_BAD_REQUEST);
      }

      $extension_list = \Drupal::service('extension.list.theme');
      $theme_path = $extension_list->getPath($theme);
      $full_path = $theme_path . '/' . $file;

      // Verify file exists and is writable
      if (!file_exists($full_path) || !is_writable($full_path)) {
        return new Response('', Response::HTTP_BAD_REQUEST);
      }

      // Save file content
      if (file_put_contents($full_path, $content) === false) {
        return new Response('', Response::HTTP_BAD_REQUEST);
      }

      // Return completely empty successful response
      return new Response('', Response::HTTP_OK);
    } catch (\Exception $e) {
      return new Response('', Response::HTTP_BAD_REQUEST);
    }
  }
}
