(function ($, Drupal, drupalSettings) {
  Drupal.behaviors.themeFileEditor = {
    attach: function (context, settings) {
      if (
        $("#editor", context).length &&
        !$("#editor").data("ace-initialized")
      ) {
        $("#editor").data("ace-initialized", true);
        let editor = null;

        $.when($.getScript(drupalSettings.themeFileEditor.aceEditorUrl))
          .then(function () {
            ace.config.set(
              "basePath",
              drupalSettings.themeFileEditor.aceBaseUrl
            );
            editor = ace.edit("editor");
            editor.setTheme("ace/theme/monokai");

            // Set default mode to PHP (will be updated when loading files)
            editor.session.setMode("ace/mode/php");

            initializeUI(editor);

            if (drupalSettings.themeFileEditor.selectedFile) {
              loadFileContent(
                editor,
                drupalSettings.themeFileEditor.selectedTheme,
                drupalSettings.themeFileEditor.selectedFile
              );
            }
          })
          .fail(function () {
            console.error("Failed to load Ace Editor");
            displayMessage(
              "Failed to load code editor. Please try again.",
              "error"
            );
          });

        function displayMessage(message, type) {
          const messageWrapper = Drupal.Message.defaultWrapper();
          const messageElement = $(
            Drupal.theme("message", { text: message }, type)
          );
          $(messageWrapper).prepend(messageElement);
          setTimeout(() => messageElement.remove(), 5000);
        }

        function getEditorModeForFile(filename) {
          const extension = filename.split(".").pop().toLowerCase();
          switch (extension) {
            case "css":
              return "ace/mode/css";
            case "js":
              return "ace/mode/javascript";
            case "yml":
            case "yaml":
              return "ace/mode/yaml";
            case "twig":
              return "ace/mode/twig";
            case "html":
              return "ace/mode/html";
            case "md":
              return "ace/mode/markdown";
            case "info":
              return "ace/mode/php"; // .info files are similar to PHP for syntax purposes
            default:
              return "ace/mode/php"; // Default to PHP for other files
          }
        }

        function initializeUI(editor) {
          $(document).on("change", "#theme-selector", function () {
            const theme = $(this).val();
            updateThemeFiles(theme);
          });

          $(document).on("click", ".folder-header", function (e) {
            e.stopPropagation();
            const $folder = $(this).parent();
            $folder.toggleClass("expanded");
            $folder.find("> ul").stop(true, true).slideToggle(200);
            const $icon = $(this).find(".folder-icon");
            $icon.toggleClass("fa-folder fa-folder-open");
          });

  $(".folder-header").each(function () {
    const $folder = $(this).parent();
    const $icon = $(this).find(".folder-icon");
    $icon.addClass("fa-folder"); // Default to closed folder
    $folder.removeClass("expanded");
    $folder.find("> ul").hide(); // Hide child elements by default
  });

          $(document).on("click", ".file-link", function (e) {
            e.preventDefault();
            loadFileContent(
              editor,
              $(this).data("theme"),
              $(this).data("file")
            );
          });

          $("#save-button").on("click", function () {
            const activeFile = $(".file-link.active");
            if (activeFile.length) {
              saveFileContent(
                activeFile.data("theme"),
                activeFile.data("file"),
                editor.getValue()
              );
            } else {
              displayMessage("Please select a file first.", "error");
            }
          });
        }

        function updateThemeFiles(theme) {
          const $fileTreeContainer = $(".file-tree-container");
          $fileTreeContainer.html(
            '<div class="loading-message">Loading files...</div>'
          );

          const url = new URL(window.location);
          url.searchParams.set("theme", theme);
          window.history.pushState({}, "", url);

          $.ajax({
            url: Drupal.url("theme-file-editor/file-list"),
            method: "GET",
            data: { theme: theme },
            success: function (response) {
              if (response.status === "success") {
                $fileTreeContainer.html(response.html);
                editor.setValue("");
                $(".file-link").removeClass("active");
                $(".folder-header").each(function () {
                  const $folder = $(this).parent();
                  const $icon = $(this).find(".toggle-icon");
                  $icon.text($folder.hasClass("expanded") ? "−" : "+");
                });
              }
            },
          });
        }

        function loadFileContent(editor, theme, filePath) {
          if (!theme || !filePath) return;

          $(".file-link").removeClass("active");
          $(`.file-link[data-file="${filePath}"]`).addClass("active");

          // Set the editor mode based on file extension
          const mode = getEditorModeForFile(filePath);
          editor.session.setMode(mode);

          $.ajax({
            url: Drupal.url("theme-file-editor/load-file-content"),
            method: "GET",
            data: { theme: theme, file: filePath },
            success: function (response) {
              if (response.status === "success") {
                editor.setValue(response.content, -1);
                const url = new URL(window.location);
                url.searchParams.set("file", filePath);
                window.history.pushState({}, "", url);
              }
            },
          });
        }

        function saveFileContent(theme, filePath, content) {
          const $saveButton = $("#save-button");
          $saveButton.prop("disabled", true).text("Saving...");
          $(".messages").remove();
          const csrfToken = drupalSettings.ajaxPageState.token;

          $.ajax({
            url: Drupal.url("theme-file-editor/save-file-content"),
            method: "POST",
            data: {
              theme: theme,
              file: filePath,
              content: content,
              token: csrfToken,
            },
            success: function () {
              $saveButton.prop("disabled", false).text("Save File");
              const messageMarkup = Drupal.theme(
                "message",
                { text: "File saved successfully." },
                "status"
              );
              $(Drupal.Message.defaultWrapper())
                .prepend(messageMarkup)
                .find(".messages")
                .delay(5000)
                .fadeOut("slow", function () {
                  $(this).remove();
                });
            },
            error: function () {
              $saveButton.prop("disabled", false).text("Save File");
              const messageMarkup = Drupal.theme(
                "message",
                { text: "File Saved Successfully." },
                "error"
              );
              $(Drupal.Message.defaultWrapper())
                .prepend(messageMarkup)
                .find(".messages")
                .delay(5000)
                .fadeOut("slow", function () {
                  $(this).remove();
                });
            },
          });
        }
      }
    },
  };
})(jQuery, Drupal, drupalSettings);
