(function (Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.thinProgressBar = {
    attach: function (context, settings) {
      const config = drupalSettings.thinProgressBar;

      if (!config || !config.enableOnPageLoad) {
        console.log('Thin Progress Bar: Module disabled in configuration');
        return;
      }

      // Use once() with the Drupal once API
      const progressBars = once('thin-progress-bar', '.thin-progress-bar', context);

      let progressBar;
      if (progressBars.length === 0) {
        // Create progress bar element
        progressBar = document.createElement('div');
        progressBar.className = 'thin-progress-bar';

        // Add thickness class if configured
        if (config.thickness === 'ultra-thin') {
          progressBar.classList.add('ultra-thin');
          console.log('Thin Progress Bar: Ultra-thin mode enabled');
        } else {
          console.log('Thin Progress Bar: Thin mode (2px) enabled');
        }

        document.body.prepend(progressBar);

        // Set custom color if provided
        if (config.color) {
          document.documentElement.style.setProperty('--thin-progress-color', config.color);
          console.log('Thin Progress Bar: Custom color set to', config.color);
        }

        console.log('Thin Progress Bar: Progress bar element created');
      } else {
        progressBar = progressBars[0];
        console.log('Thin Progress Bar: Existing progress bar found');
      }

      // Smart page load detection - only show for slow loads
      this.setupSmartPageLoad(progressBar, config);

      // Handle AJAX requests if enabled
      if (config.enableOnAjax) {
        console.log('Thin Progress Bar: AJAX handlers enabled');
        this.setupAjaxHandlers(progressBar, config.animationSpeed);
      } else {
        console.log('Thin Progress Bar: AJAX handlers disabled');
      }
    },

    setupSmartPageLoad: function (progressBar, config) {
      const slowLoadThreshold = config.slowLoadThreshold || 800; // milliseconds
      let pageLoadTimer;
      let pageLoaded = false;

      console.log('Thin Progress Bar: Setting up smart page load detection');
      console.log('Thin Progress Bar: Slow load threshold set to', slowLoadThreshold + 'ms');

      // Start timer when page starts loading
      pageLoadTimer = setTimeout(() => {
        if (!pageLoaded) {
          // Page is taking too long, show progress bar
          console.log('Thin Progress Bar: Page load is slow, showing progress bar');
          this.startProgress(progressBar, config.animationSpeed);
        } else {
          console.log('Thin Progress Bar: Page already loaded before threshold');
        }
      }, slowLoadThreshold);

      console.log('Thin Progress Bar: Page load timer started');

      // Hide progress bar if page loads before threshold
      window.addEventListener('load', () => {
        pageLoaded = true;
        clearTimeout(pageLoadTimer);
        console.log('Thin Progress Bar: Window load event fired');

        // If progress bar was shown, complete it quickly
        if (progressBar.classList.contains('loading')) {
          console.log('Thin Progress Bar: Completing progress bar after page load');
          progressBar.style.width = '100%';
          setTimeout(() => {
            progressBar.classList.remove('loading');
            progressBar.classList.add('complete');
            console.log('Thin Progress Bar: Progress bar marked as complete');
            setTimeout(() => {
              progressBar.style.opacity = '0';
              console.log('Thin Progress Bar: Progress bar hidden');
            }, 300);
          }, 100);
        } else {
          console.log('Thin Progress Bar: Page loaded quickly, progress bar was never shown');
        }
      });

      // Also handle DOMContentLoaded for faster perceived loading
      document.addEventListener('DOMContentLoaded', () => {
        console.log('Thin Progress Bar: DOM content loaded');
      });
    },

    startProgress: function (progressBar, speed) {
      const duration = this.getDuration(speed);
      let width = 0;

      console.log('Thin Progress Bar: Starting progress animation with duration', duration + 'ms');

      progressBar.classList.remove('complete');
      progressBar.classList.add('loading');
      progressBar.style.opacity = '1';
      progressBar.style.width = '0%';

      const intervalTime = 30;
      const increments = 100 / (duration / intervalTime);

      console.log('Thin Progress Bar: Progress interval started');

      const interval = setInterval(() => {
        if (width >= 100) {
          clearInterval(interval);
          progressBar.classList.remove('loading');
          progressBar.classList.add('complete');
          console.log('Thin Progress Bar: Progress reached 100%, stopping animation');

          // Fade out after completion
          setTimeout(() => {
            progressBar.style.opacity = '0';
            console.log('Thin Progress Bar: Progress bar faded out');
          }, 800);
        } else {
          width += increments;
          if (width > 100) width = 100;
          progressBar.style.width = width + '%';

          // Log progress every 25%
          if (Math.round(width) % 25 === 0 && Math.round(width) !== 0) {
            console.log('Thin Progress Bar: Progress at', Math.round(width) + '%');
          }
        }
      }, intervalTime);
    },

    getDuration: function (speed) {
      let duration;
      switch (speed) {
        case 'fast':
          duration = 1200;
          break;
        case 'slow':
          duration = 4000;
          break;
        case 'normal':
        default:
          duration = 2500;
          break;
      }
      console.log('Thin Progress Bar: Animation speed set to', speed, '(', duration + 'ms)');
      return duration;
    },

    setupAjaxHandlers: function (progressBar, speed) {
      console.log('Thin Progress Bar: Setting up AJAX event listeners');

      // Use Drupal AJAX events
      document.addEventListener('ajaxSend', (event) => {
        // Check if this is not a progress-enabled request
        const ajaxSettings = event.detail.ajax ? event.detail.ajax.settings : {};
        if (ajaxSettings && ajaxSettings.progress === undefined) {
          console.log('Thin Progress Bar: AJAX request started, showing progress bar');
          progressBar.style.width = '0%';
          progressBar.style.opacity = '1';
          progressBar.classList.remove('complete');
          progressBar.classList.add('loading');
          this.startProgress(progressBar, speed);
        } else {
          console.log('Thin Progress Bar: AJAX request with custom progress handler, skipping');
        }
      });

      document.addEventListener('ajaxComplete', () => {
        console.log('Thin Progress Bar: AJAX request completed');
        progressBar.style.width = '100%';
        setTimeout(() => {
          progressBar.classList.remove('loading');
          progressBar.classList.add('complete');
          console.log('Thin Progress Bar: AJAX progress bar marked as complete');
          setTimeout(() => {
            progressBar.style.opacity = '0';
            console.log('Thin Progress Bar: AJAX progress bar hidden');
          }, 800);
        }, 100);
      });

      document.addEventListener('ajaxError', (event) => {
        console.log('Thin Progress Bar: AJAX request failed');
        // Hide progress bar on error
        progressBar.style.opacity = '0';
        progressBar.classList.remove('loading', 'complete');
      });
    }
  };

})(Drupal, drupalSettings);
