<?php

declare(strict_types=1);

namespace Drupal\time_based_condition\Plugin\Condition;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Condition\ConditionPluginBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\time_based_condition\TimeframeServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Timebased' condition.
 *
 * @Condition(
 *   id = "timebased",
 *   label = @Translation("Timebased"),
 * )
 */
final class Timebased extends ConditionPluginBase implements ContainerFactoryPluginInterface, CacheableDependencyInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
  $plugin_id,
    $plugin_definition,
    private readonly TimeframeServiceInterface $timeframeService
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('time_based_condition.timeframe_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return array_merge(parent::defaultConfiguration(), [
      'date_from' => NULL,
      'date_to' => NULL,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['date_from'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Date from'),
      '#default_value' => $this->configuration['date_from'] ? DrupalDateTime::createFromTimestamp($this->configuration['date_from']) : NULL,
    ];

    $form['date_to'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Date to'),
      '#default_value' => $this->configuration['date_to'] ? DrupalDateTime::createFromTimestamp($this->configuration['date_to']) : NULL,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    return $this->timeframeService->applies($this->configuration['date_from'], $this->configuration['date_to']);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    if (!$form_state->isValueEmpty('date_from') && !$form_state->getValue('date_from') instanceof DrupalDateTime) {
      $form_state->setErrorByName('date_from', $this->t('The date from must be a valid date format.'));
    }

    if (!$form_state->isValueEmpty('date_to') && !$form_state->getValue('date_to') instanceof DrupalDateTime) {
      $form_state->setErrorByName('date_to', $this->t('The date from must be a valid date format.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['date_from'] = $form_state->getValue('date_from') instanceof DrupalDateTime ? $form_state->getValue('date_from')->getTimestamp() : NULL;
    $this->configuration['date_to'] = $form_state->getValue('date_to') instanceof DrupalDateTime ? $form_state->getValue('date_to')->getTimestamp() : NULL;
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    // Provide a summary of the condition configuration.
    return $this->t('Timebased condition: from @date_from to @date_to', [
      '@date_from' => $this->configuration['date_from'] ?? '',
      '@date_to' => $this->configuration['date_to'] ?? '',
    ]);
  }

  /**
   *
   */
  public function getCacheMaxAge(): int {
    $timestamp_from = $this->configuration['date_from'] ?? NULL;
    $timestamp_to = $this->configuration['date_to'] ?? NULL;

    // If no dates are set, use a default cache max age or return Cache::PERMANENT.
    if (empty($timestamp_from) && empty($timestamp_to)) {
      return parent::getCacheMaxAge();
    }

    return $this->timeframeService->computeCacheMaxAge($timestamp_from, $timestamp_to);
  }

}
