<?php

declare(strict_types=1);

namespace Drupal\time_based_condition;

use Drupal\Component\Datetime\TimeInterface;

/**
 * A service for the computation of the timeframe.
 */
class TimeframeService implements TimeframeServiceInterface {

  public function __construct(private readonly TimeInterface $time) {
  }

  /**
   * {@inheritdoc}
   */
  public function applies(int|null $from_timestamp, int|null $to_timestamp): bool {
    $current_time = $this->time->getCurrentTime();

    if (empty($from_timestamp) && empty($to_timestamp)) {
      return TRUE;
    }

    if (empty($to_timestamp) && !empty($from_timestamp)) {
      return $current_time >= $from_timestamp;
    }

    if (empty($from_timestamp) && !empty($to_timestamp)) {
      return $current_time <= $to_timestamp;
    }

    return $current_time >= $from_timestamp && $current_time <= $to_timestamp;
  }

  /**
   * {@inheritdoc}
   */
  public function computeCacheMaxAge(int|null $timestamp_from, int|null $timestamp_to): int {
    $current_time = $this->time->getCurrentTime();

    // If only date_from is set, calculate the max age based on date_from.
    if (empty($timestamp_to) && !empty($timestamp_from)) {
      $max_age = $timestamp_from - $current_time;
      // Ensure max age is not negative.
      return max($max_age, 0);
    }

    // If only date_to is set, calculate the max age based on date_to.
    if (empty($timestamp_from) && !empty($timestamp_to)) {
      $max_age = $timestamp_to - $current_time;
      // Ensure max age is not negative.
      return max($max_age, 0);
    }

    // If both date_from and date_to are set, calculate the max age based on date_to.
    $max_age = $timestamp_to - $current_time;
    // Ensure max age is not negative.
    return max($max_age, 0);
  }

}
