<?php

declare(strict_types=1);

namespace Drupal\Tests\time_based_condition\Unit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\time_based_condition\TimeframeService;

/**
 * @coversDefaultClass \Drupal\time_based_condition\TimeframeService
 * @group time_based_condition
 */
class TimeBasedConditionTest extends UnitTestCase {

  /**
   * The mocked time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  private $timeMock;

  /**
   * The timeframe service.
   *
   * @var \Drupal\time_based_condition\TimeframeService
   */
  private $timeframeService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->timeMock = $this->createMock(TimeInterface::class);
    $this->timeframeService = new TimeframeService($this->timeMock);
  }

  /**
   * Tests the time-based condition.
   *
   * @covers ::applies
   */
  public function testTimeBasedCondition(): void {
    $this->timeMock->expects($this->once())
      ->method('getCurrentTime')
      ->willReturn(1320966000);

    $from_date = 1349820000;
    $to_date = 1355266800;

    $this->assertFalse($this->timeframeService->applies($from_date, $to_date));
  }

  /**
   * Tests the cache max age calculation with only from timestamp.
   *
   * @covers ::computeCacheMaxAge
   */
  public function testComputeCacheMaxAgeWithFromTimestamp(): void {
    $current_time = 1320966000;
    $this->timeMock->method('getCurrentTime')->willReturn($current_time);

    $from_timestamp = 1355266800;
    $expected_max_age = $from_timestamp - $current_time;

    $this->assertEquals($expected_max_age, $this->timeframeService->computeCacheMaxAge($from_timestamp, NULL));
  }

  /**
   * Tests the cache max age calculation with only to timestamp.
   *
   * @covers ::computeCacheMaxAge
   */
  public function testComputeCacheMaxAgeWithToTimestamp(): void {
    $current_time = 1355266800;
    $this->timeMock->method('getCurrentTime')->willReturn($current_time);

    $to_timestamp = 1381206000;
    $expected_max_age = $to_timestamp - $current_time;

    $this->assertEquals($expected_max_age, $this->timeframeService->computeCacheMaxAge(NULL, $to_timestamp));
  }

  /**
   * Tests the cache max age calculation with both timestamps.
   *
   * @covers ::computeCacheMaxAge
   */
  public function testComputeCacheMaxAgeWithBothTimestamps(): void {
    $current_time = 1320966000;
    $this->timeMock->method('getCurrentTime')->willReturn($current_time);

    $from_timestamp = 1349820000;
    $to_timestamp = 1355266800;
    $expected_max_age = $to_timestamp - $current_time;

    $this->assertEquals($expected_max_age, $this->timeframeService->computeCacheMaxAge($from_timestamp, $to_timestamp));
  }

  /**
   * Tests the cache max age calculation with no timestamps.
   *
   * @covers ::computeCacheMaxAge
   */
  public function testComputeCacheMaxAgeWithNoTimestamps(): void {
    $this->timeMock->method('getCurrentTime')->willReturn(1320966000);

    $this->assertEquals(0, $this->timeframeService->computeCacheMaxAge(NULL, NULL));
  }

}
