<?php

namespace Drupal\tmgmt_asymmetric_block\Service;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Entity\TranslatableRevisionableStorageInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\entity_reference_revisions\EntityNeedsSaveInterface;
use Drupal\layout_builder\LayoutEntityHelperTrait;
use Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionComponent;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt_asymmetric_block\Events\TmgmtFinishedEvent;
use Drupal\tmgmt_content\FieldProcessorInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Provides a service for handling job block on finished.
 */
class TmgmtJobBlockJobService implements TmgmtJobBlockServiceInterface {

  use StringTranslationTrait;
  use LayoutEntityHelperTrait;

  /**
   * The mapped uuids.
   *
   * @var array<string, string>
   */
  protected array $mappedUuids = [];

  /**
   * The target langcode.
   *
   * @var string
   */
  protected string $targetLangcode;

  /**
   * The source entity.
   *
   * @var \Drupal\Core\Entity\ContentEntityInterface
   */
  protected ContentEntityInterface $sourceEntity;

  /**
   * The job.
   *
   * @var \Drupal\tmgmt\JobInterface
   */
  protected JobInterface $job;

  /**
   * The constructor to handle the services.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager interface.
   * @param \Drupal\Core\TempStore\SharedTempStoreFactory $sharedTempStore
   *   The Shared temp store factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactor
   *   The logger factory.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID interface.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected SharedTempStoreFactory $sharedTempStore,
    protected LoggerChannelFactoryInterface $loggerFactor,
    protected UuidInterface $uuid,
    protected ConfigFactoryInterface $configFactory,
    protected EventDispatcherInterface $eventDispatcher,
  ) {
  }

  /**
   * {@inheritDoc}
   */
  public function onFinished(JobInterface $job): void {
    $this->targetLangcode = $job->getTargetLanguage()->getId();

    // Get all the job items.
    $job_items = $job->getItems();
    $this->job = $job;
    $source_job_item = array_shift($job_items);
    /** @var \Drupal\Core\Entity\ContentEntityInterface */
    /** @var \Drupal\Core\Entity\ContentEntityStorageInterface */
    $storage = $this->entityTypeManager->getStorage($source_job_item->getItemType());
    $entity = $storage->load($source_job_item->getItemId());
    $this->sourceEntity = $this->getLatestRevisionEntity($entity);
    if (!$this->sourceEntity) {
      return;
    }
    // Lets replace any fields that are translable on the entity, excluding the
    // layout builder section.
    $translated_entity = $this->translateEntityFields(
      $this->sourceEntity, $source_job_item->getData(),
      $job->getTargetLanguage()->getId(),
      $source_job_item
    );

    $event = new TmgmtFinishedEvent($this->job);
    $event->setTranslatedEntity($translated_entity);
    $event->setSourceEntity($this->sourceEntity);
    $event->setJobItems($job_items);
    $this->eventDispatcher->dispatch($event, TmgmtFinishedEvent::TRANSLATE_FIELDS);
    $translated_entity = $event->getTranslatedEntity();

    // For each items, look for the translation within the layout builder
    // section component to see if it matches, if so create that block content.
    if ($this->isLayoutCompatibleEntity($translated_entity)) {
      $updated_sections = [];
      /** @var \Drupal\layout_builder\Field\LayoutSectionItemList $layout_builder */
      $layout_builder = $this->sourceEntity->get(OverridesSectionStorage::FIELD_NAME);
      $sections = $layout_builder->getSections();
      foreach ($job_items as $delta => $job_item) {
        if ($job_item->getPlugin() === 'section_layout') {
          $updated_sections = $this->updateSections($sections, $job_item);
          if ($updated_sections) {
            $translated_entity->set(OverridesSectionStorage::FIELD_NAME, $updated_sections);
            array_splice($job_items, $delta, 1);
            break;
          }
        }
      }
      // If there is no translated sections, clone the original sections
      // and use that.
      if (empty($updated_sections)) {
        $updated_sections = $this->cloneSections($sections);
      }
      $this->updateSectionsComponents($updated_sections, $job_items);
      $this->updateLayoutReferenceUuid($updated_sections, $this->mappedUuids);
    }

    if (!empty($updated_sections)) {
      $translated_entity->set(OverridesSectionStorage::FIELD_NAME, $updated_sections);
    }
    $this->save($translated_entity);
  }

  /**
   * {@inheritDoc}
   */
  public function getLatestRevisionEntity(ContentEntityInterface $entity): ContentEntityInterface {
    /** @var \Drupal\Core\Entity\TranslatableRevisionableStorageInterface|\Drupal\Core\Entity\RevisionableStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
    $latest_revision_id = $storage->getLatestRevisionId($entity->id());
    if ($latest_revision_id && $latest_revision_id != $entity->getRevisionId()) {
      /** @var \Drupal\Core\Entity\RevisionableInterface $entity */
      $entity = $storage->loadRevision($latest_revision_id);
    }
    return $entity;
  }

  /**
   * {@inheritDoc}
   */
  public function save(ContentEntityInterface $entity): void {
    $entity_type = $entity->getEntityTypeId();
    /** @var \Drupal\Core\Entity\TranslatableRevisionableStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage($entity_type);

    $translated_entity = $storage->createRevision($entity, FALSE);
    /** @var \Drupal\Core\Entity\ContentEntityInterface $translated_entity */
    $moderation_info = \Drupal::service('content_moderation.moderation_information');
    $workflow = $moderation_info->getWorkflowForEntity($this->sourceEntity);
    $moderation_state = $this->configFactory->get('tmgmt_content.settings')->get('default_moderation_states.' . $workflow->id());
    if ($moderation_state) {
      $translated_entity->set('moderation_state', $moderation_state);
    }

    if ($translated_entity instanceof RevisionLogInterface) {
      $translated_entity->setRevisionLogMessage($this->t('The job id: @job_id has been translated for @target_langcode.', [
        '@job_id' => $this->job->id(),
        '@target_langcode' => $this->targetLangcode,
      ]));
    }
    $translated_entity->setNewRevision(TRUE);
    $translated_entity->save();

    // Since content in layout is being updated in code, there will not be
    // a revert option, we want to cleanup any stale tempstore for that entity.
    $collection = 'layout_builder.section_storage.overrides';
    $key = implode('.', [
      $translated_entity->getEntityTypeId(),
      $translated_entity->id(),
      // The view mode to clear.
      'default',
      $this->targetLangcode,
    ]);
    $tempstore = $this->sharedTempStore->get($collection, $key);
    if ($tempstore) {
      $tempstore->delete($key);
    }

    // Provide a link to /latest if source node is published and the
    // translated node is in draft state.
    /** @var \Drupal\Node\NodeInterface $node */
    if ($entity->getEntityTypeId() === 'node') {
      // Use the normal link for now, canonical route for latest revision does
      // not allow for language code to be pass in.
      $language_prefix = $this->configFactory->get('language.negotiation')->get('url.prefixes');
      $link_suffix = '/edit';
      if ($this->isLayoutCompatibleEntity($entity)) {
        $link_suffix = '/layout';
      }
      $this->job->addMessage('Latest revision can be view at this
        <a href="@url">link</a>.', [
          '@url' => '/' . $language_prefix[$this->targetLangcode]
          . '/node/' . $entity->id()
          . $link_suffix,
        ]
      );
    }
  }

  /**
   * {@inheritDoc}
   */
  public function updateLayoutReferenceUuid(array $sections): void {
    $uuid = $this->mappedUuids;
    foreach ($sections as $section) {
      $settings = $section->getLayoutSettings();
      $update = FALSE;
      if (!empty($settings['sub_section']['section_uid'])) {
        $translated_uuid = $settings['sub_section']['section_uid'];
        if (isset($uuid[$translated_uuid])) {
          $settings['sub_section']['section_uid'] = $uuid[$translated_uuid];
          $update = TRUE;
        }
      }
      // Take into account Red Hat Page builder layouts with children subset.
      if (isset($settings['children'])) {
        foreach ($settings['children'] as $region => $items) {
          foreach ($items as $delta => $item) {
            if (isset($uuid[$item])) {
              $settings['children'][$region][$delta] = $uuid[$item];
              $update = TRUE;
            }
          }
        }
      }
      if ($update) {
        $section->setLayoutSettings($settings);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function updateSectionsComponents(array $sections, array $job_items): void {
    $event = new TmgmtFinishedEvent($this->job);
    foreach ($sections as $section) {
      // Clone the components so that when inserting a new components does not
      // affect array.
      $immutable_section = clone $section;
      $components = $immutable_section->getComponents();
      $event->setBlockComponents($section, $components, $job_items);
      $event->setMappedUuids($this->mappedUuids);
      $this->eventDispatcher->dispatch($event, TmgmtFinishedEvent::UPDATE_SECTION_COMPONENTS);
    }
    $this->mappedUuids = $event->getMappedUuids();
  }

  /**
   * {@inheritDoc}
   */
  public function createComponent(ContentEntityInterface $entity, JobItemInterface $job_item): ContentEntityInterface {
    $entity_type = $job_item->getItemType();
    $block_storage = $this->entityTypeManager->getStorage($entity_type);
    $data = $job_item->getData();

    $values = [
      'data' => $data['data'][0]['value']['#translation']['#text'] ?? $entity->get('data')->value,
      'info' => $entity->label(),
      'pattern_id' => $entity->get('pattern_id')->value,
      'published' => TRUE,
      'reusable' => $entity->get('reusable')->value,
      'langcode' => $this->targetLangcode,
    ];
    if ($entity->getEntityTypeId() === 'patternkit_block') {
      $values['pattern_id'] = $entity->get('pattern_id')->value;
    }
    else {
      $values['id'] = $entity->id();
    }
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $new_entity = $block_storage->create($values);

    return $new_entity;
  }

  /**
   * {@inheritdoc}
   */
  public function removeComponent(Section $section, SectionComponent $component) {
    $section->removeComponent($component->getUuid());
  }

  /**
   * {@inheritDoc}
   */
  public function cloneSections(array $sections): array {
    $cloneSections = [];
    foreach ($sections as $delta => $section) {
      $cloneSections[$delta] = clone $section;
    }
    return $cloneSections;
  }

  /**
   * {@inheritDoc}
   */
  public function updateSections(
    array $sections,
    JobItemInterface $job_item,
  ): array {
    $data = $job_item->getData();
    $cloneSections = [];
    foreach ($sections as $delta => $section) {
      $cloneSections[$delta] = clone $section;
      // Check if the section exists for that data.
      if (isset($data[$delta])) {
        $layoutSettings = $cloneSections[$delta]->getLayoutSettings();
        foreach ($data[$delta] as $key => $translated_data) {
          // Skip the grid portion and have it process below.
          if ($key === 'grid') {
            continue;
          }
          if (!empty($layoutSettings[$key]) && !empty($translated_data['#translation']['#text'])) {
            $layoutSettings[$key] = $translated_data['#translation']['#text'];
            $updateLabel = TRUE;
          }
        }
        if (!empty($layoutSettings['grid'])) {
          $updateLabel = FALSE;
          // Update the section labels if any.
          // Update the grid/tabs.
          foreach ($layoutSettings['grid'] as &$grid) {
            if (!empty($grid['detail']['name'])) {
              $key = $grid['detail']['name'];
              if (!empty($data[$delta]['region'][$key]['#translate'])) {
                $grid['detail']['label'] = $data[$delta]['region'][$key]['#translation']['#text'];
                $updateLabel = TRUE;
              }
            }
          }
          if (isset($layoutSettings['section_uid'])) {
            // Update the uuid and save the section.
            $uuid = $this->uuid->generate();
            $this->mappedUuids[$layoutSettings['section_uid']] = $uuid;
            $layoutSettings['section_uid'] = $uuid;
            $updateLabel = TRUE;
          }
          if ($updateLabel) {
            $cloneSections[$delta]->setLayoutSettings($layoutSettings);
          }
        }
      }
    }
    $event = new TmgmtFinishedEvent($this->job);
    $event->setSections($cloneSections);
    $this->eventDispatcher->dispatch($event, TmgmtFinishedEvent::UPDATE_SECTIONS);
    $cloneSections = $event->getSections();
    return $cloneSections;
  }

  /**
   * Translat any of the entity fields that are translable before the save.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity for which the translation should be saved.
   * @param array $data
   *   The translation data for the fields.
   * @param string $target_langcode
   *   The target language.
   * @param \Drupal\tmgmt\JobItemInterface $item
   *   The job item.
   * @param bool $save
   *   (optional) Whether to save the translation or not.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The translation entity.
   *
   * @throws \Exception
   *   Thrown when a field or field offset is missing.
   */
  protected function translateEntityFields(
    ContentEntityInterface $entity,
    array $data,
    string $target_langcode,
    JobItemInterface $item,
    bool $save = TRUE,
  ): ContentEntityInterface {
    // If the translation for this language does not exist yet, initialize it.
    if (!$entity->hasTranslation($target_langcode)) {
      $entity->addTranslation($target_langcode, $entity->toArray());
    }

    /** @var \Drupal\Core\Entity\ContentEntityInterface $translation */
    $translation = $entity->getTranslation($target_langcode);

    // Create a revision if the entity is going to be saved. Embedded entities
    // will have a revision created later on.
    if ($save && $entity->getEntityType()->isRevisionable()) {
      /** @var \Drupal\Core\Entity\TranslatableRevisionableStorageInterface $storage */
      $storage = \Drupal::entityTypeManager()->getStorage($entity->getEntityTypeId());

      if ($storage instanceof TranslatableRevisionableStorageInterface) {
        // Always create a new revision of the translation.
        /** @var \Drupal\Core\Entity\ContentEntityInterface $translation */
        $translation = $storage->createRevision($translation, $translation->isDefaultRevision());
      }
    }

    $manager = \Drupal::service('content_translation.manager');
    if ($manager->isEnabled($translation->getEntityTypeId(), $translation->bundle())) {
      $manager->getTranslationMetadata($translation)->setSource($entity->language()->getId());
    }

    if ($translation instanceof RevisionLogInterface) {
      $translation->setRevisionLogMessage($this->t('Created by translation job <a href=":url">@label</a>.', [
        ':url' => $item->getJob()->toUrl()->toString(),
        '@label' => $item->label(),
      ]));
      $translation->setRevisionUserId(\Drupal::currentUser()->id());
      $translation->setRevisionCreationTime(\Drupal::time()->getRequestTime());
    }

    foreach (Element::children($data) as $field_name) {
      $field_data = $data[$field_name];

      if (!$translation->hasField($field_name)) {
        $message = 'Skipping field %field for job item @item because it does not exist on entity <em>@type/@id</em>.';
        $item_label = $translation->hasLinkTemplate('canonical') ? $translation->toLink()->toString() : $translation->label();
        $args = [
          '%field' => $field_name,
          '@item' => $item_label,
          '@type' => $translation->getEntityTypeId(),
          '@id' => $translation->id(),
        ];
        $item->addMessage($message, $args, 'warning');
        continue;
      }

      $field = $translation->get($field_name);
      $field_processor = $this->getFieldProcessor($field->getFieldDefinition()->getType());
      $field_processor->setTranslations($field_data, $field);
    }

    $embeddable_fields = static::getEmbeddableFields($entity);
    foreach ($embeddable_fields as $field_name => $field_definition) {

      if (!isset($data[$field_name])) {
        continue;
      }

      $field = $translation->get($field_name);
      $target_type = $field->getFieldDefinition()->getFieldStorageDefinition()->getSetting('target_type');
      $is_target_type_translatable = $manager->isEnabled($target_type);
      // In case the target type is not translatable, the referenced entity will
      // be duplicated. As a consequence, remove all the field items from the
      // translation, update the field value to use the field object from the
      // source language.
      if (!$is_target_type_translatable) {
        $field = clone $entity->get($field_name);

        if (!$translation->get($field_name)->isEmpty()) {
          $translation->set($field_name, NULL);
        }
      }

      foreach (Element::children($data[$field_name]) as $delta) {
        $field_item = $data[$field_name][$delta];
        foreach (Element::children($field_item) as $property) {
          // Find the referenced entity. In case we are dealing with
          // untranslatable target types, the source entity will be returned.
          /** @var \Drupal\entity_reference_revisions\EntityNeedsSaveInterface|null $target_entity */
          if ($target_entity = $this->findReferencedEntity($field, $field_item, $delta, $property, $is_target_type_translatable)) {
            if ($is_target_type_translatable) {
              // If the field is an embeddable reference and the property is a
              // content entity, process it recursively.
              // If the field is ERR and the target entity supports
              // the needs saving interface, do not save it immediately to avoid
              // creating two versions when content moderation is used but just
              // ensure it will be saved.
              $target_save = TRUE;
              if ($field->getFieldDefinition()->getType() == 'entity_reference_revisions' && $target_entity instanceof EntityNeedsSaveInterface) {
                $target_save = FALSE;
                $target_entity->setNeedsSave(TRUE);
              }

              $this->translateEntityFields($target_entity, $field_item[$property], $target_langcode, $item, $target_save);
            }
            else {
              $duplicate = $this->createTranslationDuplicate($target_entity, $target_langcode);
              // Do not save the duplicate as it's going to be saved with the
              // main entity.
              $this->translateEntityFields($duplicate, $field_item[$property], $target_langcode, $item, FALSE);
              $translation->get($field_name)->set($delta, $duplicate);
            }
          }
        }
      }
    }

    if (static::isModeratedEntity($translation)) {
      // Use the given moderation status if set. Otherwise, fallback to the
      // configured one in TMGMT settings.
      if (isset($data['#moderation_state'][0])) {
        $moderation_state = $data['#moderation_state'][0];
      }
      else {
        $moderation_info = \Drupal::service('content_moderation.moderation_information');
        $workflow = $moderation_info->getWorkflowForEntity($entity);
        $moderation_state = \Drupal::config('tmgmt_content.settings')->get('default_moderation_states.' . $workflow->id());
      }
      if ($moderation_state) {
        $translation->set('moderation_state', $moderation_state);
      }
    }
    // Otherwise, try to set a published status.
    elseif (isset($data['#published'][0]) && $translation instanceof EntityPublishedInterface) {
      if ($data['#published'][0]) {
        $translation->setPublished();
      }
      else {
        $translation->setUnpublished();
      }
    }

    return $translation;
  }

  /**
   * Creates a translation duplicate of the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $target_entity
   *   The target entity to clone.
   * @param string $langcode
   *   Language code for all the clone entities created.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   New entity object with the data from the original entity. Not
   *   saved. No sub-entities are cloned.
   */
  public function createTranslationDuplicate(ContentEntityInterface $target_entity, string $langcode): ContentEntityInterface {
    $duplicate = $target_entity->createDuplicate();

    // Change the original language.
    if ($duplicate->getEntityType()->hasKey('langcode')) {
      $duplicate->set($duplicate->getEntityType()->getKey('langcode'), $langcode);
    }

    return $duplicate;
  }

  /**
   * {@inheritDoc}
   */
  public static function isModeratedEntity(EntityInterface $entity): bool {
    if (!\Drupal::moduleHandler()->moduleExists('content_moderation')) {
      return FALSE;
    }

    return \Drupal::service('content_moderation.moderation_information')->isModeratedEntity($entity);
  }

  /**
   * {@inheritDoc}
   */
  public static function getEmbeddableFields(ContentEntityInterface $entity): array {
    // Get the configurable embeddable references.
    $field_definitions = $entity->getFieldDefinitions();
    $embeddable_field_names = \Drupal::config('tmgmt_content.settings')->get('embedded_fields');
    $embeddable_fields = array_filter($field_definitions, function (FieldDefinitionInterface $field_definition) use ($embeddable_field_names) {
      return isset($embeddable_field_names[$field_definition->getTargetEntityTypeId()][$field_definition->getName()]);
    });

    // Get always embedded references.
    $content_translation_manager = \Drupal::service('content_translation.manager');
    foreach ($field_definitions as $field_name => $field_definition) {
      $storage_definition = $field_definition->getFieldStorageDefinition();

      $property_definitions = $storage_definition->getPropertyDefinitions();
      foreach ($property_definitions as $property_definition) {
        // Look for entity_reference properties where the storage definition
        // has a target type setting.
        if (in_array($property_definition->getDataType(), ['entity_reference', 'entity_revision_reference']) && ($target_type_id = $storage_definition->getSetting('target_type'))) {
          $is_target_type_enabled = $content_translation_manager->isEnabled($target_type_id);
          $target_entity_type = \Drupal::entityTypeManager()->getDefinition($target_type_id);

          // Include current entity reference field that is considered a
          // composite and translatable or if the parent entity is considered a
          // composite as well. This allows to embed nested untranslatable
          // fields (For example: Paragraphs).
          if ($target_entity_type->get('entity_revision_parent_type_field') && ($is_target_type_enabled || $entity->getEntityType()->get('entity_revision_parent_type_field'))) {
            $embeddable_fields[$field_name] = $field_definition;
          }
        }
      }
    }

    return $embeddable_fields;
  }

  /**
   * Returns the field processor for a given field type.
   *
   * @param string $field_type
   *   The field type.
   *
   * @return \Drupal\tmgmt_content\FieldProcessorInterface
   *   The field processor for this field type.
   */
  protected function getFieldProcessor(string $field_type): FieldProcessorInterface {
    $definition = \Drupal::service('plugin.manager.field.field_type')->getDefinition($field_type);

    return \Drupal::service('class_resolver')->getInstanceFromDefinition($definition['tmgmt_field_processor']);
  }

  /**
   * Finds the referenced entity.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $field
   *   The field.
   * @param array $field_item
   *   The field item.
   * @param int $delta
   *   The delta.
   * @param string $property
   *   The property.
   * @param bool $is_target_type_translatable
   *   (optional) Whether the target entity type is translatable.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|null
   *   The referenced entity.
   */
  public function findReferencedEntity(
    FieldItemListInterface $field,
    array $field_item,
    int $delta,
    string $property,
    bool $is_target_type_translatable = TRUE,
  ): ?ContentEntityInterface {
    // If an id is provided, loop over the field item deltas until we find the
    // matching entity. In case of untranslatable target types return the
    // source target entity as it will be duplicated.
    if (isset($field_item[$property]['#id'])) {
      foreach ($field as $item_delta => $item) {
        if ($item->$property instanceof ContentEntityInterface) {
          /** @var \Drupal\Core\Entity\ContentEntityInterface $referenced_entity */
          $referenced_entity = $item->$property;
          if ($referenced_entity->id() == $field_item[$property]['#id'] || ($item_delta === $delta && !$is_target_type_translatable)) {
            return $referenced_entity;
          }
        }
      }

      // @todo Support loading an entity, throw an exception or log a warning?
    }
    // For backwards compatiblity, also support matching based on the delta.
    elseif ($field->offsetExists($delta) && $field->offsetGet($delta)->$property instanceof ContentEntityInterface) {
      return $field->offsetGet($delta)->$property;
    }
    return NULL;
  }

}
