<?php

namespace Drupal\tmgmt_asymmetric_block\Service;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;

/**
 * The translation service interface for the content.
 *
 * @package Drupal\tmgmt_asymmetric_block\Service
 */
interface TmgmtJobBlockServiceInterface {

  /**
   * Handles the job block on finished.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The job interface.
   */
  public function onFinished(JobInterface $job): void;

  /**
   * Saves the entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity interface.
   */
  public function save(ContentEntityInterface $entity): void;

  /**
   * Update the sections base on data provided from the job item.
   *
   * @param array<int,\Drupal\layout_builder\Section> $sections
   *   The layout builder sections.
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item interface.
   *
   * @return array<int,\Drupal\layout_builder\Section>
   *   Return an array of sections.
   */
  public function updateSections(array $sections, JobItemInterface $job_item): array;

  /**
   * Remap the sub section uuid with the new one.
   *
   * @param array<int, \Drupal\layout_builder\Section> $sections
   *   The updated layout sections with new changes.
   */
  public function updateLayoutReferenceUuid(array $sections): void;

  /**
   * Update the sections components.
   *
   * @param array<int, \Drupal\layout_builder\Section> $sections
   *   The sections that has the sextions translated already but is still using
   *   the a copy of the componet from the source language.
   * @param array<int, \Drupal\tmgmt\JobItemInterface> $job_items
   *   The job items.
   */
  public function updateSectionsComponents(array $sections, array $job_items): void;

  /**
   * Create a component.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The created entity.
   */
  public function createComponent(ContentEntityInterface $entity, JobItemInterface $job_item): ContentEntityInterface;

  /**
   * Determines whether an entity is moderated.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if the entity is moderated. Otherwise, FALSE.
   */
  public static function isModeratedEntity(EntityInterface $entity): bool;

  /**
   * Returns fields that should be embedded into the data for the given entity.
   *
   * Includes explicitly enabled fields and composite entities that are
   * implicitly included to the translatable data.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to get the translatable data from.
   *
   * @return array<int, \Drupal\Core\Field\FieldDefinitionInterface> $embeddable_fields
   *   A list of field definitions that can be embedded.
   */
  public static function getEmbeddableFields(ContentEntityInterface $entity): array;

  /**
   * Creates a translation duplicate of the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $target_entity
   *   The target entity.
   * @param string $langcode
   *   The language code.
   */
  public function createTranslationDuplicate(ContentEntityInterface $target_entity, string $langcode): ContentEntityInterface;

  /**
   * Finds a referenced entity.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $field
   *   The field.
   * @param array $field_item
   *   The field item.
   * @param int $delta
   *   The delta.
   * @param string $property
   *   The property.
   * @param bool $is_target_type_translatable
   *   Whether the target type is translatable.
   */
  public function findReferencedEntity(
    FieldItemListInterface $field,
    array $field_item,
    int $delta,
    string $property,
    bool $is_target_type_translatable = TRUE,
  ): ?ContentEntityInterface;

  /**
   * Clone the sections.
   *
   * @param array $sections
   *   The sections to clone.
   *
   * @return array<int, \Drupal\layout_builder\Section>
   *   The cloned sections.
   */
  public function cloneSections(array $sections): array;

  /**
   * Get the latest revision entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   */
  public function getLatestRevisionEntity(ContentEntityInterface $entity): ContentEntityInterface;

}
