<?php

declare(strict_types=1);

namespace Drupal\tmgmt_auto_translate;

use Drupal\content_moderation\ModerationInformationInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\node\NodeInterface;

/**
 * Manages automatic translation of nodes based on configuration.
 */
final class TmgmtAutoTranslateManager {

  use StringTranslationTrait;

  /**
   * The auto translate configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a TmgmtAutoTranslateManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $loggerChannel
   *   The logger channel.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\content_moderation\ModerationInformationInterface|null $moderationInfo
   *   The moderation information service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MessengerInterface $messenger,
    protected LoggerChannelInterface $loggerChannel,
    protected ModuleHandlerInterface $moduleHandler,
    ConfigFactoryInterface $config_factory,
    protected ?ModerationInformationInterface $moderationInfo,
  ) {
    $this->config = $config_factory->get('tmgmt_auto_translate.settings');
  }

  /**
   * Processes a node for automatic translation based on configuration.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to be processed.
   */
  public function process(NodeInterface $node): void {
    // Skip if not default translation.
    if (!$node->isDefaultTranslation()) {
      return;
    }

    $bundle = $node->bundle();

    // Check if this node's bundle is configured for auto translation.
    if (!in_array($node->bundle(), ($this->config->get('bundles') ?: []))) {
      return;
    }

    // If content_moderation is enabled, check for moderation state states.
    if ($this->moduleHandler->moduleExists('content_moderation') && $this->moderationInfo->isModeratedEntity($node)) {
      // Get the configured state for this bundle.
      $moderation_state_id = $this->config->get("bundle_settings.$bundle.moderation_state") ?: [];

      // Check if a state changed.
      $state_id = $this->moderationInfo->getWorkflowForEntity($node)
        ?->getTypePlugin()->getState($node->moderation_state->value)->id();

      // Check if the state matches the configured one.
      if ($state_id === $moderation_state_id) {
        $this->translateNode($node);
        return;
      }
    }

    // If no content_moderation or the entity is not moderated,
    // check if it's published.
    if ($node->isPublished()) {
      // Node is published, so translate it.
      $this->translateNode($node);
    }
  }

  /**
   * Translates a given node into configured target languages.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to be translated.
   */
  protected function translateNode(NodeInterface $node): void {
    $bundle = $node->bundle();

    // Get bundle-specific settings or fall back to global settings.
    $bundle_settings = $this->config->get('bundle_settings.' . $bundle) ?: [];

    // Get available languages.
    $source_langcode = $node->language()->getId();

    // Get selected target languages from bundle-specific config
    // or fall back to global settings.
    $target_languages = $bundle_settings['target_languages'] ?? [];

    // Define job and job_item storages.
    $job_storage = $this->entityTypeManager->getStorage('tmgmt_job');
    $job_item_storage = $this->entityTypeManager->getStorage('tmgmt_job_item');

    // Create a job for each target language.
    foreach ($target_languages as $langcode => $language_settings) {
      if (
        // Skip if language disabled.
        empty($language_settings['value']) ||
        // Skip the source language.
        $langcode === $source_langcode ||
        // Skip if a translation already exists.
        $node->hasTranslation($langcode)
      ) {
        continue;
      }

      $provider = $bundle_settings['target_languages'][$langcode]['provider'] ?? NULL;

      if (empty($provider)) {
        $message = $this->t('No translation provider configured for automatic translation of @type.', ['@type' => $bundle]);
        $this->messenger->addError($message);
        $this->loggerChannel->error($message);
        return;
      }

      // Create a translation job.
      /** @var \Drupal\tmgmt\Entity\Job $job */
      $job = $job_storage->create([
        'source_language' => $source_langcode,
        'target_language' => $langcode,
        'translator' => $provider,
        'settings' => [],
        'uid' => $node->getOwnerId(),
      ]);

      $job->save();

      // Add the node to the job.
      /** @var \Drupal\tmgmt\Entity\JobItem $job_item */
      $job_item = $job_item_storage->create([
        'tjid' => $job->id(),
        'plugin' => 'content',
        'item_type' => 'node',
        'item_id' => $node->id(),
      ]);

      $job_item->save();

      // Request translation.
      if ($job->canRequestTranslation()) {
        $job->requestTranslation();
        $this->loggerChannel->notice('Automatic translation requested for node @id to @language using @provider.', [
          '@id' => $node->id(),
          '@language' => $langcode,
          '@provider' => $provider,
        ]);
      }
      else {
        $this->loggerChannel->error('Could not request automatic translation for node @id to @language using @provider.', [
          '@id' => $node->id(),
          '@language' => $langcode,
          '@provider' => $provider,
        ]);
      }
    }
  }

}
