<?php

namespace Drupal\tmgmt_deepl_glossary;

use DeepL\AppInfo;
use DeepL\DeepLException;
use DeepL\GlossaryEntries;
use DeepL\GlossaryInfo;
use DeepL\Translator as DeepLTranslator;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\TranslatorInterface;
use Psr\Log\LoggerInterface;

/**
 * A service for managing DeepL glossary API calls.
 */
class DeeplGlossaryApi implements DeeplGlossaryApiInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The translator.
   *
   * @var \Drupal\tmgmt\TranslatorInterface
   */
  protected TranslatorInterface $translator;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The Drupal logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Constructs a new DeeplGlossaryApi.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The Messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The Drupal logger.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger, LoggerChannelFactoryInterface $logger_factory, KeyRepositoryInterface $key_repository) {
    // @codeCoverageIgnoreStart
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->logger = $logger_factory->get('tmgmt_deepl_glossary');
    $this->keyRepository = $key_repository;
    // @codeCoverageIgnoreEnd
  }

  /**
   * {@inheritdoc}
   */
  public function setTranslator(TranslatorInterface $translator): void {
    $this->translator = $translator;
  }

  /**
   * {@inheritdoc}
   */
  public function getTranslator(): DeepLTranslator {
    if (isset($this->translator)) {
      $key_id = $this->translator->getSetting('auth_key_entity');
      if ($key_id != '') {
        $key_entity = $this->keyRepository->getKey($key_id);
        assert($key_entity instanceof KeyInterface);

        // @codeCoverageIgnoreStart
        $api_key = $key_entity->getKeyValue();
        $app_info = new AppInfo('tmgmt_deepl_glossary', '2.3.x');
        return new DeepLTranslator($api_key, ['app_info' => $app_info]);
        // @codeCoverageIgnoreEnd
      }
    }

    throw new \RuntimeException('Translator not set.');
  }

  /**
   * {@inheritdoc}
   */
  public function createGlossary(string $name, string $source_lang, string $target_lang, array $entries): ?GlossaryInfo {
    try {
      $deepl_translator = $this->getTranslator();
      $entries = GlossaryEntries::fromEntries($entries);
      return $deepl_translator->createGlossary(trim($name), $source_lang, $target_lang, $entries);
    }
    catch (DeepLException $e) {
      $this->messenger->addMessage($e->getMessage(), 'error');
      $this->logger->error($e);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getGlossaries(): array {
    try {
      $deepl_translator = $this->getTranslator();
      return $deepl_translator->listGlossaries();
    }
    catch (DeepLException $e) {
      $this->messenger->addMessage($e->getMessage(), 'error');
      $this->logger->error($e);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteGlossary(string $glossary_id): bool {
    try {
      $deepl_translator = $this->getTranslator();
      $deepl_translator->deleteGlossary($glossary_id);
      return TRUE;
    }
    catch (DeepLException $e) {
      $this->messenger->addMessage($e->getMessage(), 'error');
      $this->logger->error($e);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getGlossaryEntries(string $glossary_id): array {
    try {
      $deepl_translator = $this->getTranslator();
      $entries = $deepl_translator->getGlossaryEntries($glossary_id);
      return $entries->getEntries();
    }
    catch (DeepLException $e) {
      $this->messenger->addMessage($e->getMessage(), 'error');
      $this->logger->error($e);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createGlossaryFromCsv(string $name, string $source_lang, string $target_lang, string $csv_file_path): ?GlossaryInfo {
    try {
      // Read CSV data from the file.
      $csvData = file_get_contents($csv_file_path);
      if ($csvData === FALSE) {
        throw new \RuntimeException("Unable to read CSV file: $csv_file_path");
      }

      // Create the glossary using the DeepL API.
      $deepl_translator = $this->getTranslator();
      return $deepl_translator->createGlossaryFromCsv($name, $source_lang, $target_lang, $csvData);
    }
    catch (DeepLException | \Exception $e) {
      $this->messenger->addMessage($e->getMessage(), 'error');
      $this->logger->error($e);
      return NULL;
    }
  }

}
