<?php

namespace Drupal\tmgmt_deepl_glossary;

use DeepL\GlossaryInfo;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslator;

/**
 * A service for managing DeepL glossary API batch.
 */
class DeeplGlossaryBatch implements DeeplGlossaryBatchInterface {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * The DeepL glossary service.
   *
   * @var \Drupal\tmgmt_deepl_glossary\DeeplGlossaryApi
   */
  protected DeeplGlossaryApi $deeplGlossaryApi;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The messenger interface.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The glossary storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $glossaryStorage;

  /**
   * The DeepL glossary helper service.
   *
   * @var \Drupal\tmgmt_deepl_glossary\DeeplGlossaryHelperInterface
   */
  protected DeeplGlossaryHelperInterface $glossaryHelper;

  /**
   * Constructs a new DeeplGlossaryApi.
   *
   * @param \Drupal\tmgmt_deepl_glossary\DeeplGlossaryApi $deepl_glossary_api
   *   The DeepL glossary service.
   * @param \Drupal\tmgmt_deepl_glossary\DeeplGlossaryHelperInterface $glossary_helper
   *   The DeepL glossary helper service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger interface.
   */
  public function __construct(DeeplGlossaryApi $deepl_glossary_api, DeeplGlossaryHelperInterface $glossary_helper, EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger) {
    // @codeCoverageIgnoreStart
    $this->deeplGlossaryApi = $deepl_glossary_api;
    $this->glossaryHelper = $glossary_helper;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;

    // Get glossary storage.
    $this->glossaryStorage = $this->entityTypeManager->getStorage('deepl_glossary');
    // @codeCoverageIgnoreEnd
  }

  /**
   * {@inheritDoc}
   */
  public function buildBatch(): void {
    // Get all available translators.
    $deepl_translators = $this->getDeeplTranslators();
    if (count($deepl_translators) > 0) {
      $batch_builder = $this->createBatchBuilder();
      $this->addBatchOperations($batch_builder, $deepl_translators);
      $this->setBatch($batch_builder->toArray());
    }
  }

  /**
   * {@inheritDoc}
   */
  public function syncOperation(TranslatorInterface $translator, GlossaryInfo $glossary, array $entries, array &$context): void {
    // Prepare glossary entries.
    $glossary_entries = [];
    foreach ($entries as $subject => $definition) {
      $glossary_entries[] = [
        'subject' => $subject,
        'definition' => $definition,
      ];
    }

    // Save or update glossary.
    $this->glossaryHelper->saveGlossary($glossary, $glossary_entries, $translator);

    // Add context message.
    $context['message'] = $this->formatPlural($glossary->entryCount, 'Syncing glossary @glossary_name with @entry_count entry.', 'Syncing glossary @glossary_name with @entry_count entries.', [
      '@glossary_name' => $glossary->name,
      '@entry_count' => $glossary->entryCount,
    ]);

    if (!isset($context['results']) || !is_array($context['results'])) {
      $context['results'] = [];
    }
    // Ensure 'glossaries' key within 'results' exists and is an array.
    if (!isset($context['results']['glossaries']) || !is_array($context['results']['glossaries'])) {
      // Initialize as an empty array (list).
      $context['results']['glossaries'] = [];
    }

    // Add context results.
    $context['results']['glossaries'][] = [
      'name' => $glossary->name,
      'entry_count' => $glossary->entryCount,
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function cleanUpOperation(array $deepl_glossaries, string $translator, array &$context): void {
    // Get glossary_ids.
    $deepl_glossary_ids = array_column($deepl_glossaries, 'glossaryId');
    $glossary_entities = $this->glossaryStorage->loadByProperties(['tmgmt_translator' => $translator]);

    /** @var \Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface $glossary_entity */
    foreach ($glossary_entities as $glossary_entity) {
      // Delete non-matching glossary entities.
      $glossary_id = $glossary_entity->get('glossary_id')->value;
      if ((isset($glossary_id) && !in_array($glossary_id, $deepl_glossary_ids, TRUE))) {
        $glossary_entity->delete();
      }
      // Delete glossaries without glossary_id.
      // This could be caused by an error in the creation process.
      elseif (!isset($glossary_id)) {
        $glossary_entity->delete();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function finishedOperation(bool $success, array $results, array $operations): void {
    if ($success) {
      // Glossaries were found and synced.
      if (isset($results['glossaries']) && is_array($results['glossaries']) && count($results['glossaries']) > 0) {
        $this->messenger->addStatus($this->t('DeepL glossaries were synced successfully.'));
      }
      else {
        $message = $this->t('Could not find any glossary for syncing.');
        $this->messenger->addWarning($message);
      }
    }
    else {
      $this->messenger->addError($this->t('An error occurred while syncing glossaries.'));
    }
  }

  /**
   * Add batch operations for translating items.
   *
   * @param \Drupal\Core\Batch\BatchBuilder $batch_builder
   *   The batch builder.
   * @param array $deepl_translators
   *   Array of available deepl_translators.
   */
  protected function addBatchOperations(BatchBuilder $batch_builder, array $deepl_translators): void {
    // Get all available translators.
    // Get all available glossaries (by translator).
    foreach (array_keys($deepl_translators) as $deepl_translator) {
      /** @var \Drupal\tmgmt\TranslatorInterface $translator */
      $translator = $this->entityTypeManager->getStorage('tmgmt_translator')
        ->load($deepl_translator);
      $glossary_api = $this->deeplGlossaryApi;
      // Set active translator.
      $glossary_api->setTranslator($translator);
      // Get all glossaries.
      $glossaries = $glossary_api->getGlossaries();

      /** @var \DeepL\GlossaryInfo $glossary */
      foreach ($glossaries as $glossary) {
        $glossary_entries = $glossary_api->getGlossaryEntries($glossary->glossaryId);
        // Add sync operation.
        if ($glossary->ready && $glossary->entryCount > 0) {
          $batch_builder->addOperation([$this, 'syncOperation'], [
            $translator,
            $glossary,
            $glossary_entries,
          ]);
        }
      }
      // Add cleanup operation.
      $batch_builder->addOperation([$this, 'cleanUpOperation'], [
        $glossaries,
        $translator->id(),
      ]);
    }
  }

  /**
   * Create and configure the batch builder.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The batch builder.
   */
  protected function createBatchBuilder(): BatchBuilder {
    return (new BatchBuilder())
      ->setTitle('Syncing DeepL glossaries')
      ->setFinishCallback([$this, 'finishedOperation'])
      ->setInitMessage('Initializing sync');
  }

  /**
   * Set the batch.
   *
   * @param array $batch
   *   The batch operations array.
   */
  protected function setBatch(array $batch): void {
    // @codeCoverageIgnoreStart
    batch_set($batch);
    // @codeCoverageIgnoreEnd
  }

  /**
   * Get all available translators.
   */
  protected function getDeeplTranslators(): array {
    // @codeCoverageIgnoreStart
    return DeeplTranslator::getTranslators();
    // @codeCoverageIgnoreEnd
  }

}
