<?php

namespace Drupal\tmgmt_deepl_glossary;

use DeepL\GlossaryInfo;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\TranslatorInterface;

/**
 * Helper class for DeepL Glossary functionality.
 */
class DeeplGlossaryHelper implements DeeplGlossaryHelperInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The glossary storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $glossaryStorage;

  /**
   * Constructs a new DeeplGlossaryHelper object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager) {
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;

    // Get glossary storage.
    $this->glossaryStorage = $this->entityTypeManager->getStorage('deepl_glossary');
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedLanguages(): array {
    $allowed_languages = [
      'AR' => $this->t('Arabic'),
      'BG' => $this->t('Bulgarian'),
      'CS' => $this->t('Czech'),
      'DA' => $this->t('Danish'),
      'DE' => $this->t('German'),
      'EL' => $this->t('Greek'),
      'EN' => $this->t('English (all variants)'),
      'ES' => $this->t('Spanish'),
      'ET' => $this->t('Estonian'),
      'FI' => $this->t('Finnish'),
      'FR' => $this->t('French'),
      'HU' => $this->t('Hungarian'),
      'ID' => $this->t('Indonesian'),
      'IT' => $this->t('Italian'),
      'JA' => $this->t('Japanese'),
      'KO' => $this->t('Korean'),
      'LT' => $this->t('Lithuanian'),
      'LV' => $this->t('Latvian'),
      'NB' => $this->t('Norwegian (Bokmål)'),
      'NL' => $this->t('Dutch'),
      'PL' => $this->t('Polish'),
      'PT' => $this->t('Portuguese (all variants)'),
      'RO' => $this->t('Romanian'),
      'RU' => $this->t('Russian'),
      'SK' => $this->t('Slovak'),
      'SL' => $this->t('Slovenian'),
      'SV' => $this->t('Swedish'),
      'TR' => $this->t('Turkish'),
      'UK' => $this->t('Ukrainian'),
      'ZH' => $this->t('Chinese (all variants)'),
    ];

    // Allow alteration of allowed languages.
    $this->moduleHandler->alter('tmgmt_deepl_glossary_allowed_languages', $allowed_languages);
    assert(is_array($allowed_languages));

    return $allowed_languages;
  }

  /**
   * {@inheritdoc}
   */
  public function getValidSourceTargetLanguageCombinations(): array {
    $languages = array_keys($this->getAllowedLanguages());
    $combinations = [];
    foreach ($languages as $lang1) {
      foreach ($languages as $lang2) {
        // Avoid duplicate pairs.
        if ($lang1 !== $lang2) {
          $combinations[] = [$lang1 => $lang2];
        }
      }
    }

    return $combinations;
  }

  /**
   * {@inheritdoc}
   */
  public function getMatchingGlossaries(string $translator, string $source_lang, string $target_lang): array {
    $deepl_glossary_storage = $this->entityTypeManager->getStorage('deepl_glossary');
    // Fix language mappings for complex language codes.
    $source_lang = $this->fixLanguageMappings($source_lang);
    $target_lang = $this->fixLanguageMappings($target_lang);

    return $deepl_glossary_storage->loadByProperties([
      'tmgmt_translator' => $translator,
      'source_lang' => $source_lang,
      'target_lang' => $target_lang,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function fixLanguageMappings(string $langcode): string {
    $language_mapping = [
      'EN-GB' => 'EN',
      'EN-US' => 'EN',
      'PT-BR' => 'PT',
      'PT-PT' => 'PT',
      'ZH-HANS' => 'ZH',
      'ZH-HANT' => 'ZH',
    ];

    if (isset($language_mapping[strtoupper($langcode)])) {
      return $language_mapping[strtoupper($langcode)];
    }

    return strtoupper($langcode);
  }

  /**
   * {@inheritdoc}
   */
  public function validateUniqueGlossary(array &$form, FormStateInterface $form_state, ?DeeplGlossaryInterface $entity = NULL): void {
    $user_input = $form_state->getUserInput();
    $translator = $user_input['tmgmt_translator'] ?? '';
    $source_lang = $user_input['source_lang'] ?? '';
    $target_lang = $user_input['target_lang'] ?? '';
    $translator = Translator::load($translator);

    // Check tmgmt_deepl_glossary settings.
    /** @var array $tmgmt_deepl_glossary_settings */
    $tmgmt_deepl_glossary_settings = ($translator instanceof TranslatorInterface) ? $translator->getSetting('tmgmt_deepl_glossary') : [];
    if (isset($tmgmt_deepl_glossary_settings['allow_multiple']) && $tmgmt_deepl_glossary_settings['allow_multiple'] === 0) {
      assert($translator instanceof TranslatorInterface);
      $existing_glossaries = $this->entityTypeManager->getStorage('deepl_glossary')
        ->loadByProperties([
          'tmgmt_translator' => $translator->id(),
          'source_lang' => $source_lang,
          'target_lang' => $target_lang,
        ]);

      // In case saving existing entity, we need to remove entity before
      // checking for duplicates.
      if (($entity instanceof DeeplGlossaryInterface && !$entity->isNew()) && isset($existing_glossaries[$entity->id()])) {
        unset($existing_glossaries[$entity->id()]);
      }

      // Show error, if we find existing glossary for source/ target language.
      if (count($existing_glossaries) >= 1) {
        $message = $this->t('You cannot add more than one glossary for the selected source/ target language combination.');
        $form_state->setErrorByName('source_lang', $message);
        $form_state->setErrorByName('target_lang', $message);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateSourceTargetLanguage(array &$form, FormStateInterface $form_state): void {
    $user_input = $form_state->getValues();

    // Get language codes.
    $source_lang = $this->getLanguageCode($user_input['source_lang']);
    $target_lang = $this->getLanguageCode($user_input['target_lang']);
    $source_lang = $source_lang != '' ? $source_lang : '';
    $target_lang = $target_lang != '' ? $target_lang : '';

    // Define valid language pairs.
    $valid_language_pairs = $this->getValidSourceTargetLanguageCombinations();

    // Get valid match for source/ target language..
    $match = FALSE;
    foreach ($valid_language_pairs as $valid_language_pair) {
      if ((is_array($valid_language_pair) && isset($valid_language_pair[$source_lang])) && ($valid_language_pair[$source_lang] == $target_lang)) {
        $match = TRUE;
      }
    }

    // If we don't find a valid math, set error to fields.
    if (!$match) {
      $message = $this->t('Select a valid source/ target language.');
      $form_state->setErrorByName('source_lang', $message);
      $form_state->setErrorByName('target_lang', $message);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function saveGlossary(GlossaryInfo $glossary, array $glossary_entries, TranslatorInterface $translator): void {
    // Load deepl_glossary entities (if available).
    $existing_glossaries = $this->glossaryStorage->loadByProperties(['glossary_id' => $glossary->glossaryId]);
    if (count($existing_glossaries) > 0) {
      // Update glossary entries for existing glossary.
      $existing_glossary = reset($existing_glossaries);
      if ($existing_glossary instanceof DeeplGlossaryInterface) {
        $existing_glossary->set('entries', $glossary_entries);
        $existing_glossary->save();
      }
    }
    else {
      // Add new deepl_glossary entity with entries.
      $this->glossaryStorage->create(
        [
          'label' => $glossary->name,
          'glossary_id' => $glossary->glossaryId,
          'source_lang' => strtoupper($glossary->sourceLang),
          'target_lang' => strtoupper($glossary->targetLang),
          'tmgmt_translator' => $translator->id(),
          'ready' => $glossary->ready,
          'entries' => $glossary_entries,
          'entries_format' => 'tsv',
          'entry_count' => $glossary->entryCount,
        ]
      )->save();
    }
  }

  /**
   * Helper function to retrieve language code from form state.
   *
   * @param mixed $input
   *   The input to extract the language code from.
   *
   * @return string
   *   The language code.
   */
  protected function getLanguageCode(mixed $input): string {
    if (is_array($input)) {
      if (is_array($input[0]) && isset($input[0]['value'])) {
        return is_string($input[0]['value']) ? $input[0]['value'] : '';
      }
      if (isset($input['value'])) {
        return is_string($input['value']) ? $input['value'] : '';
      }
    }
    // Handle simple string structure.
    return is_string($input) ? $input : '';
  }

}
