<?php

namespace Drupal\tmgmt_deepl_glossary;

use DeepL\GlossaryInfo;
use Drupal\Core\Form\FormStateInterface;
use Drupal\tmgmt\TranslatorInterface;

/**
 * Helper class for DeepL Glossary functionality.
 */
interface DeeplGlossaryHelperInterface {

  /**
   * Get allowed languages for DeepL Glossary.
   *
   * @return array
   *   An array of allowed languages.
   */
  public function getAllowedLanguages(): array;

  /**
   * Get valid source-target language combinations.
   *
   * @return array
   *   An array of valid language combinations.
   */
  public function getValidSourceTargetLanguageCombinations(): array;

  /**
   * Get matching glossaries for given translator and languages.
   *
   * @param string $translator
   *   The translator ID.
   * @param string $source_lang
   *   The source language code.
   * @param string $target_lang
   *   The target language code.
   *
   * @return array
   *   An array of matching DeepL glossary entities.
   */
  public function getMatchingGlossaries(string $translator, string $source_lang, string $target_lang): array;

  /**
   * Fix language mappings for DeepL Glossary.
   *
   * @param string $langcode
   *   The language code to fix.
   *
   * @return string
   *   The fixed language code.
   */
  public function fixLanguageMappings(string $langcode): string;

  /**
   * Validate unique glossary for source/ target language within forms.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param \Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface|null $entity
   *   The DeeplGlossary entity object.
   */
  public function validateUniqueGlossary(array &$form, FormStateInterface $form_state, ?DeeplGlossaryInterface $entity): void;

  /**
   * Validate valid source/ target language pair.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function validateSourceTargetLanguage(array &$form, FormStateInterface $form_state): void;

  /**
   * Save or update glossary.
   *
   * @param \DeepL\GlossaryInfo $glossary
   *   The glossary info object.
   * @param array $glossary_entries
   *   Array of glossary entries.
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator.
   */
  public function saveGlossary(GlossaryInfo $glossary, array $glossary_entries, TranslatorInterface $translator): void;

}
