<?php

namespace Drupal\Tests\tmgmt_deepl_glossary\Unit\Controller;

use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl_glossary\Controller\DeeplGlossaryDownloadController;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface;
use Drupal\tmgmt_deepl_glossary\Entity\DeeplGlossary;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the DeeplGlossaryDownloadController.
 */
#[CoversClass(DeeplGlossaryDownloadController::class)]
#[Group('tmgmt_deepl_glossary')]
class DeeplGlossaryDownloadControllerTest extends UnitTestCase {

  /**
   * The mocked glossary API.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject|\Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface
   */
  protected MockObject|DeeplGlossaryApiInterface $glossaryApi;

  /**
   * The controller under test.
   *
   * @var \Drupal\tmgmt_deepl_glossary\Controller\DeeplGlossaryDownloadController
   */
  protected DeeplGlossaryDownloadController $controller;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->glossaryApi = $this->createMock(DeeplGlossaryApiInterface::class);
    $this->controller = new DeeplGlossaryDownloadController($this->glossaryApi);
  }

  /**
   * Tests the methods ::downloadCsv, ::buildCsv, ::serveCsv.
   */
  public function testDownloadCsvSuccess(): void {
    // Mock the translator.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the glossary entity.
    $glossary = $this->createMock(DeeplGlossary::class);
    $glossary->expects($this->once())
      ->method('getTranslator')
      ->willReturn($translator);

    // Set up glossary ID expectation - called only once.
    $glossary_id = 'test-glossary-123';
    $glossary->expects($this->atLeastOnce())
      ->method('getGlossaryId')
      ->willReturn($glossary_id);
    $glossary->expects($this->once())
      ->method('label')
      ->willReturn('Test Glossary');

    // Mock the field values.
    $source_lang_field = (object) ['value' => 'en'];
    $target_lang_field = (object) ['value' => 'de'];

    $glossary->expects($this->exactly(2))
      ->method('get')
      ->willReturnMap([
        ['source_lang', $source_lang_field],
        ['target_lang', $target_lang_field],
      ]);

    // Mock the glossary API response.
    $entries = [
      'Hello' => 'Hallo',
      'World' => 'Welt',
    ];

    $this->glossaryApi->expects($this->once())
      ->method('setTranslator')
      ->with($translator);
    $this->glossaryApi->expects($this->once())
      ->method('getGlossaryEntries')
      ->with('test-glossary-123')
      ->willReturn($entries);

    // Test the controller.
    $response = $this->controller->downloadCsv($glossary);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals(
      'attachment; filename="Test_Glossary_en_to_de.csv"',
      $response->headers->get('Content-Disposition')
    );
    $this->assertEquals(
      "text/csv; charset=utf-8",
      $response->headers->get('Content-Type')
    );

    // Verify CSV content.
    $expected_csv = "Hello,Hallo\nWorld,Welt\n";
    $this->assertEquals($expected_csv, $response->getContent());
  }

  /**
   * Tests the method ::downloadCsv.
   */
  public function testDownloadCsvFailure(): void {
    // Mock the glossary entity with invalid data.
    $glossary = $this->createMock(DeeplGlossary::class);
    $glossary->expects($this->once())
      ->method('getTranslator')
      ->willReturn(NULL);

    // Test the controller.
    $response = $this->controller->downloadCsv($glossary);

    // Verify error response.
    $this->assertEquals(403, $response->getStatusCode());
    $this->assertEquals(
      'Error while downloading the glossary entries.',
      $response->getContent()
    );
  }

}
