<?php

namespace Drupal\Tests\tmgmt_deepl_glossary\Unit;

use DeepL\DeepLException;
use DeepL\GlossaryEntries;
use DeepL\GlossaryInfo;
use DeepL\Translator as DeepLTranslator;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApi;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the DeeplGlossaryApi class.
 */
#[CoversClass(DeeplGlossaryApi::class)]
#[Group('tmgmt_deepl_glossary')]
class DeeplGlossaryApiTest extends UnitTestCase {

  /**
   * Tests the method ::setTranslator.
   */
  public function testSetTranslator(): void {
    $translator = $this->createMock(TranslatorInterface::class);

    $class = $this->createClassMock();
    $class->setTranslator($translator);

    // @phpstan-ignore-next-line.
    $this->assertInstanceOf(TranslatorInterface::class, $class->translator);
  }

  /**
   * Tests the method ::getTranslator.
   */
  public function testGetTranslator(): void {
    $key_id = 'test_deepl_key_entity_id';
    $api_key_value = 'valid-deepl-api-key-from-mock';

    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with('auth_key_entity')
      ->willReturn($key_id);

    // Mock a key entity and related methods.
    $key_entity_mock = $this->createMock(KeyInterface::class);
    $key_entity_mock->expects($this->once())
      ->method('getKeyValue')
      ->willReturn($api_key_value);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock();

    // Mock key repository and related methods.
    $key_repository_mock = $this->createMock(KeyRepositoryInterface::class);
    $key_repository_mock->expects($this->once())
      ->method('getKey')
      ->with($key_id)
      ->willReturn($key_entity_mock);

    // Set key repository for class.
    $class->setKeyRepository($key_repository_mock);

    $class->translator = $translator;
    $actual = $class->getTranslator();

    // @phpstan-ignore-next-line.
    $this->assertInstanceOf(DeepLTranslator::class, $actual);
  }

  /**
   * Tests the method ::getTranslator.
   */
  public function testGetTranslatorException(): void {
    $class = $this->createClassMock();

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Translator not set.');

    $class->getTranslator();
  }

  /**
   * Tests the method ::createGlossary.
   */
  public function testCreateGlossary(): void {
    $name = 'Test Glossary';
    $source_lang = 'en';
    $target_lang = 'de';
    $entries = ['hello' => 'hallo', 'world' => 'welt'];

    // Mock the glossary info object.
    $glossary_info = $this->createMock(GlossaryInfo::class);

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('createGlossary')
      ->with($name, $source_lang, $target_lang, $this->isInstanceOf(GlossaryEntries::class))
      ->willReturn($glossary_info);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->createGlossary($name, $source_lang, $target_lang, $entries);
    // Check for the expected result.
    $this->assertInstanceOf(GlossaryInfo::class, $result);
  }

  /**
   * Tests the method ::createGlossaryFromCsv.
   */
  public function testCreateGlossaryFromCsv(): void {
    $name = 'Test Glossary';
    $source_lang = 'en';
    $target_lang = 'de';
    $csv_data = "artist,Maler,en,de\nprize,Gewinn,en,de";
    $csv_file_path = '/tmp/glossary.csv';
    file_put_contents($csv_file_path, $csv_data);

    // Mock the glossary info object.
    $glossary_info = $this->createMock(GlossaryInfo::class);

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('createGlossaryFromCsv')
      ->with($name, $source_lang, $target_lang, $csv_data)
      ->willReturn($glossary_info);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->createGlossaryFromCsv($name, $source_lang, $target_lang, $csv_file_path);

    // Check for the expected result.
    $this->assertInstanceOf(GlossaryInfo::class, $result);
    // Cleanup test file.
    unlink($csv_file_path);
  }

  /**
   * Tests the method ::createGlossary.
   */
  public function testCreateGlossaryException(): void {
    $name = 'Test Glossary';
    $source_lang = 'en';
    $target_lang = 'de';
    $entries = ['hello' => 'hallo', 'world' => 'welt'];

    // Create an DeepLException.
    $exception = new DeepLException('API Error');

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('createGlossary')
      ->willThrowException($exception);

    // Mock the messenger.
    $messenger = $this->createMock(MessengerInterface::class);

    $messenger->expects($this->once())
      ->method('addMessage')
      ->with($exception->getMessage(), 'error');

    // Mock the logger.
    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('error')
      ->with($exception);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);
    $class->setLogger($logger);
    $class->setMessenger($messenger);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->createGlossary($name, $source_lang, $target_lang, $entries);
    // Check for the expected result.
    $this->assertNull($result);
  }

  /**
   * Tests the method ::getGlossaries.
   */
  public function testGetGlossaries(): void {
    $mock_glossaries = [
      ['id' => '1', 'name' => 'Glossary 1'],
      ['id' => '2', 'name' => 'Glossary 2'],
    ];

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('listGlossaries')
      ->willReturn($mock_glossaries);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getGlossaries();

    $this->assertEquals($mock_glossaries, $result);
  }

  /**
   * Tests the method ::getGlossaries.
   */
  public function testGetGlossariesException(): void {
    // Create an DeepLException.
    $exception = new DeepLException('API Error');

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('listGlossaries')
      ->willThrowException($exception);

    // Mock the messenger.
    $messenger = $this->createMock(MessengerInterface::class);

    $messenger->expects($this->once())
      ->method('addMessage')
      ->with($exception->getMessage(), 'error');

    // Mock the logger.
    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('error')
      ->with($exception);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);
    $class->setLogger($logger);
    $class->setMessenger($messenger);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getGlossaries();

    // Check for the expected result.
    $this->assertEmpty($result);
  }

  /**
   * Tests the method ::deleteGlossary.
   */
  public function testDeleteGlossary(): void {
    $glossary_id = 'test_glossary_id';

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('deleteGlossary')
      ->with($glossary_id);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);
    $result = $class->deleteGlossary($glossary_id);
    $this->assertTrue($result);
  }

  /**
   * Tests the method ::deleteGlossary.
   */
  public function testDeleteGlossaryException(): void {
    $glossary_id = 'test_glossary_id';
    // Create an DeepLException.
    $exception = new DeepLException('API Error');

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('deleteGlossary')
      ->willThrowException($exception);

    // Mock the messenger.
    $messenger = $this->createMock(MessengerInterface::class);

    $messenger->expects($this->once())
      ->method('addMessage')
      ->with($exception->getMessage(), 'error');

    // Mock the logger.
    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('error')
      ->with($exception);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);
    $class->setLogger($logger);
    $class->setMessenger($messenger);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->deleteGlossary($glossary_id);

    // Check for the expected result.
    $this->assertFalse($result);
  }

  /**
   * Tests the method ::getGlossaryEntries.
   */
  public function testGetGlossaryEntries(): void {
    $glossary_id = 'test_glossary_id';
    $mock_entries = ['hello' => 'hallo', 'world' => 'welt'];

    $glossary_entries = $this->createMock(GlossaryEntries::class);
    $glossary_entries->expects($this->once())
      ->method('getEntries')
      ->willReturn($mock_entries);

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('getGlossaryEntries')
      ->with($glossary_id)
      ->willReturn($glossary_entries);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getGlossaryEntries($glossary_id);
    $this->assertEquals($mock_entries, $result);
  }

  /**
   * Tests the method ::getGlossaryEntries.
   */
  public function testGetGlossaryEntriesException(): void {
    $glossary_id = 'test_glossary_id';
    // Create an DeepLException.
    $exception = new DeepLException('API Error');

    // Mock a translator object.
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the DeepL translator library.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('getGlossaryEntries')
      ->willThrowException($exception);

    // Mock the messenger.
    $messenger = $this->createMock(MessengerInterface::class);

    $messenger->expects($this->once())
      ->method('addMessage')
      ->with($exception->getMessage(), 'error');

    // Mock the logger.
    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('error')
      ->with($exception);

    // Mock the DeeplGlossaryApi class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);

    $class->setTranslator($translator);
    $class->setLogger($logger);
    $class->setMessenger($messenger);

    $class->expects($this->once())
      ->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getGlossaryEntries($glossary_id);

    // Check for the expected result.
    $this->assertEmpty($result);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryApi_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplGlossaryApi_Test|MockObject {
    return $this->getMockBuilder(DeeplGlossaryApi_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart
/**
 * Mocked DeeplGlossaryApi class for tests.
 */
class DeeplGlossaryApi_Test extends DeeplGlossaryApi {

  /**
   * {@inheritdoc}
   */
  public LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  public MessengerInterface $messenger;

  /**
   * {@inheritdoc}
   */
  public TranslatorInterface $translator;

  /**
   * Set Messenger for use in tests.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   */
  public function setMessenger(MessengerInterface $messenger): void {
    $this->messenger = $messenger;
  }

  /**
   * Set Logger for use in tests.
   *
   * @param \Psr\Log\LoggerInterface $logger
   */
  public function setLogger(LoggerInterface $logger): void {
    $this->logger = $logger;
  }

  /**
   * Set Key repository for use in tests.
   *
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   */
  public function setKeyRepository(KeyRepositoryInterface $key_repository): void {
    $this->keyRepository = $key_repository;
  }

}
// @codingStandardsIgnoreEnd
