<?php

namespace Drupal\Tests\tmgmt_deepl_glossary\Unit;

use DeepL\GlossaryInfo;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslator;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApi;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryBatch;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryHelperInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests the DeeplGlossaryBatch service.
 */
#[CoversClass(DeeplGlossaryBatch::class)]
#[Group('tmgmt_deepl_glossary')]
class DeeplGlossaryBatchTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create/register string translation mock.
    $container = new ContainerBuilder();
    $container->set('string_translation', $this->getStringTranslationStub());
    \Drupal::setContainer($container);
  }

  /**
   * Tests the method ::buildBatch.
   */
  public function testBuildBatch(): void {
    // Mock the batch builder.
    $batch_builder = $this->createMock(BatchBuilder::class);
    $batch_builder->expects($this->once())
      ->method('toArray')
      ->willReturn([]);

    // Mock translators.
    $translator_01 = $this->createMock(Translator::class);
    $translator_02 = $this->createMock(Translator::class);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test&MockObject $class */
    $class = $this->createClassMock([
      'getDeeplTranslators',
      'createBatchBuilder',
      'addBatchOperations',
      'setBatch',
    ]);

    $class->expects($this->once())
      ->method('getDeeplTranslators')
      ->willReturn([
        $translator_01,
        $translator_02,
      ]);

    // Mocking method calls on the DeeplTranslatorBatch instance.
    $class->expects($this->once())
      ->method('createBatchBuilder')
      ->willReturn($batch_builder);

    $class->expects($this->once())
      ->method('addBatchOperations')
      ->with($batch_builder);

    $class->expects($this->once())
      ->method('setBatch')
      ->with([]);

    $class->buildBatch();
  }

  /**
   * Tests the method ::syncOperation.
   */
  public function testSyncOperation(): void {

    // Mock translator.
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->method('id')->willReturn('test_translator');

    // Mock GlossaryInfo.
    $glossary = new GlossaryInfo(
      'test_glossary',
      'Test Glossary',
      TRUE,
      'en',
      'de',
      new \DateTime(),
      2,
    );

    // Mock entries (in the format they are passed to syncOperation)
    $mock_glossary_entries = [
      'hello' => 'hallo',
      'world' => 'welt',
    ];

    // Mock context.
    $context = [];

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test&MockObject $class */
    $class = $this->createClassMock();

    // Mock the glossary helper.
    $glossary_helper = $this->createMock(DeeplGlossaryHelperInterface::class);
    $class->setDeeplGlossaryHelper($glossary_helper);

    // Run the test.
    $class->syncOperation($translator, $glossary, $mock_glossary_entries, $context);

    // Assert context message and results.
    $this->assertArrayHasKey('message', $context);
    $this->assertArrayHasKey('results', $context);

    assert(is_array($context['results']));
    $this->assertArrayHasKey('glossaries', $context['results']);

    assert(is_array($context['results']['glossaries']));
    $this->assertCount(1, $context['results']['glossaries']);

    assert(is_array($context['results']['glossaries'][0]));
    $this->assertEquals('Test Glossary', $context['results']['glossaries'][0]['name']);
    $this->assertEquals(2, $context['results']['glossaries'][0]['entry_count']);
  }

  /**
   * Tests the method ::cleanUpOperation.
   */
  public function testCleanUpOperation(): void {
    // Define DeepL glossaries from the API.
    $deepl_glossaries = [
      ['glossaryId' => 'glossary1'],
      ['glossaryId' => 'glossary2'],
    ];

    // Machine name of translator for testing.
    $translator = 'translator1';

    // Mock context.
    $context = [];

    // Create mock glossary entities to represent different scenarios.
    // Scenario 1: Glossary exists in both DeepL API and local storage.
    $glossary_01 = $this->createMock(DeeplGlossaryInterface::class);
    $glossary_01->expects($this->once())
      ->method('get')
      ->with('glossary_id')
      ->willReturn((object) ['value' => 'glossary1']);
    $glossary_01->expects($this->never())
      ->method('delete');

    // Scenario 2: Glossary exists in local storage but not in DeepL API.
    $glossary_02 = $this->createMock(DeeplGlossaryInterface::class);
    $glossary_02->expects($this->once())
      ->method('get')
      ->with('glossary_id')
      ->willReturn((object) ['value' => 'glossary3']);
    $glossary_02->expects($this->once())
      ->method('delete');

    // Mock the glossary storage.
    $glossary_storage = $this->createMock(EntityStorageInterface::class);
    $glossary_storage->expects($this->once())
      ->method('loadByProperties')
      ->with(['tmgmt_translator' => $translator])
      ->willReturn([$glossary_01, $glossary_02]);

    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setGlossaryStorage($glossary_storage);

    $class->cleanUpOperation($deepl_glossaries, $translator, $context);
  }

  /**
   * Data provider for testFinishedOperation.
   *
   * @return array
   *   Array of test data.
   */
  public static function dataProviderTestFinishedOperation(): array {
    // @phpcs:disable
    return [
      'Operation with success' => [
        'message_type' => 'addStatus',
        'results' => [
          'glossaries' => [
            ['name' => 'Glossary 1', 'entry_count' => 10],
            ['name' => 'Glossary 2', 'entry_count' => 5],
          ],
        ],
        'expected_message' => 'DeepL glossaries were synced successfully.',
        'success' => TRUE,
      ],
      'Operation with warning' => [
        'message_type' => 'addWarning',
        'results' => [],
        'expected_message' => 'Could not find any glossary for syncing.',
        'success' => TRUE,
      ],
      'Operation with error' => [
        'message_type' => 'addError',
        'results' => [],
        'expected_message' => 'An error occurred while syncing glossaries.',
        'success' => FALSE,
      ],
    ];
    // @phpcs:enable
  }

  /**
   * Tests the method ::finishedOperation.
   *
   * @param string $message_type
   *   The type of message to add to the messenger.
   * @param array $results
   *   The results of the operation.
   * @param string $expected_message
   *   The expected message to add to the messenger.
   * @param bool $success
   *   The success status of the operation.
   */
  #[DataProvider('dataProviderTestFinishedOperation')]
  public function testFinishedOperation(string $message_type, array $results, string $expected_message, bool $success = TRUE): void {
    $operations = [];

    // Mock the translation.
    $translation = $this->createMock(TranslationInterface::class);

    $messenger = $this->createMock(MessengerInterface::class);
    $messenger->expects($this->once())
      ->method($message_type)
      // @codingStandardsIgnoreStart
      ->with($this->equalTo(new TranslatableMarkup($expected_message, [], [], $translation)));
      // @codingStandardsIgnoreEnd

    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setMessenger($messenger);

    $class->finishedOperation($success, $results, $operations);
  }

  /**
   * Tests the method ::addBatchOperations.
   */
  public function testAddBatchOperations(): void {
    // Mock the batch builder.
    $batch_builder = $this->createMock(BatchBuilder::class);
    // Mock translators.
    $deepl_translators = ['translator1' => [], 'translator2' => []];

    // Create glossaries for testing.
    $glossary_01 = new GlossaryInfo(
      'test_glossary_1',
      'Test Glossary 1',
      TRUE,
      'fr',
      'de',
      new \DateTime(),
      5,
    );

    $glossary_02 = new GlossaryInfo(
      'test_glossary_2',
      'Test Glossary 2',
      TRUE,
      'en',
      'de',
      new \DateTime(),
      2,
    );

    // Build array of glossaries.
    $glossaries = [$glossary_01, $glossary_02];

    // Mock the translator.
    $translator = $this->createMock(TranslatorInterface::class);

    // Set expectations for translator.
    $translator->expects($this->atLeast(2))
      ->method('id')
      ->willReturn('Translator');

    // Mock the glossary storage.
    $translator_storage = $this->createMock(EntityStorageInterface::class);

    // Set expectations for translator storage.
    $translator_storage->expects($this->exactly(2))
      ->method('load')
      ->willReturn($translator);

    // Mock the entity type manager.
    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);

    // Set expectations for entity type manager.
    $entity_type_manager->expects($this->exactly(2))
      ->method('getStorage')
      ->with('tmgmt_translator')
      ->willReturn($translator_storage);

    // Mock the glossary api.
    $deepl_glossary_api = $this->createMock(DeeplGlossaryApi::class);

    // Set expectations for glossary api.
    $deepl_glossary_api->expects($this->exactly(2))
      ->method('getGlossaries')
      ->willReturn($glossaries);

    $deepl_glossary_api->expects($this->exactly(2))
      ->method('setTranslator')
      ->with($translator);

    // Build up array of glossary entries.
    $glossary_01_entries = [
      'entry_01' => 'definition_01',
      'entry_02' => 'definition_02',
    ];

    $glossary_02_entries = [
      'entry_03' => 'definition_03',
      'entry_04' => 'definition_04',
    ];

    $deepl_glossary_api->expects($this->any())
      ->method('getGlossaryEntries')
      ->willReturnCallback(function ($glossaryId) use ($glossary_01_entries, $glossary_02_entries) {
        if ($glossaryId === 'test_glossary_1') {
          return $glossary_01_entries;
        }
        elseif ($glossaryId === 'test_glossary_2') {
          return $glossary_02_entries;
        }
        return [];
      });

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setEntityTypeManager($entity_type_manager);
    $class->setDeeplGlossaryApi($deepl_glossary_api);

    // Define expected arguments for each invocation of addOperation.
    $expected_calls = [
      // First call to syncOperation.
      [
        [$class, 'syncOperation'],
        [$translator, $glossary_01, $glossary_01_entries],
      ],
      // Second call to syncOperation.
      [
        [$class, 'syncOperation'],
        [$translator, $glossary_02, $glossary_02_entries],
      ],
      // Call to cleanUpOperation.
      [
        [$class, 'cleanUpOperation'],
        [$glossaries, $translator->id()],
      ],
    ];

    // Initialize a counter to track the calls.
    $callIndex = 0;

    // Set the expectation with a callback to check arguments.
    $batch_builder->expects($this->atLeast(3))
      ->method('addOperation')
      ->willReturnCallback(function ($method, $args) use (&$callIndex, $expected_calls) {
        if ($callIndex < count($expected_calls)) {
          [$expected_method, $expected_args] = $expected_calls[$callIndex];
          $this->assertEquals($expected_method, $method);
          $this->assertEquals($expected_args, $args);
        }
        $callIndex++;
      });

    $class->addBatchOperations($batch_builder, $deepl_translators);
  }

  /**
   * Tests the method ::createBatchBuilder.
   */
  public function testCreateBatchBuilder(): void {
    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $batch_builder = $class->createBatchBuilder();

    // @phpstan-ignore-next-line.
    $this->assertInstanceOf(BatchBuilder::class, $batch_builder);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryBatch_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplGlossaryBatch_Test|MockObject {
    return $this->getMockBuilder(DeeplGlossaryBatch_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart

/**
 * Mocked DeeplGlossaryBatch class for tests.
 */
class DeeplGlossaryBatch_Test extends DeeplGlossaryBatch {

  /**
   * {@inheritDoc}
   */
  public EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritDoc}
   */
  public DeeplGlossaryApi $deeplGlossaryApi;

  /**
   * {@inheritDoc}
   */
  public function createBatchBuilder(): BatchBuilder {
    return parent::createBatchBuilder();
  }

  /**
   * {@inheritDoc}
   */
  public function addBatchOperations(BatchBuilder $batch_builder, array $deepl_translators): void {
    parent::addBatchOperations($batch_builder, $deepl_translators);
  }

  /**
   * {@inheritDoc}
   */
  public function getDeeplTranslators(): array {
    return DeeplTranslator::getTranslators();
  }

  /**
   * Set the glossary storage for tests.
   */
  public function setGlossaryStorage(EntityStorageInterface $glossary_storage): void {
    $this->glossaryStorage = $glossary_storage;
  }

  /**
   * Set the messenger for tests.
   */
  public function setMessenger(MessengerInterface $messenger): void {
    $this->messenger = $messenger;
  }

  /**
   * Set the entity type manager for tests.
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager): void {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Set the deeplGlossaryApi for tests.
   */
  public function setDeeplGlossaryApi(DeeplGlossaryApi $deeplGlossaryApi): void {
    $this->deeplGlossaryApi = $deeplGlossaryApi;
  }

  /**
   * Set the glossary helper for tests.
   */
  public function setDeeplGlossaryHelper(DeeplGlossaryHelperInterface $deeplGlossaryHelper): void {
    $this->glossaryHelper = $deeplGlossaryHelper;
  }

}
// @codingStandardsIgnoreEnd
