<?php

namespace Drupal\tmgmt_deepl;

use DeepL\AppInfo;
use DeepL\DeepLException;
use DeepL\TextResult;
use DeepL\Translator as DeepLTranslator;
use DeepL\Usage;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\TranslatorInterface;

/**
 * The DeeplTranslatorApi service.
 */
class DeeplTranslatorApi implements DeeplTranslatorApiInterface {

  /**
   * The translator.
   *
   * @var \Drupal\tmgmt\TranslatorInterface
   */
  protected TranslatorInterface $translator;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Define source language mappings.
   */
  const array LANGUAGE_MAPPING = [
    'EN-GB' => 'EN',
    'EN-US' => 'EN',
    'PT-BR' => 'PT',
    'PT-PT' => 'PT',
  ];

  /**
   * Constructs a new DeeplTranslatorApi.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   */
  public function __construct(MessengerInterface $messenger, KeyRepositoryInterface $key_repository) {
    // @codeCoverageIgnoreStart
    $this->messenger = $messenger;
    $this->keyRepository = $key_repository;
    // @codeCoverageIgnoreEnd
  }

  /**
   * {@inheritdoc}
   */
  public function getTranslator(): DeepLTranslator {
    if (isset($this->translator)) {
      $key_id = $this->translator->getSetting('auth_key_entity');
      if ($key_id != '') {
        $key_entity = $this->keyRepository->getKey($key_id);
        assert($key_entity instanceof KeyInterface);

        // @codeCoverageIgnoreStart
        $api_key = $key_entity->getKeyValue();
        $app_info = new AppInfo('tmgmt_deepl', '2.3.x');
        return new DeepLTranslator($api_key, ['app_info' => $app_info]);
        // @codeCoverageIgnoreEnd
      }
    }

    throw new \RuntimeException('Translator not set.');
  }

  /**
   * {@inheritdoc}
   */
  public function setTranslator(TranslatorInterface $translator): void {
    $this->translator = $translator;
  }

  /**
   * {@inheritDoc}
   */
  public function translate(Translator $translator, $texts, string $source_language, string $target_language, array $options = [], ?string $context = NULL): array|TextResult|null {
    try {
      $deepl_translator = $this->getTranslator();

      // Add context to the options, if provided.
      $request_options = $options;
      if ($context !== '') {
        $request_options['context'] = $context;
      }

      return $deepl_translator->translateText($texts, $source_language, $target_language, $request_options);
    }
    catch (DeepLException $e) {
      $this->messenger->addMessage($e->getMessage(), 'error');
      return NULL;
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getUsage(): Usage|string {
    try {
      $deepl_translator = $this->getTranslator();
      return $deepl_translator->getUsage();
    }
    catch (\RuntimeException | DeepLException $e) {
      return $e->getMessage();
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getSourceLanguages(): array {
    try {
      $deepl_translator = $this->getTranslator();
      $source_languages = $deepl_translator->getSourceLanguages();
      // Build associative array of source languages.
      $languages = [];
      foreach ($source_languages as $source_language) {
        $languages[$source_language->code] = $source_language->name;
      }
      return $languages;
    }
    catch (\RuntimeException | DeepLException $exception) {
      return [];
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getTargetLanguages(): array {
    try {
      $deepl_translator = $this->getTranslator();
      $target_languages = $deepl_translator->getTargetLanguages();
      // Build associative array of target languages.
      $languages = [];
      foreach ($target_languages as $target_language) {
        $languages[$target_language->code] = $target_language->name;
      }
      return $languages;
    }
    catch (\RuntimeException | DeepLException $exception) {
      return [];
    }
  }

  /**
   * {@inheritDoc}
   */
  public function fixSourceLanguageMappings(string $source_lang): string {
    if (isset(self::LANGUAGE_MAPPING[$source_lang])) {
      return self::LANGUAGE_MAPPING[$source_lang];
    }

    return $source_lang;
  }

}
