<?php

namespace Drupal\tmgmt_deepl\Plugin\QueueWorker;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt_deepl\DeeplTranslatorBatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executes TMGMT DeepL translation queue tasks.
 *
 *  @QueueWorker(
 *    id = "deepl_translate_worker",
 *    title = @Translation("TMGMT DeepL translate queue worker"),
 *    cron = {"time" = 120}
 *  )
 */
class DeeplTranslateWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * The logger channel interface.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The batch service for the DeepL translator.
   *
   * @var \Drupal\tmgmt_deepl\DeeplTranslatorBatchInterface
   */
  protected DeeplTranslatorBatchInterface $batch;

  /**
   * {@inheritDoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, LoggerChannelInterface $logger, DeeplTranslatorBatchInterface $batch) {
    // @codeCoverageIgnoreStart
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->logger = $logger;
    $this->batch = $batch;
    // @codeCoverageIgnoreEnd
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): DeeplTranslateWorker {
    // @phpstan-ignore-next-line
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('tmgmt_deepl'),
      $container->get('tmgmt_deepl.batch'),
    );
  }

  /**
   * Job item translation handler via Cron.
   *
   * @param mixed $data
   *   Associative array containing the following, passed from DeeplTranslator.
   *   [
   *     'job' => $job,
   *     'job_item' => $job,
   *     'q' => $q,
   *     'keys_sequence' => $keys_sequence,
   *   ].
   */
  public function processItem($data): void {
    if (is_array($data) && isset($data['job'], $data['job_item'], $data['q'], $data['keys_sequence'])) {
      // Get actual job.
      $job = $data['job'];
      assert($job instanceof Job);

      // Get the job item.
      $job_item = $data['job_item'];

      // Get array of text to translate.
      /** @var array<string> $q */
      $q = $data['q'];

      // Get the key sequence of translated fields.
      $keys_sequence = $data['keys_sequence'];
      assert(is_array($keys_sequence));

      // Create context array with explicitly defined structure.
      /** @var array{
       *   results: array<string, mixed>} $context
       */
      $context = ['results' => []];

      // Simply run the regular batch operations here.
      $this->batch->translateOperation($job, $q, $keys_sequence, $context);

      // Create a new array for results to be absolutely explicit.
      /** @var array<string, mixed> $results */
      $results = $context['results'] ?? [];

      // Add job item to results.
      $results['job_item'] = $job_item;

      // Finally call finishedOperation with properly typed $results.
      $this->batch->finishedOperation(TRUE, $results, []);
    }
  }

}
