<?php

namespace Drupal\tmgmt_deepl\Plugin\tmgmt\Translator;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueInterface;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt_deepl\DeeplTranslatorApiInterface;
use Drupal\tmgmt_deepl\DeeplTranslatorBatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * DeepL translator base class.
 */
abstract class DeeplTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface, ContinuousTranslatorInterface, DeeplTranslatorInterface {

  /**
   * {@inheritdoc}
   */
  protected $escapeStart = '<deepl translate="no">';

  /**
   * {@inheritdoc}
   */
  protected $escapeEnd = '</deepl>';

  /**
   * The DeepL translator api service.
   *
   * @var \Drupal\tmgmt_deepl\DeeplTranslatorApiInterface
   */
  protected DeeplTranslatorApiInterface $deeplTranslatorApi;

  /**
   * The DeepL translator batch service.
   *
   * @var \Drupal\tmgmt_deepl\DeeplTranslatorBatchInterface
   */
  protected DeeplTranslatorBatchInterface $deeplTranslatorBatch;

  /**
   * The queue object.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected QueueInterface $queue;

  /**
   * TMGMT data service.
   *
   * @var \Drupal\tmgmt\Data
   */
  protected Data $tmgmtData;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * If the process is being run via cron or not.
   *
   * @var bool|null
   */
  protected ?bool $isCron = NULL;

  /**
   * Define deepl translators.
   */
  const DEEPL_TRANSLATORS = [
    'deepl_pro',
    'deepl_free',
    'deepl_api',
  ];

  /**
   * Constructs a DeeplTranslator object.
   *
   * @param \Drupal\tmgmt\Data $tmgmt_data
   *   All data related functionality form tmgmt.
   * @param \Drupal\tmgmt_deepl\DeeplTranslatorApiInterface $deepl_translator_api
   *   The DeeplTranslatorApi service.
   * @param \Drupal\tmgmt_deepl\DeeplTranslatorBatchInterface $deepl_translator_batch
   *   The DeeplTranslatorBatch service.
   * @param \Drupal\Core\Queue\QueueInterface $queue
   *   The queue object.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   */
  public function __construct(Data $tmgmt_data, DeeplTranslatorApiInterface $deepl_translator_api, DeeplTranslatorBatchInterface $deepl_translator_batch, QueueInterface $queue, ModuleHandlerInterface $module_handler, array $configuration, string $plugin_id, array $plugin_definition) {
    // @codeCoverageIgnoreStart
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->tmgmtData = $tmgmt_data;
    $this->deeplTranslatorApi = $deepl_translator_api;
    $this->deeplTranslatorBatch = $deepl_translator_batch;
    $this->queue = $queue;
    $this->moduleHandler = $module_handler;
    $this->isCron = NULL;
    // @codeCoverageIgnoreEnd
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    // @codeCoverageIgnoreStart
    // @phpstan-ignore-next-line
    return new static(
      $container->get('tmgmt.data'),
      $container->get('tmgmt_deepl.api'),
      $container->get('tmgmt_deepl.batch'),
      $container->get('queue')->get('deepl_translate_worker', TRUE),
      $container->get('module_handler'),
      $configuration,
      $plugin_id,
      $plugin_definition
    );
    // @codeCoverageIgnoreEnd
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator): AvailableResult {
    if ($translator->getSetting('auth_key') !== '') {
      return AvailableResult::yes();
    }

    return AvailableResult::no($this->t('@translator is not available. Make sure it is properly <a href=:configured>configured</a>.', [
      '@translator' => $translator->label(),
      ':configured' => $translator->toUrl()->toString(),
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function fixSourceLanguageMappings(string $source_lang): string {
    $language_mapping = [
      'EN-GB' => 'EN',
      'EN-US' => 'EN',
      'PT-BR' => 'PT',
      'PT-PT' => 'PT',
    ];

    if (isset($language_mapping[$source_lang])) {
      return $language_mapping[$source_lang];
    }

    return $source_lang;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultRemoteLanguagesMappings(): array {
    return [
      'ar' => 'AR',
      'bg' => 'BG',
      'cs' => 'CS',
      'da' => 'DA',
      'de' => 'DE',
      'el' => 'EL',
      'en' => 'EN-GB',
      'es' => 'ES',
      'et' => 'ET',
      'fi' => 'FI',
      'fr' => 'FR',
      'hu' => 'HU',
      'id' => 'ID',
      'it' => 'IT',
      'ja' => 'JA',
      'ko' => 'KO',
      'lt' => 'LT',
      'lv' => 'LV',
      'nb' => 'NB',
      'nl' => 'NL',
      'pl' => 'PL',
      'pt-br' => 'PT-BR',
      'pt-pt' => 'PT-PT',
      'ro' => 'RO',
      'ru' => 'RU',
      'sk' => 'SK',
      'sl' => 'SL',
      'sv' => 'SV',
      'tr' => 'TR',
      'uk' => 'UK',
      'zh-hans' => 'ZH',
      'zh-hant' => 'ZH',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultSettings(TranslatorInterface $translator): array {
    // Array of available settings.
    $available_setting_keys = [
      'model_type',
      'split_sentences',
      'preserve_formatting',
      'formality',
      'tag_handling',
      'outline_detection',
      'splitting_tags',
      'non_splitting_tags',
      'ignore_tags',
    ];
    $settings = [];
    foreach ($available_setting_keys as $available_setting_key) {
      $setting = $translator->getSetting($available_setting_key);
      if ($setting !== '') {
        $settings[$available_setting_key] = $setting;
      }
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedRemoteLanguages(TranslatorInterface $translator): array {
    $this->deeplTranslatorApi->setTranslator($translator);
    return $this->deeplTranslatorApi->getTargetLanguages();
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line
   */
  public function getSupportedTargetLanguages(TranslatorInterface $translator, $source_language): array {
    $languages = $this->getSupportedRemoteLanguages($translator);
    // There are no language pairs, any supported language can be translated
    // into the others. If the source language is part of the languages,
    // then return them all, just remove the source language.
    if (array_key_exists($source_language, $languages)) {
      unset($languages[$source_language]);
      return $languages;
    }

    return [];
  }

  /**
   * {@inheritdoc}
   */
  public static function getTranslators(): array {
    $tmgmt_translator_storage = \Drupal::entityTypeManager()->getStorage('tmgmt_translator');
    $deepl_translators = $tmgmt_translator_storage->loadByProperties(['plugin' => self::DEEPL_TRANSLATORS]);
    // Build array of allowed translators.
    $options = [];
    foreach ($deepl_translators as $key => $deepl_translator) {
      $options[$key] = $deepl_translator->label();
    }
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function hasCheckoutSettings(JobInterface $job): bool {
    // Defaults to not having checkout settings.
    $has_checkout_settings = FALSE;
    // Allow alteration of hasCheckoutSettings.
    $this->moduleHandler->alter('tmgmt_deepl_has_checkout_settings', $has_checkout_settings, $job);
    assert(is_bool($has_checkout_settings));

    // Enable checkout settings if context is configurable.
    // @phpstan-ignore-next-line
    $translator = $job->getTranslator();
    assert($translator instanceof TranslatorInterface);

    if ($translator->getSetting('enable_context')) {
      $has_checkout_settings = TRUE;
    }

    return $has_checkout_settings;
  }

  /**
   * {@inheritdoc}
   */
  public function requestJobItemsTranslation(array $job_items): void {
    $job_item = reset($job_items);
    if ($job_item instanceof JobItemInterface) {
      /** @var \Drupal\tmgmt\Entity\Job $job */
      $job = $job_item->getJob();
      foreach ($job_items as $item) {
        if ($job->isContinuous()) {
          $item->active();
        }
        // Pull the source data array through the job and flatten it.
        $data = $this->tmgmtData->filterTranslatable($item->getData());

        $q = [];
        $keys_sequence = [];

        // Build DeepL API q param and preserve initial array keys.
        foreach ($data as $key => $value) {
          assert(is_array($value));
          $q[] = $this->escapeText($value);
          $keys_sequence[] = $key;
        }

        // Use the Queue Worker if running via tmgmt_cron.
        if ($this->isCron()) {
          $this->queue->createItem([
            'job' => $job,
            'job_item' => $item,
            'q' => $q,
            'keys_sequence' => $keys_sequence,
          ]);
        }
        else {
          // Build batch for translations.
          $this->deeplTranslatorBatch->buildBatch($job, $item, $q, $keys_sequence);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job): void {
    $this->requestJobItemsTranslation($job->getItems());
    if (!$job->isRejected()) {
      $job->submitted('The translation job has been submitted.');
    }
  }

  /**
   * Wrapper for debug_backtrace to allow easier testing.
   *
   * @param int $limit
   *   The amount of items to limit in the backtrace.
   *
   * @return array
   *   Array of debug backtrace.
   */
  protected function getDebugBacktrace(int $limit = 5): array {
    // @codeCoverageIgnoreStart
    return debug_backtrace(DEBUG_BACKTRACE_PROVIDE_OBJECT, $limit);
    // @codeCoverageIgnoreEnd
  }

  /**
   * Determine whether the process is being run via TMGMT cron.
   *
   * @param int $backtrace_limit
   *   The amount of items to limit in the backtrace.
   *
   * @return bool
   *   TRUE if processed via cron, FALSE otherwise.
   */
  protected function isCron(int $backtrace_limit = 5): bool {

    if ($this->isCron === NULL) {
      $this->isCron = FALSE;
      foreach ($this->getDebugBacktrace($backtrace_limit) as $item) {
        assert(is_array($item));
        if ($item['function'] === 'tmgmt_cron') {
          $this->isCron = TRUE;
          break;
        }
      }
    }
    return $this->isCron;
  }

}
