<?php

namespace Drupal\Tests\tmgmt_deepl_glossary\Unit\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface;
use Drupal\tmgmt_deepl_glossary\Form\DeeplGlossaryDeleteForm;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests the DeeplGlossaryDeleteForm class.
 */
#[CoversClass(DeeplGlossaryDeleteForm::class)]
#[group('tmgmt_deepl_glossary')]
class DeeplGlossaryDeleteFormTest extends UnitTestCase {

  /**
   * The glossary entity mock.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject|\Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface
   */
  protected MockObject|DeeplGlossaryInterface $glossary;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create/register string translation mock.
    $container = new ContainerBuilder();
    $container->set('string_translation', $this->getStringTranslationStub());
    \Drupal::setContainer($container);

    // Mock a glossary entity.
    $this->glossary = $this->createMock(
      'Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface'
    );
  }

  /**
   * Tests the confirmation question.
   *
   * Tests the method ::getQuestion.
   */
  public function testGetQuestion(): void {
    $this->glossary
      ->expects($this->once())
      ->method('label')
      ->willReturn('Test Glossary');

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\Form\DeeplGlossaryDeleteForm_Test&MockObject $class */
    $class = $this->createClassMock();

    $class->setEntity($this->glossary);

    $question = $class->getQuestion();
    $this->assertEquals(
      'Are you sure you want to delete the DeepL glossary "<em class="placeholder">Test Glossary</em>"?',
      $question->render()
    );
  }

  /**
   * Tests the cancel URL.
   *
   * Tests the method ::getCancelUrl.
   */
  public function testGetCancelUrl(): void {
    // Mock the entity to return a test URL.
    $url = $this->createMock(Url::class);
    $this->glossary->expects($this->once())->method('toUrl')->willReturn($url);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\Form\DeeplGlossaryDeleteForm_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setEntity($this->glossary);

    $cancel_url = $class->getCancelUrl();
    $this->assertSame($url, $cancel_url);
  }

  /**
   * Tests the method ::submitForm.
   */
  public function testSubmitForm(): void {
    // Mock glossary ID and translator.
    $glossary_id = '12345';
    $this->glossary->method('getGlossaryId')->willReturn($glossary_id);
    $translator = $this->createMock(TranslatorInterface::class);
    $this->glossary->method('getTranslator')->willReturn($translator);

    // Ensure the form is submitted as expected.
    $this->glossary->expects($this->exactly(1))->method('delete');

    $glossary_api = $this->createMock(DeeplGlossaryApiInterface::class);
    // Verify if deleteGlossary was called once with the right argument.
    $glossary_api
      ->expects($this->exactly(1))
      ->method('deleteGlossary')
      ->with($this->equalTo($glossary_id));
    // Check for translator.
    $glossary_api
      ->expects($this->once())
      ->method('setTranslator')
      ->with($this->equalTo($translator));

    // Simulate form submission.
    $form_state = $this->createMock(FormStateInterface::class);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\Form\DeeplGlossaryDeleteForm_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setEntity($this->glossary);
    $class->setGlossaryApi($glossary_api);

    $form = [];
    $class->submitForm($form, $form_state);
  }

  /**
   * The data provider for testDeleteDeeplGlossaryNoAction.
   *
   * @return array
   *   Array of test data.
   */
  public static function dataProviderTestDeleteDeeplGlossaryNoAction(): array {
    return [
      'no glossary_id defined' => [
        NULL,
        TranslatorInterface::class,
      ],
      'no translator defined' => ['12345', NULL],
      'no translator and no glossary_id defined' => [NULL, NULL],
    ];
  }

  /**
   * Tests the method ::deleteDeeplGlossary.
   */
  #[DataProvider('dataProviderTestDeleteDeeplGlossaryNoAction')]
  public function testDeleteDeeplGlossaryNoAction(
    ?string $glossary_id,
    ?string $translatorClass,
  ): void {

    // Mock glossary methods.
    $this->glossary->method('getGlossaryId')->willReturn($glossary_id);
    // @phpstan-ignore-next-line.
    $translator = $translatorClass !== NULL ? $this->createMock($translatorClass) : NULL;
    $this->glossary->method('getTranslator')->willReturn($translator);

    // Mock the glossary API.
    $glossary_api = $this->createMock(DeeplGlossaryApiInterface::class);
    $glossary_api->expects($this->never())->method('deleteGlossary');

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\Form\DeeplGlossaryDeleteForm_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setGlossaryApi($glossary_api);
    $class->setEntity($this->glossary);

    // Mock the form state.
    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];
    $class->submitForm($form, $form_state);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl_glossary\Unit\Form\DeeplGlossaryDeleteForm_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(
    array $only_methods = [],
  ): DeeplGlossaryDeleteForm_Test|MockObject {
    return $this->getMockBuilder(DeeplGlossaryDeleteForm_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart

/**
 * Mocked DeeplGlossaryDeleteForm class for tests.
 */
class DeeplGlossaryDeleteForm_Test extends DeeplGlossaryDeleteForm {

  /**
   * Setter for the glossaryApi property to allow manual setting in tests.
   *
   * @param \Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface $glossaryApi
   *  The DeeplGlossaryApiInterface.
   */
  public function setGlossaryApi(DeeplGlossaryApiInterface $glossaryApi): void {
    $this->glossaryApi = $glossaryApi;
  }

}
// @codingStandardsIgnoreEnd
