<?php

namespace Drupal\tmgmt_deepl_glossary\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryHelperInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for deepl_glossary edit forms.
 *
 * @ingroup tmgmt_deepl_glossary
 */
class DeeplGlossaryForm extends ContentEntityForm {

  public function __construct(EntityRepositoryInterface $entity_repository, EntityTypeBundleInfoInterface $entity_type_bundle_info, TimeInterface $time, protected DeeplGlossaryApiInterface $glossaryApi, protected DeeplGlossaryHelperInterface $glossaryHelper, protected AccountInterface $account) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    // @phpstan-ignore-next-line
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('tmgmt_deepl_glossary.api'),
      $container->get('tmgmt_deepl_glossary.helper'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    $deepl_glossary = $form_object->getEntity();
    assert($deepl_glossary instanceof DeeplGlossaryInterface);

    $translator = '';

    // While updating existing entities, we disable field for translator.
    if (!$deepl_glossary->isNew()) {
      assert(is_array($form['tmgmt_translator']));
      assert(is_array($form['tmgmt_translator']['widget']));
      $form['tmgmt_translator']['widget']['#attributes'] = ['disabled' => 'disabled'];
      $form['tmgmt_translator']['widget']['#description'] = $this->t('The translator cannot be changed for existing glossaries.');

      // In case user has permission 'edit deepl_glossary glossary entries' and
      // not 'edit deepl_glossary entities', we disable the following fields:
      // Name, Source language, Target language.
      if ($this->account->hasPermission('edit deepl_glossary glossary entries') && !$this->account->hasPermission('edit deepl_glossary entities')) {
        assert(is_array($form['label']));
        assert(is_array($form['label']['widget']));
        assert(is_array($form['label']['widget'][0]));
        assert(is_array($form['label']['widget'][0]['value']));
        $form['label']['widget'][0]['value']['#attributes'] = ['disabled' => TRUE];

        assert(is_array($form['source_lang']));
        assert(is_array($form['source_lang']['widget']));
        $form['source_lang']['widget']['#attributes'] = ['disabled' => TRUE];

        assert(is_array($form['target_lang']));
        assert(is_array($form['target_lang']['widget']));
        $form['target_lang']['widget']['#attributes'] = ['disabled' => TRUE];
      }

      $deepl_translator = $deepl_glossary->getTranslator();
      assert($deepl_translator instanceof TranslatorInterface);

      $translator = $deepl_translator->id();
    }

    // Add ajax callback for language selection.
    assert(is_array($form['tmgmt_translator']));
    assert(is_array($form['tmgmt_translator']['widget']));
    $form['tmgmt_translator']['#weight'] = 2;
    $form['tmgmt_translator']['widget']['#ajax'] = [
      'callback' => '::ajaxLanguageCallback',
      'wrapper' => 'tmgmt-language-options-wrapper',
      'event' => 'change',
      'progress' => [
        'type' => 'throbber',
        'message' => $this->t('Loading languages...'),
      ],
    ];

    // Add ajax wrapper for language selection.
    $form['language_options_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'tmgmt-language-options-wrapper'],
      '#weight' => 3,
    ];

    // Move the source and target language fields into the container.
    foreach (['source_lang', 'target_lang'] as $field_name) {
      assert(is_array($form[$field_name]));
      $form[$field_name]['#access'] = FALSE;
      $form['language_options_wrapper'][$field_name] = $form[$field_name];
    }

    // Handle the source and target language fields.
    $tmgmt_translator = $form_state->getValue('tmgmt_translator');
    $selected_translator_id = (is_array($tmgmt_translator) && is_array($tmgmt_translator[0]) && isset($tmgmt_translator[0]['value'])) ? $tmgmt_translator[0]['value'] : $translator;
    if ($selected_translator_id != NULL && $selected_translator_id !== '_none') {
      assert(is_string($selected_translator_id));
      $translator = $this->entityTypeManager->getStorage('tmgmt_translator')->load($selected_translator_id);
      assert($translator instanceof TranslatorInterface);

      // Get language mappings.
      $language_mappings = $translator->getRemoteLanguagesMappings();
      $source_languages = $this->glossaryHelper->getAllowedLanguages();
      $available_languages = [];
      foreach ($language_mappings as $language_mapping) {
        assert(is_string($language_mapping));
        $language_mapping = $this->glossaryHelper->fixLanguageMappings($language_mapping);
        if (isset($source_languages[$language_mapping])) {
          $available_languages[$language_mapping] = $source_languages[$language_mapping];

        }
      }
      asort($available_languages);

      // Update the form field options.
      assert(is_array($form['language_options_wrapper']['source_lang']));
      assert(is_array($form['language_options_wrapper']['source_lang']['widget']));
      $form['language_options_wrapper']['source_lang']['widget']['#options'] = $available_languages;
      $form['language_options_wrapper']['source_lang']['widget']['#empty_option'] = $this->t('- Select source language -');
      $form['language_options_wrapper']['source_lang']['#access'] = TRUE;

      assert(is_array($form['language_options_wrapper']['target_lang']));
      assert(is_array($form['language_options_wrapper']['target_lang']['widget']));
      $form['language_options_wrapper']['target_lang']['widget']['#empty_option'] = $this->t('- Select target language -');
      $form['language_options_wrapper']['target_lang']['widget']['#options'] = $available_languages;
      $form['language_options_wrapper']['target_lang']['#access'] = TRUE;
    }

    // Cancel link.
    assert(is_array($form['actions']));
    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#attributes' => ['class' => ['button']],
      '#url' => Url::fromRoute('entity.deepl_glossary.collection'),
      '#weight' => 8,
    ];
    return $form;
  }

  /**
   * Ajax callback to update the language fields.
   */
  public function ajaxLanguageCallback(array &$form, FormStateInterface $form_state): array {
    $form_state->setRebuild();
    return is_array($form['language_options_wrapper']) ? $form['language_options_wrapper'] : [];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): ContentEntityInterface {
    parent::validateForm($form, $form_state);
    /** @var \Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface $entity */
    $entity = $this->buildEntity($form, $form_state);

    // Validate matching source, target language.
    $this->glossaryHelper->validateSourceTargetLanguage($form, $form_state);
    // Validate unique glossary for source/ target language combination.
    $this->glossaryHelper->validateUniqueGlossary($form, $form_state, $entity);

    // Validate unique entries.
    $this->validateUniqueEntries($form, $form_state);

    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $status = $this->entity->save();
    $label = $this->entity->label();
    /** @var \Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface $glossary */
    $glossary = $this->entity;

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()
          ->addMessage($this->t('Created the %label DeepL glossary.', ['%label' => $label]));
        break;

      default:
        $this->messenger()
          ->addMessage($this->t('Saved the %label DeepL glossary.', ['%label' => $label]));
    }
    $this->saveDeeplGlossary($glossary, $status);

    $form_state->setRedirect('entity.deepl_glossary.collection');
    return $status;
  }

  /**
   * Save DeepL glossary to DeepL API.
   *
   * @param \Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface $glossary
   *   The DeepL glossary entity object.
   * @param int $status
   *   The save status (indicator for new or existing entities)
   */
  protected function saveDeeplGlossary(DeeplGlossaryInterface $glossary, int $status): void {
    $translator = $glossary->getTranslator();
    $glossary_api = $this->glossaryApi;

    if ($translator instanceof TranslatorInterface) {
      $glossary_api->setTranslator($translator);
    }

    // By updating and existing entry we need to delete DeepL glossary first.
    if ($status === SAVED_UPDATED && strval($glossary->getGlossaryId()) !== '') {
      $glossary_api->deleteGlossary(strval($glossary->getGlossaryId()));
    }

    // Create glossary with DeepL API.
    $result = $glossary_api->createGlossary(strval($glossary->label()), $glossary->getSourceLanguage(), $glossary->getTargetLanguage(), $glossary->getEntries());

    // Save DeepL internal glossary_id to entity.
    if (isset($result->glossaryId) && $result->ready) {
      $glossary->set('glossary_id', $result->glossaryId);
      $glossary->set('ready', TRUE);
      $glossary->set('entry_count', $result->entryCount);
      $glossary->save();
    }
  }

  /**
   * Validate unique entries.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  protected function validateUniqueEntries(array &$form, FormStateInterface $form_state): void {
    $user_input = $form_state->getUserInput();
    $entries = $user_input['entries'] ?? [];
    assert(is_array($entries));

    $subjects = [];
    foreach ($entries as $entry) {
      assert(is_array($entry));
      if (isset($entry['subject']) && $entry['subject'] !== '') {
        $subjects[] = $entry['subject'];
      }
    }

    // Duplicate check.
    $unique_subjects = array_unique($subjects);
    $duplicates = array_diff_assoc($subjects, $unique_subjects);
    if (count($duplicates) > 0) {
      foreach (array_keys($duplicates) as $key) {
        $form_state->setErrorByName('entries][' . $key . '][subject', $this->t('Please check your glossary entries, the subjects should be unique.'));
      }
    }
  }

}
