<?php

namespace Drupal\Tests\tmgmt_deepl_glossary\Kernel;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt_deepl_glossary\Controller\DeeplGlossaryDownloadController;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface;
use Drupal\tmgmt_deepl_glossary\Entity\DeeplGlossary;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the DeepL glossary entity operations.
 */
#[Group('tmgmt_deepl_glossary')]
class DeeplGlossaryOperationsTest extends DeeplGlossaryKernelTestBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Install required schema and config.
    $this->installConfig(['tmgmt_deepl']);
    $this->installEntitySchema('tmgmt_translator');
  }

  /**
   * Tests the download CSV operation link.
   */
  public function testDownloadCsvOperation(): void {
    // Create a mock glossary API service.
    $mock_api = $this->createMock(DeeplGlossaryApiInterface::class);
    $mock_api->method('getGlossaryEntries')
      ->willReturn([
        'hello' => 'hallo',
        'world' => 'welt',
        'computer' => 'rechner',
      ]);
    $this->container->set('tmgmt_deepl_glossary.api', $mock_api);

    // Create a translator.
    $translator = Translator::create([
      'name' => 'deepl_test',
      'label' => 'DeepL Test',
      'plugin' => 'deepl_api',
      'settings' => [
        'auth_key_entity' => 'test_api_key_entity',
      ],
    ]);
    $translator->save();

    // Create a test glossary entity with entries.
    $glossary = DeeplGlossary::create([
      'label' => 'Test Glossary',
      'source_lang' => 'en',
      'target_lang' => 'de',
      'tmgmt_translator' => 'deepl_test',
      'glossary_id' => 'test-glossary-id',
      'entries' => [
        [
          'subject' => 'hello',
          'definition' => 'hallo',
        ],
        [
          'subject' => 'world',
          'definition' => 'welt',
        ],
        [
          'subject' => 'computer',
          'definition' => 'rechner',
        ],
      ],
    ]);
    $glossary->save();

    // Mock the getTranslator method to return our translator.
    $mock_glossary = $this->getMockBuilder(DeeplGlossary::class)
      ->onlyMethods(['getTranslator'])
      ->setConstructorArgs([[], 'deepl_glossary'])
      ->getMock();
    $mock_glossary->method('getTranslator')
      ->willReturn($translator);

    // Copy over the values from our real glossary.
    foreach (['label', 'source_lang', 'target_lang', 'tmgmt_translator', 'glossary_id', 'entries'] as $field) {
      $mock_glossary->set($field, $glossary->get($field)->getValue());
    }

    // Get the operations for the entity.
    $operations = [];
    \Drupal::moduleHandler()->alter('entity_operation', $operations, $glossary);
    assert(is_array($operations));

    // Assert that the download_csv operation exists and correctly configured.
    $this->assertArrayHasKey('download_csv', $operations);

    assert(is_array($operations['download_csv']));
    assert($operations['download_csv']['title'] instanceof TranslatableMarkup);
    $this->assertEquals('Download CSV', $operations['download_csv']['title']->render());
    $this->assertInstanceOf(DeeplGlossaryInterface::class, $glossary);

    assert($operations['download_csv']['url'] instanceof Url);
    $this->assertEquals('/admin/tmgmt/deepl_glossaries/' . $glossary->id() . '/download-csv', $operations['download_csv']['url']->toString());
    $this->assertEquals(10, $operations['download_csv']['weight']);

    // Get the download controller and request the CSV.
    $controller = new DeeplGlossaryDownloadController($mock_api);
    $response = $controller->downloadCsv($mock_glossary);

    // Check response headers.
    $this->assertEquals('text/csv; charset=utf-8', $response->headers->get('Content-Type'));
    $this->assertStringContainsString('Test_Glossary_en_to_de.csv', strval($response->headers->get('Content-Disposition')));

    // Parse the CSV content and verify entries.
    $csv_content = $response->getContent();
    $this->assertNotEmpty($csv_content);

    // Parse CSV content into array.
    $lines = explode("\n", trim($csv_content));
    // Provide the default separator, enclosure, and escape characters.
    $entries = array_map(function ($line) {
      // Use the standard defaults.
      return str_getcsv($line, ',', '"', '\\');
    }, $lines);

    $this->assertCount(3, $entries);

    // Create associative array for easier testing.
    $parsed_entries = [];
    foreach ($entries as $entry) {
      if (count($entry) === 2) {
        $parsed_entries[$entry[0]] = $entry[1];
      }
    }

    // Verify the CSV content matches our test data.
    $this->assertEquals('hallo', $parsed_entries['hello']);
    $this->assertEquals('welt', $parsed_entries['world']);
    $this->assertEquals('rechner', $parsed_entries['computer']);
  }

  /**
   * Tests the download CSV route access.
   */
  public function testDownloadCsvAccess(): void {
    // Create a test glossary entity with entries.
    $glossary = DeeplGlossary::create([
      'label' => 'Test Glossary',
      'source_lang' => 'en',
      'target_lang' => 'de',
      'entries' => [
        [
          'subject' => 'test',
          'definition' => 'test',
        ],
      ],
    ]);
    $glossary->save();

    // Get the URL object for the download CSV route.
    $url = $glossary->toUrl('download_csv');

    // Assert that the route exists and has the correct path.
    $this->assertEquals('/admin/tmgmt/deepl_glossaries/' . $glossary->id() . '/download-csv', $url->toString());
  }

}
