<?php

namespace Drupal\Tests\tmgmt_deepl_glossary\Unit;

use DeepL\GlossaryInfo;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApi;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryHelper;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests the DeeplGlossaryBatch service.
 */
#[CoversClass(DeeplGlossaryHelper::class)]
#[group('tmgmt_deepl_glossary')]
class DeeplGlossaryHelperTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create/register string translation mock.
    $container = new ContainerBuilder();
    $container->set('string_translation', $this->getStringTranslationStub());
    \Drupal::setContainer($container);
  }

  /**
   * Define allowed languages for use in tests.
   *
   * @var array
   *  The array of allowed languages.
   */
  public static array $allowedLanguages = [
    'AR' => 'Arabic',
    'BG' => 'Bulgarian',
    'CS' => 'Czech',
    'DA' => 'Danish',
    'DE' => 'German',
    'EL' => 'Greek',
    'EN' => 'English (all variants)',
    'ES' => 'Spanish',
    'ET' => 'Estonian',
    'FI' => 'Finnish',
    'FR' => 'French',
    'HU' => 'Hungarian',
    'ID' => 'Indonesian',
    'IT' => 'Italian',
    'JA' => 'Japanese',
    'KO' => 'Korean',
    'LT' => 'Lithuanian',
    'LV' => 'Latvian',
    'NB' => 'Norwegian (Bokmål)',
    'NL' => 'Dutch',
    'PL' => 'Polish',
    'PT' => 'Portuguese (all variants)',
    'RO' => 'Romanian',
    'RU' => 'Russian',
    'SK' => 'Slovak',
    'SL' => 'Slovenian',
    'SV' => 'Swedish',
    'TR' => 'Turkish',
    'UK' => 'Ukrainian',
    'ZH' => 'Chinese (all variants)',
  ];

  /**
   * Data provider for testGetAllowedLanguages().
   *
   * @return array
   *   Array of test data.
   */
  public static function dataProviderTestGetAllowedLanguages(): array {
    return [
      'default' => [
        [],
        self::$allowedLanguages,
      ],
      'altered allowed languages' => [
        [
          'SM-IT' => 'San Marino - Italian',
        ],
        self::$allowedLanguages + ['SM-IT' => 'San Marino - Italian'],
      ],
    ];
  }

  /**
   * Tests the method ::getAllowedLanguages.
   */
  #[DataProvider('dataProviderTestGetAllowedLanguages')]
  public function testGetAllowedLanguages(array $altered_languages, array $expected_languages): void {
    // Mock the entity type manager.
    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);

    // Mock the module handler.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);

    // Set expectations for module handler.
    $module_handler->expects($this->once())
      ->method('alter')
      ->with('tmgmt_deepl_glossary_allowed_languages', self::$allowedLanguages)
      ->willReturnCallback(function ($hook, array &$languages) use ($altered_languages) {
        // Apply the alterations.
        $languages = array_merge($languages, $altered_languages);
      });

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setModuleHandler($module_handler);
    $class->setEntityTypeManager($entity_type_manager);

    $languages = $class->getAllowedLanguages();

    // @phpstan-ignore-next-line.
    $this->assertEquals($expected_languages, array_map('strval', $languages));
  }

  /**
   * Tests the method ::getValidSourceTargetLanguageCombinations.
   */
  public function testGetValidSourceTargetLanguageCombinations(): void {
    // Mock the getAllowedLanguages method to return predefined languages.
    $allowed_languages = [
      'EN' => 'English',
      'FR' => 'French',
      'DE' => 'German',
    ];

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test&MockObject $class */
    $class = $this->createClassMock(['getAllowedLanguages']);
    $class->setModuleHandler($this->createMock(ModuleHandlerInterface::class));
    $class->setEntityTypeManager($this->createMock(EntityTypeManagerInterface::class));

    $class->expects($this->once())
      ->method('getAllowedLanguages')
      ->willReturn($allowed_languages);

    // Expected combinations.
    $expected_combinations = [
      ['EN' => 'FR'],
      ['EN' => 'DE'],
      ['FR' => 'EN'],
      ['FR' => 'DE'],
      ['DE' => 'EN'],
      ['DE' => 'FR'],
    ];

    $combinations = $class->getValidSourceTargetLanguageCombinations();
    $this->assertEquals($expected_combinations, $combinations);
  }

  /**
   * Tests the method ::getMatchingGlossaries.
   */
  public function testGetMatchingGlossaries(): void {
    // Set up mock data.
    $translator = 'test_translator';
    $source_lang = 'EN-US';
    $target_lang = 'DE-DE';

    // Expected result from loadByProperties.
    $expected_glossaries = ['glossary1', 'glossary2'];

    // Mock glossary storage.
    $glossary_storage = $this->createMock(EntityStorageInterface::class);

    // Set expectations for glossary storage.
    $glossary_storage->expects($this->once())
      ->method('loadByProperties')
      ->with([
        'tmgmt_translator' => $translator,
        'source_lang' => 'EN',
        'target_lang' => 'DE',
      ])
      ->willReturn($expected_glossaries);

    // Mock the entity type manager.
    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);

    // Set expectations for entity type manager.
    $entity_type_manager->expects($this->once())
      ->method('getStorage')
      ->with('deepl_glossary')
      ->willReturn($glossary_storage);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test&MockObject $class */
    $class = $this->createClassMock(['fixLanguageMappings']);
    $class->expects($this->exactly(2))
      ->method('fixLanguageMappings')
      ->willReturnCallback(function ($langcode) {
        return match ($langcode) {
          'EN-US' => 'EN',
          'DE-DE' => 'DE',
          default => $langcode,
        };
      });

    $class->setModuleHandler($this->createMock(ModuleHandlerInterface::class));
    $class->setEntityTypeManager($entity_type_manager);

    // Call the method.
    $glossaries = $class->getMatchingGlossaries($translator, $source_lang, $target_lang);

    // Assert the result is what we expect.
    $this->assertEquals($expected_glossaries, $glossaries);
  }

  /**
   * Tests the method ::fixLanguageMappings.
   */
  public function testFixLanguageMappings(): void {
    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test&MockObject $class */
    $class = $this->createClassMock();

    // Define test cases.
    $test_cases = [
      'EN' => 'EN',
      'en' => 'EN',
      'EN-US' => 'EN',
      'EN-GB' => 'EN',
      'ZH-Hans' => 'ZH',
      'A' => 'A',
    ];

    // Iterate over the test cases and assert the results.
    foreach ($test_cases as $input => $expected_output) {
      $this->assertEquals($expected_output, $class->fixLanguageMappings($input), "Failed asserting that '$input' maps to '$expected_output'");
    }
  }

  /**
   * Tests the method ::saveGlossary.
   */
  public function testSaveGlossary(): void {
    // Create mock objects for dependencies.
    $deepl_glossary_api = $this->createMock(DeeplGlossaryApi::class);
    $glossary_storage = $this->createMock(EntityStorageInterface::class);
    $translator = $this->createMock(TranslatorInterface::class);

    // Add a real glossary for testing.
    $glossary_info = new GlossaryInfo(
      'test_glossary_id',
      'Test Glossary',
      TRUE,
      'en',
      'de',
      new \DateTime(),
      10,
    );

    // Define glossary entries.
    $glossary_entries = [
      ['subject' => 'test', 'definition' => 'test_definition'],
    ];

    // Set up mock expectations for Translator.
    $translator->expects($this->once())
      ->method('id')
      ->willReturn('test_translator');

    // Ensure we check for the glossary_id in the loadByProperties method.
    $glossary_storage->expects($this->once())
      ->method('loadByProperties')
      ->with(['glossary_id' => 'test_glossary_id'])
      ->willReturn([]);

    $expected_create_args = [
      'label' => 'Test Glossary',
      'glossary_id' => 'test_glossary_id',
      'source_lang' => 'EN',
      'target_lang' => 'DE',
      'tmgmt_translator' => 'test_translator',
      'ready' => TRUE,
      'entries' => $glossary_entries,
      'entries_format' => 'tsv',
      'entry_count' => 10,
    ];

    $new_glossary_entity = $this->createMock(DeeplGlossaryInterface::class);
    // Ensure that create method is run once for saving.
    $glossary_storage->expects($this->once())
      ->method('create')
      ->with($this->equalTo($expected_create_args))
      ->willReturn($new_glossary_entity);

    $new_glossary_entity
      ->expects($this->once())
      ->method('save');

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setDeeplGlossaryApi($deepl_glossary_api);
    $class->setGlossaryStorage($glossary_storage);

    // Save the glossary.
    $class->saveGlossary($glossary_info, $glossary_entries, $translator);
  }

  /**
   * Tests the method ::saveGlossary for existing glossaries.
   */
  public function testSaveGlossaryExisting(): void {
    // Create mock objects for dependencies.
    $deepl_glossary_api = $this->createMock(DeeplGlossaryApi::class);
    $glossary_storage = $this->createMock(EntityStorageInterface::class);
    $translator = $this->createMock(TranslatorInterface::class);

    // Set up mock expectations for GlossaryInfo
    // Add a real glossary for testing.
    $glossary_info = new GlossaryInfo(
      'existing_glossary_id',
      'Test Glossary 1',
      TRUE,
      'fr',
      'de',
      new \DateTime(),
      5,
    );

    // Define glossary entries.
    $glossary_entries = [
      ['subject' => 'test', 'definition' => 'test_definition'],
    ];

    // Create mock for existing glossary.
    $existing_glossary = $this->createMock(DeeplGlossaryInterface::class);

    // Set up mock expectations for existing glossary.
    $existing_glossary->expects($this->once())
      ->method('set')
      ->with('entries', $glossary_entries);
    $existing_glossary->expects($this->once())
      ->method('save');

    // Set mock expectations for GlossaryStorage.
    $glossary_storage->expects($this->once())
      ->method('loadByProperties')
      ->with(['glossary_id' => 'existing_glossary_id'])
      ->willReturn([$existing_glossary]);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setDeeplGlossaryApi($deepl_glossary_api);
    $class->setGlossaryStorage($glossary_storage);

    // Call the save method.
    $class->saveGlossary($glossary_info, $glossary_entries, $translator);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl_glossary\Unit\DeeplGlossaryHelper_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplGlossaryHelper_Test|MockObject {
    return $this->getMockBuilder(DeeplGlossaryHelper_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart

/**
 * Mocked DeeplGlossaryApi class for tests.
 */
class DeeplGlossaryHelper_Test extends DeeplGlossaryHelper {

  /**
   * {@inheritdoc }
   */
  public EntityTypeManagerInterface $entityTypeManager;

  /**
   * @var \Drupal\tmgmt_deepl_glossary\DeeplGlossaryApi
   */
  public DeeplGlossaryApi $deeplGlossaryApi;

  /**
   * Set the entity type manager for tests.
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager): void {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Set ModuleHandler for use in tests.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   */
  public function setModuleHandler(ModuleHandlerInterface $module_handler): void {
    $this->moduleHandler = $module_handler;
  }

  /**
   * Set the deeplGlossaryApi for tests.
   */
  public function setDeeplGlossaryApi(DeeplGlossaryApi $deeplGlossaryApi): void {
    $this->deeplGlossaryApi = $deeplGlossaryApi;
  }

  /**
   * Set the glossary storage for tests.
   */
  public function setGlossaryStorage(EntityStorageInterface $glossary_storage): void {
    $this->glossaryStorage = $glossary_storage;
  }

}
// @codingStandardsIgnoreEnd
