<?php

/**
 * @file
 * Post update functions for tmgmt_deepl module.
 */

use Drupal\tmgmt\Entity\Translator;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Migrate DeepL translator API keys from direct config to Key module reference.
 *
 * @param array $sandbox
 *   Stores information for batch processing (not used in this simple case).
 *
 * @return \Drupal\Core\StringTranslation\TranslatableMarkup|string
 *   Status message for Drush console output.
 */
function tmgmt_deepl_post_update_migrate_api_key_to_key_module(array &$sandbox = []): TranslatableMarkup|string {
  $logger = \Drupal::logger('tmgmt_deepl_update');

  // Array to collect names of translators needing manual update.
  $translators_requiring_action = [];
  $translator_ids = \Drupal::entityQuery('tmgmt_translator')
    ->condition('plugin', 'deepl_api')
    ->accessCheck(FALSE)
    ->execute();

  if (empty($translator_ids)) {
    return t('No DeepL translators (plugin: deepl_api) found to check for API key migration.');
  }

  assert(is_array($translator_ids));

  $checked_count = 0;
  foreach ($translator_ids as $translator_id) {
    $translator = Translator::load($translator_id);
    if (!$translator instanceof Translator) {
      $logger->warning('Could not load translator with ID @id during API key migration check.', ['@id' => $translator_id]);
      continue;
    }
    $checked_count++;

    // Get translator settings.
    $settings = $translator->getSettings();
    $translator_label = $translator->label();
    $translator_machine_name = $translator->id();

    // Check if the old setting exists and the new one doesn't.
    if (!empty($settings['auth_key']) && empty($settings['auth_key_entity'])) {

      // Log the detailed warning to Drupal logs.
      $logger->warning('DeepL translator "%label" (%id) still uses the deprecated direct API key configuration. Please create a key entity (type: DeepL API Key or Authentication) with the API key and re-save the translator settings, selecting the new Key entity. The old key setting has been removed for security.', [
        '%label' => $translator_label,
        '%id' => $translator_machine_name,
      ]);

      // Add this translator to the list requiring action for the Drush message.
      $translators_requiring_action[] = sprintf('%s (%s)', $translator_label, $translator_machine_name);

      // Remove the old setting for security.
      $translator->setSetting('auth_key', NULL);
      $translator->save();
    }
    elseif (isset($settings['auth_key']) && empty($settings['auth_key']) && $translator->getSetting('auth_key') != NULL) {
      $translator->setSetting('auth_key', NULL);
      $translator->save();
      $logger->info('Removed empty legacy auth_key setting from DeepL translator "%label" (%id).', [
        '%label' => $translator_label,
        '%id' => $translator_machine_name,
      ]);
    }
    elseif (isset($settings['auth_key']) && !empty($settings['auth_key_entity'])) {
      // New key exists, ensure old one is gone if it somehow remained.
      if ($translator->getSetting('auth_key') != NULL) {
        $translator->setSetting('auth_key', NULL);
        $translator->save();
        $logger->info('Removed legacy auth_key setting from DeepL translator "%label" (%id) as new auth_key_entity is set.', [
          '%label' => $translator_label,
          '%id' => $translator_machine_name,
        ]);
      }
    }
  }

  // Construct the final message for Drush based on findings.
  if (!empty($translators_requiring_action)) {
    $count = count($translators_requiring_action);
    $drush_message = t('%count DeepL translator(s) require manual API key migration to the Key module:', ['%count' => $count]);
    $drush_message .= "\n - " . implode("\n - ", $translators_requiring_action);
    $drush_message .= "\n" . t('ACTION REQUIRED: For each translator listed above, please create a Key entity (type: DeepL API Key) containing its API key. Then, edit the translator configuration in the TMGMT settings UI and select the corresponding Key entity.');
    $drush_message .= "\n" . t('The old, insecurely stored keys have been removed from the configuration. See Drupal logs (channel: tmgmt_deepl_update) for details.');
    return $drush_message;
  }
  else {
    // If the loop ran but no action was needed.
    return t('Checked @count DeepL translator(s). No configurations required manual API key migration to the Key module.', ['@count' => $checked_count]);
  }
}
