<?php

namespace Drupal\tmgmt_deepl_glossary;

use DeepL\GlossaryInfo;
use DeepL\Translator;
use Drupal\tmgmt\TranslatorInterface;

/**
 * Provides an interface defining DeepL glossary API service.
 */
interface DeeplGlossaryApiInterface {

  /**
   * Get translator.
   *
   * @throws \DeepL\DeepLException
   */
  public function getTranslator(): Translator;

  /**
   * Set translator for all glossary API calls.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator entity.
   */
  public function setTranslator(TranslatorInterface $translator): void;

  /**
   * Get all glossaries for active translator.
   *
   * @return array
   *   Array of available glossaries.
   */
  public function getGlossaries(): array;

  /**
   * Get entries for a given glossary id.
   *
   * @param string $glossary_id
   *   The unique ID assigned to the glossary.
   *
   * @return array
   *   Array of glossary entries.
   */
  public function getGlossaryEntries(string $glossary_id): array;

  /**
   * Delete glossary for a given glossary id.
   *
   * @param string $glossary_id
   *   The unique ID assigned to the glossary.
   *
   * @return bool
   *   Whether the glossary was deleted successfully.
   */
  public function deleteGlossary(string $glossary_id): bool;

  /**
   * Create new glossary.
   *
   * @param string $name
   *   Name to be associated with the glossary.
   * @param string $source_lang
   *   The language in which the source texts in the glossary are specified.
   * @param string $target_lang
   *   The language in which the target texts in the glossary are specified.
   * @param array $entries
   *   The entries of the glossary.
   *
   * @return \DeepL\GlossaryInfo|null
   *   Array with results after creating a glossary or NULL in case of error.
   */
  public function createGlossary(string $name, string $source_lang, string $target_lang, array $entries): ?GlossaryInfo;

  /**
   * Creates a glossary from a CSV file.
   *
   * @param string $name
   *   The name of the glossary.
   * @param string $source_lang
   *   The source language code.
   * @param string $target_lang
   *   The target language code.
   * @param string $csv_file_path
   *   The file path to the CSV glossary file.
   *
   * @return \DeepL\GlossaryInfo|null
   *   The created glossary object or NULL on failure.
   */
  public function createGlossaryFromCsv(string $name, string $source_lang, string $target_lang, string $csv_file_path): ?GlossaryInfo;

}
