<?php

namespace Drupal\tmgmt_deepl_glossary\Form;

use DeepL\GlossaryInfo;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslator;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryApiInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryHelperInterface;
use Drupal\tmgmt_deepl_glossary\Entity\DeeplGlossary;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Form for uploading a CSV file to create a DeepL glossary.
 */
class DeeplGlossaryUploadCsvForm extends FormBase {

  public function __construct(
    protected DeeplGlossaryApiInterface $glossaryApi,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DeeplGlossaryHelperInterface $glossaryHelper,
    protected RendererInterface $renderer,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    // @phpstan-ignore-next-line
    return new static(
      $container->get('tmgmt_deepl_glossary.api'),
      $container->get('entity_type.manager'),
      $container->get('tmgmt_deepl_glossary.helper'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'deepl_glossary_upload_csv_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // The name.
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Glossary Name'),
      '#description' => $this->t('Enter the name of the glossary to be created.'),
      '#required' => TRUE,
    ];

    // Add the translator selection field.
    $form['tmgmt_translator'] = [
      '#type' => 'select',
      '#title' => $this->t('Translator'),
      '#description' => $this->t('Select the TMGMT translator.'),
      '#options' => DeeplTranslator::getTranslators(),
      '#required' => TRUE,
    ];

    // Add the source language field.
    $form['source_lang'] = [
      '#type' => 'select',
      '#title' => $this->t('Source Language'),
      '#description' => $this->t('The language in which the source texts in the glossary are specified.'),
      '#options' => DeeplGlossary::getAllowedLanguages(),
      '#required' => TRUE,
    ];

    // Add the target language field.
    $form['target_lang'] = [
      '#type' => 'select',
      '#title' => $this->t('Target Language'),
      '#description' => $this->t('The language in which the target texts in the glossary are specified.'),
      '#options' => DeeplGlossary::getAllowedLanguages(),
      '#required' => TRUE,
    ];

    // Add the CSV file field.
    $file_size_limit = 10 * 1024 * 1024;
    $upload_validators = [
      'FileExtension' => ['extensions' => 'csv'],
      'FileSizeLimit' => ['fileLimit' => $file_size_limit],
    ];
    $file_upload_help = [
      '#theme' => 'file_upload_help',
      '#description' => $this->t("Upload the CSV file containing comma separated list of glossary terms. (Source, Target)"),
      '#upload_validators' => $upload_validators,
      '#cardinality' => 1,
    ];
    $form['csv_file'] = [
      '#type' => 'file',
      '#title' => $this->t('CSV File'),
      '#description' => $this->renderer->renderInIsolation($file_upload_help),
      '#required' => TRUE,
      '#upload_validators' => $upload_validators,
    ];

    // Add form actions.
    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Upload'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate matching source, target language.
    $this->glossaryHelper->validateSourceTargetLanguage($form, $form_state);
    // Validate unique glossary for source/ target language combination.
    $this->glossaryHelper->validateUniqueGlossary($form, $form_state, NULL);

    // Validate the csv file.
    $this->validateCsvFile($form, $form_state);
  }

  /**
   * Validate unique entries.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  protected function validateCsvFile(array &$form, FormStateInterface $form_state): void {
    // Retrieve all uploaded files.
    $all_files = $this->getRequest()->files->get('files', []);

    // Check if the 'csv_file' input has a valid uploaded file.
    if (is_array($all_files) && isset($all_files['csv_file'])) {
      $file_upload = $all_files['csv_file'];

      // Check for valid file upload and run further validation.
      if ($file_upload instanceof UploadedFile && $file_upload->isValid()) {
        // Validate the file extension.
        if (!in_array($file_upload->getClientOriginalExtension(), ['csv', 'CSV'], TRUE)) {
          $form_state->setErrorByName('csv_file', $this->t('The file must be a CSV file.'));
          return;
        }

        // Store the real file path in the form state for further processing.
        $form_state->setValue('csv_file', $file_upload->getRealPath());
        return;
      }
    }

    // Set an error if the file could not be uploaded or is invalid.
    $form_state->setErrorByName('csv_file', $this->t('The file could not be uploaded.'));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var string $name */
    $name = $form_state->getValue('name');
    /** @var string $source_lang */
    $source_lang = $form_state->getValue('source_lang');
    /** @var string $target_lang */
    $target_lang = $form_state->getValue('target_lang');
    /** @var string $translator_id */
    $translator_id = $form_state->getValue('tmgmt_translator');
    /** @var string $csv_path */
    $csv_path = $form_state->getValue('csv_file');

    if ($csv_path !== '' && $translator_id !== '') {
      // Load the translator using the entity type manager.
      $translator = $this->entityTypeManager->getStorage('tmgmt_translator')->load($translator_id);
      if ($translator instanceof TranslatorInterface) {
        // Pass the translator to the glossary API.
        $this->glossaryApi->setTranslator($translator);
        // Use the DeeplGlossaryApi to create the glossary.
        $glossary = $this->glossaryApi->createGlossaryFromCsv($name, $source_lang, $target_lang, $csv_path);
        if ($glossary instanceof GlossaryInfo) {
          // Save glossary entity.
          $this->saveGlossaryEntity($glossary, $translator);

          // Show success message.
          $this->messenger()->addMessage($this->t('Glossary %name was created successfully.', ['%name' => $name]));

          // Redirect to the glossary collection page.
          $form_state->setRedirect('entity.deepl_glossary.collection');
        }
        else {
          $this->messenger()->addError($this->t('Failed to create the glossary. Please check the CSV file and try again.'));
        }
      }
    }
    else {
      $this->messenger()->addError($this->t('The uploaded file could not be processed.'));
    }
  }

  /**
   * Save glossary to drupal entity.
   *
   * @param \DeepL\GlossaryInfo $glossary
   *   The glossary object.
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator object.
   */
  protected function saveGlossaryEntity(GlossaryInfo $glossary, TranslatorInterface $translator): void {
    // Save glossary entity.
    $entries = $this->glossaryApi->getGlossaryEntries($glossary->glossaryId);
    // Prepare glossary entries.
    $glossary_entries = [];
    foreach ($entries as $subject => $definition) {
      $glossary_entries[] = [
        'subject' => $subject,
        'definition' => $definition,
      ];
    }
    $this->glossaryHelper->saveGlossary($glossary, $glossary_entries, $translator);
  }

}
