<?php

namespace Drupal\Tests\tmgmt_deepl\Unit;

use DeepL\TextResult;
use Drupal\Component\Utility\Html;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt\TranslatorPluginInterface;
use Drupal\tmgmt_deepl\DeeplTranslatorApi;
use Drupal\tmgmt_deepl\DeeplTranslatorBatch;
use Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslator;
use Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslatorInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the DeeplTranslatorBatch service.
 */
#[CoversClass(DeeplTranslatorBatch::class)]
#[Group('tmgmt_deepl')]
class DeeplTranslatorBatchTest extends UnitTestCase {

  /**
   * The job object.
   *
   * @var \Drupal\tmgmt\Entity\Job|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Job|MockObject $job;

  /**
   * The job item object.
   *
   * @var \Drupal\tmgmt\JobItemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected JobItemInterface|MockObject $jobItem;

  /**
   * The translator object.
   *
   * @var \Drupal\tmgmt\Entity\Translator|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Translator|MockObject $translator;

  /**
   * The DeepL translator plugin object.
   *
   * @var \Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslatorInterface|\PHPUnit\Framework\MockObject\MockObject|\Drupal\tmgmt\TranslatorPluginBase
   */
  protected DeeplTranslatorInterface|MockObject|TranslatorPluginBase $translatorPlugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->job = $this->createMock(Job::class);
    $this->jobItem = $this->createMock(JobItemInterface::class);
    $this->translatorPlugin = $this->createMock(DeeplTranslator::class);
    $this->translator = $this->createMock(Translator::class);
  }

  /**
   * Tests the buildBatch method.
   *
   * Tests the method ::buildBatch.
   */
  public function testBuildBatch(): void {
    $q = ['text1', 'text2', 'text3', 'text4', 'text5', 'text6'];
    $keys_sequence = ['key1', 'key2', 'key3', 'key4', 'key5', 'key6'];

    // Mock the batch builder.
    $batch_builder = $this->createMock(BatchBuilder::class);
    $batch_builder->expects($this->once())
      ->method('toArray')
      ->willReturn([]);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock([
      'createBatchBuilder',
      'addBatchOperations',
      'setBatch',
    ]);

    // Mocking method calls on the DeeplTranslatorBatch instance.
    $class->expects($this->once())
      ->method('createBatchBuilder')
      ->willReturn($batch_builder);

    $class->expects($this->once())
      ->method('addBatchOperations')
      ->with($batch_builder, $this->job, $this->jobItem, $q, $keys_sequence);

    $class->expects($this->once())
      ->method('setBatch')
      ->with([]);

    $class->buildBatch($this->job, $this->jobItem, $q, $keys_sequence);
  }

  /**
   * Tests the createBatchBuilder method.
   *
   * Tests the method ::createBatchBuilder.
   */
  public function testCreateBatchBuilder():void {
    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $batch_builder = $class->createBatchBuilder();

    // @phpstan-ignore-next-line.
    $this->assertInstanceOf(BatchBuilder::class, $batch_builder);
  }

  /**
   * Data provider for testTranslateOperation.
   *
   * @return array
   *   Test data for with and without translation context.
   */
  public static function dataProviderTranslateOperation(): array {
    return [
      'Without context' => [
        'context_setting' => NULL,
        'expected_context' => NULL,
      ],
      'With context' => [
        'context_setting' => 'Sample context text.',
        'expected_context' => 'Sample context text.',
      ],
    ];
  }

  /**
   * Tests the translateOperation method.
   *
   * @param string|null $context_setting
   *   The context setting for the job.
   * @param string|null $expected_context
   *   The expected context passed to the translator API.
   */
  #[DataProvider('dataProviderTranslateOperation')]
  public function testTranslateOperation(?string $context_setting, ?string $expected_context): void {
    $text = ['text1', 'text2'];
    $keys_sequence = ['key1', 'key2'];
    $context = [];

    // Mocking the translator and its plugin.
    $this->translator->expects($this->atLeastOnce())
      ->method('getPlugin')
      ->willReturn($this->translatorPlugin);

    $this->translatorPlugin->expects($this->once())
      ->method('getDefaultSettings')
      ->with($this->translator)
      ->willReturn([]);

    $this->translatorPlugin->expects($this->exactly(2))
      ->method('unescapeText')
      ->willReturnOnConsecutiveCalls('translated_text1', 'translated_text2');

    // Mocking the TextResult object.
    $translated_text1 = $this->createMock(TextResult::class);
    $translated_text2 = $this->createMock(TextResult::class);

    // Mock the job object.
    $this->job->expects($this->any())
      ->method('getTranslator')
      ->willReturn($this->translator);
    $this->job->expects($this->once())
      ->method('getRemoteSourceLanguage')
      ->willReturn('EN-GB');
    $this->job->expects($this->once())
      ->method('getRemoteTargetLanguage')
      ->willReturn('DE');
    $this->job->expects($this->any())
      ->method('getSetting')
      ->with('context')
      ->willReturn($context_setting);

    // Mock the DeeplApi.
    $tmgmt_deepl_api = $this->createMock(DeeplTranslatorApi::class);
    $tmgmt_deepl_api->expects($this->once())
      ->method('setTranslator')
      ->with($this->translator);
    $tmgmt_deepl_api->expects($this->once())
      ->method('fixSourceLanguageMappings')
      ->with('EN-GB')
      ->willReturn('EN');
    $tmgmt_deepl_api->expects($this->once())
      ->method('translate')
      ->with($this->translator, $text, 'EN', 'DE', [], $expected_context)
      ->willReturn([$translated_text1, $translated_text2]);

    // Mock the module handler.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslationContext']);
    $class->expects($this->once())
      ->method('getTranslationContext')
      ->with($this->job)
      ->willReturn($expected_context);

    $class->setDeeplApi($tmgmt_deepl_api);
    $class->setModuleHandler($module_handler);

    $class->translateOperation($this->job, $text, $keys_sequence, $context);

    /** @var array{
     *    'results': array{
     *      'translation': array{
     *        ...
     *      }
     *    }
     * } $context
     */
    $this->assertEquals([
      'key1' => ['#text' => 'translated_text1'],
      'key2' => ['#text' => 'translated_text2'],
    ], $context['results']['translation']);
  }

  /**
   * Tests the beforeFinishedOperation method.
   *
   * Tests the method ::beforeFinishedOperation.
   */
  public function testBeforeFinishedOperation(): void {
    $context = [];

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->beforeFinishedOperation($this->jobItem, $context);

    /** @var array{
     *    'results': array{
     *      'job_item': \Drupal\tmgmt\JobItemInterface
     *    }
     * } $context
     */
    $this->assertEquals($this->jobItem, $context['results']['job_item']);
  }

  /**
   * Tests the finishedOperation method.
   *
   * Tests the method ::finishedOperation.
   */
  public function testFinishedOperation(): void {
    $this->jobItem->expects($this->once())
      ->method('addTranslatedData')
      ->with(['key1' => ['#text' => 'translated_text1']]);
    $this->jobItem->expects($this->once())
      ->method('getJob')
      ->willReturn($this->job);

    $results = [
      'job_item' => $this->jobItem,
      'translation' => ['key1' => ['#text' => 'translated_text1']],
    ];
    $operations = [];

    // Mock the tmgmt Data object.
    $tmgmt_data = $this->createMock(Data::class);
    $tmgmt_data->expects($this->once())
      ->method('unflatten')
      ->with(['key1' => ['#text' => 'translated_text1']])
      ->willReturn(['key1' => ['#text' => 'translated_text1']]);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock(['tmgmtWriteRequestMessages']);
    $class->setTmgmtData($tmgmt_data);
    $class->expects($this->once())
      ->method('tmgmtWriteRequestMessages')
      ->with($this->job);

    $class->finishedOperation(TRUE, $results, $operations);
  }

  /**
   * Data provider for mergeTranslations.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestMergeTranslations(): array {
    return [
      'with available context' => [
        [
          'results' => [
            'translation' => [
              'existing_translation_1',
              'existing_translation_2',
            ],
          ],
        ],
        [
          'existing_translation_1',
          'existing_translation_2',
          'new_translation_1',
          'new_translation_2',
        ],
      ],
      'with empty context' => [
        [],
        ['new_translation_1', 'new_translation_2'],
      ],
    ];
  }

  /**
   * Tests the mergeTranslations method.
   *
   * @param array $context
   *   The given translation context (can be empty array or prefilled).
   * @param array $expected_result
   *   The expected result.
   *
   *   Test the method ::mergeTranslations.
   */
  #[DataProvider('dataProviderTestMergeTranslations')]
  public function testMergeTranslations(array $context, array $expected_result): void {
    $translation = ['new_translation_1', 'new_translation_2'];

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock(['tmgmtWriteRequestMessages']);

    $result = $class->mergeTranslations($context, $translation);
    $this->assertEquals($expected_result, $result);
  }

  /**
   * Data provider for decode and unescape text with tag handling.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestDecodeAndUnescapeText(): array {
    return [
      'xml' => [
        'xml',
        'translated &amp; text',
        'translated & text',
      ],
      'html' => [
        'html',
        'translated &amp; text',
        'translated & text',
      ],
      'off' => [
        '0',
        'translated text',
        'translated text',
      ],
    ];
  }

  /**
   * Tests decodeAndUnescapeText method.
   *
   * @param string $tag_handling
   *   Type of tag handling (can be 0, html or xml)
   * @param string $text
   *   The text to be decoded/ unescaped.
   * @param string $expected_result
   *   The expected result.
   *
   *   Test the method ::decodeAndUnescapeText.
   */
  #[DataProvider('dataProviderTestDecodeAndUnescapeText')]
  public function testDecodeAndUnescapeText(string $tag_handling, string $text, string $expected_result): void {
    $this->translator->expects($this->once())
      ->method('getSetting')
      ->with('tag_handling')
      ->willReturn($tag_handling);

    // Mock the translated text.
    $translated_text = $this->createMock(TextResult::class);
    $translated_text->text = $text;

    // Expect Html::decodeEntities to be called.
    if ($tag_handling === 'xml' or $tag_handling === 'html') {
      $text = Html::decodeEntities($text);
    }

    // Expect the unescapeText method to be called with decoded text.
    $this->translatorPlugin->expects($this->once())
      ->method('unescapeText')
      ->with($text)
      ->willReturn($expected_result);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();

    // Call the method and assert the result.
    $result = $class->decodeAndUnescapeText($this->translator, $this->translatorPlugin, $translated_text);
    $this->assertEquals($expected_result, $result);
  }

  /**
   * Tests the processTranslatedTexts method.
   *
   * Tests the method ::processTranslatedTexts.
   */
  public function testProcessTranslatedTexts(): void {
    $translated_texts = [
      $this->createMock(TextResult::class),
      // Mock for the first translated text.
      $this->createMock(TextResult::class),
      // Mock for the second translated text.
    ];

    $keys_sequence = ['key1', 'key2'];
    $i = 0;

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock(['decodeAndUnescapeText']);

    // Set expected values for decodeAndUnescapeText method.
    $class->expects($this->exactly(2))
      ->method('decodeAndUnescapeText')
      ->willReturnMap([
          [$this->translator, $this->translatorPlugin, $translated_texts[0], 'decoded text 1'],
          [$this->translator, $this->translatorPlugin, $translated_texts[1], 'decoded text 2'],
      ]
      );

    // Call the method under test.
    $result = $class->processTranslatedTexts($this->translatorPlugin, $this->translator, $translated_texts, $keys_sequence, $i);

    // Expected result after processing.
    $expected_result = [
      'key1' => ['#text' => 'decoded text 1'],
      'key2' => ['#text' => 'decoded text 2'],
    ];

    // Assert the result matches the expected translation array.
    $this->assertEquals($expected_result, $result);

    // Assert that the counter $i has been incremented correctly.
    $this->assertEquals(2, $i);
  }

  /**
   * Data provider for the ensureArray test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestEnsureArray(): array {
    return [
      'string' => [
        'hello world',
        ['hello world'],
      ],
      'array' => [
        ['hello world', 'hello'],
        ['hello world', 'hello'],
      ],
    ];
  }

  /**
   * Tests the ensureArray function.
   *
   * @param mixed $input
   *   The input value.
   * @param array $expected_result
   *   The expected result.
   *
   *   Test the method ::ensureArray.
   */
  #[DataProvider('dataProviderTestEnsureArray')]
  public function testEnsureArray(mixed $input, array $expected_result): void {
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $result = $class->ensureArray($input);

    $this->assertEquals($expected_result, $result);
  }

  /**
   * Tests getTranslationContext method.
   *
   * Tests the method ::getTranslationContext.
   */
  public function testGetTranslationContext(): void {
    // Create a mock translator that includes the `enable_context` setting.
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with('enable_context')
      ->willReturn(TRUE);

    // Create a mock job and attach the translator.
    $job = $this->createMock(Job::class);
    $job->expects($this->once())
      ->method('getTranslator')
      ->willReturn($translator);

    // Mock the job's getSetting method for the 'context' key.
    $job->expects($this->once())
      ->method('getSetting')
      ->with('context')
      ->willReturn('Sample context text.');

    // Create an instance of the class under test.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();

    // Call getTranslationContext and assert the result.
    $result = $class->getTranslationContext($job);

    // Assert that the correct context is returned.
    $this->assertEquals('Sample context text.', $result, 'The context should match the expected value.');
  }

  /**
   * Tests getTranslationOptions method.
   *
   * Tests the method ::getTranslationOptions.
   */
  public function testGetTranslationOptions(): void {
    $default_options = ['option1' => 'value1', 'option2' => 'value2'];
    $altered_options = [
      'option1' => 'value1',
      'option2' => 'value2',
      'altered' => 'value3',
    ];

    $this->translator->expects($this->once())
      ->method('getPlugin')
      ->willReturn($this->translatorPlugin);

    // Mock the module handler.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);

    // Mock the job entity.
    $this->job = $this->createMock(Job::class);
    $this->job->method('getSetting')
      ->willReturnMap([
        ['option1', 'value1'],
        ['option2', 'value2'],
      ]);

    // Mock getDefaultSettings method of the translator plugin.
    $this->translatorPlugin->expects($this->once())
      ->method('getDefaultSettings')
      ->with($this->translator)
      ->willReturn($default_options);

    // Mock the alter method of the module handler and simulate alteration.
    $module_handler->expects($this->once())
      ->method('alter')
      ->with('tmgmt_deepl_translate_options', $this->job, $default_options)
      ->willReturnCallback(function ($hook, $translation_job, &$options) use ($altered_options) {
        $options = $altered_options;
      });

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setModuleHandler($module_handler);

    // Call the method under test.
    $result = $class->getTranslationOptions($this->job, $this->translator);

    // Assert the result is the altered options array.
    $this->assertEquals($altered_options, $result);
  }

  /**
   * Data provider for the initializeContext test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestInitializeContext(): array {
    return [
      'no results given' => [
        ['results' => []],
        0,
      ],
      'results given' => [
        ['results' => ['i' => 3]],
        3,
      ],
    ];
  }

  /**
   * Tests the method ::initializeContext.
   *
   * @param array $context
   *   The context array.
   * @param int $expected_result
   *   The expected result.
   */
  #[DataProvider('dataProviderTestInitializeContext')]
  public function testInitializeContext(array $context, int $expected_result): void {
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();
    $result = $class->initializeContext($context);

    // Assert that 'i' is initialized to 0.
    $this->assertArrayHasKey('results', $result);
    assert(is_array($result['results']));
    $this->assertArrayHasKey('i', $result['results']);
    $this->assertEquals($expected_result, $result['results']['i']);
  }

  /**
   * Tests the method ::addBatchOperations.
   */
  public function testAddBatchOperations(): void {
    // Prepare test data.
    $q = ['text1', 'text2', 'text3', 'text4', 'text5', 'text6', 'text7'];
    $keys_sequence = ['key1', 'key2', 'key3', 'key4', 'key5', 'key6', 'key7'];

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test&MockObject $class */
    $class = $this->createClassMock();

    // Mock the batch builder.
    $batch_builder = $this->createMock(BatchBuilder::class);

    // Define expected arguments for each invocation of addOperation.
    $expected_calls = [
      // First call to translateOperation with the first batch of texts.
      [
        [$class, 'translateOperation'],
        [$this->job, ['text1', 'text2', 'text3', 'text4', 'text5'], $keys_sequence],
      ],
      // Second call to translateOperation with the remaining texts.
      [
        [$class, 'translateOperation'],
        [$this->job, ['text6', 'text7'], $keys_sequence],
      ],
      // Final call to beforeFinishedOperation.
      [
        [$class, 'beforeFinishedOperation'],
        [$this->jobItem],
      ],
    ];

    // Initialize a counter to track the calls.
    $callIndex = 0;

    // Set the expectation with a callback to check arguments.
    $batch_builder->expects($this->exactly(3))
      ->method('addOperation')
      ->willReturnCallback(function ($method, $args) use (&$callIndex, $expected_calls) {
        [$expected_method, $expected_args] = $expected_calls[$callIndex];
        $this->assertEquals($expected_method, $method);
        $this->assertEquals($expected_args, $args);
        $callIndex++;
      });

    // Call the method under test.
    $class->addBatchOperations($batch_builder, $this->job, $this->jobItem, $q, $keys_sequence);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorBatch_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplTranslatorBatch_Test|MockObject {
    return $this->getMockBuilder(DeeplTranslatorBatch_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart

/**
 * Mocked DeeplTranslatorUi class for tests.
 */
class DeeplTranslatorBatch_Test extends DeeplTranslatorBatch {

  /**
   * Setter for the deeplApi property to allow manual setting in tests.
   *
   * @param \Drupal\tmgmt_deepl\DeeplTranslatorApi $deeplApi
   *  The DeepLTranslatorApi.
   */
  public function setDeeplApi(DeeplTranslatorApi $deeplApi): void {
    $this->deeplApi = $deeplApi;
  }

  /**
   * Setter for the module handler property to allow manual setting in tests.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *  The module handler.
   */
  public function setModuleHandler(ModuleHandlerInterface $moduleHandler): void {
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * Setter for the tmgmtData property to allow manual setting in tests.
   *
   * @param \Drupal\tmgmt\Data $tmgmtData
   *  The tmgmt data service.
   */
  public function setTmgmtData(Data $tmgmtData): void {
    $this->tmgmtData = $tmgmtData;
  }

  /**
   * {@inheritDoc}
   */
  public function getTranslationContext(JobInterface $job): null|string {
    return parent::getTranslationContext($job);
  }

  /**
   * {@inheritDoc}
   */
  public function getTranslationOptions(JobInterface $job, $translator): array {
    return parent::getTranslationOptions($job, $translator);
  }

  /**
   * {@inheritDoc}
   */
  public function mergeTranslations(array $context, array $translation): array {
    return parent::mergeTranslations($context, $translation);
  }

  /**
   * {@inheritDoc}
   */
  public function decodeAndUnescapeText(TranslatorInterface $translator, TranslatorPluginInterface $translator_plugin, TextResult $translated_text): string {
    return parent::decodeAndUnescapeText($translator, $translator_plugin, $translated_text);
  }

  /**
   * {@inheritDoc}
   */
  public function processTranslatedTexts(TranslatorPluginInterface $translator_plugin, TranslatorInterface $translator, array $translated_texts, array $keys_sequence, int &$i): array {
    return parent::processTranslatedTexts($translator_plugin, $translator, $translated_texts, $keys_sequence, $i);
  }

  /**
   * {@inheritDoc}
   */
  public function ensureArray(mixed $translated_texts): array {
    return parent::ensureArray($translated_texts);
  }

  /**
   * {@inheritDoc}
   */
  public function initializeContext(array $context): array {
    return parent::initializeContext($context);
  }

  /**
   * {@inheritDoc}
   */
  public function createBatchBuilder(): BatchBuilder {
    return parent::createBatchBuilder();
  }

  /**
   * {@inheritDoc}
   */
  public function addBatchOperations(BatchBuilder $batch_builder, Job $job, JobItemInterface $job_item, array $q, array $keys_sequence, array $documents = []): void {
    parent::addBatchOperations($batch_builder, $job, $job_item, $q, $keys_sequence, $documents);
  }

}
// @codingStandardsIgnoreEnd
