<?php

namespace Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl\DeeplTranslatorApi;
use Drupal\tmgmt_deepl\DeeplTranslatorBatch;
use Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslator;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests the DeeplTranslator.
 */
#[CoversClass(DeeplTranslator::class)]
#[Group('tmgmt_deepl')]
class DeeplTranslatorTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create/register string translation mock.
    $container = new ContainerBuilder();
    $container->set('string_translation', $this->getStringTranslationStub());
    \Drupal::setContainer($container);
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
    $container = new ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Data provider for the testCheckAvailable test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestCheckAvailable(): array {
    return [
      'valid key' => [
        'test_key',
        TRUE,
      ],
      'invalid or no key' => [
        '',
        FALSE,
      ],
    ];
  }

  /**
   * Test checkAvailable method.
   *
   * @param string $key
   *   The auth key.
   * @param bool $expected_result
   *   The expected result can be true or false.
   *
   *   Test the method ::checkAvailable.
   */
  #[DataProvider('dataProviderTestCheckAvailable')]
  public function testCheckAvailable(string $key, mixed $expected_result): void {
    // Create a mock translator.
    $translator = $this->createMock(Translator::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with('auth_key')
      ->willReturn($key);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock();

    if ($expected_result) {
      $result = $class->checkAvailable($translator);
      $this->assertTrue($result->getSuccess());
    }
    else {
      $translator->expects($this->once())
        ->method('label')
        ->willReturn('Test Translator');

      $translator->expects($this->once())
        ->method('toUrl')
        ->willReturn($this->createMock('\Drupal\Core\Url'));

      $result = $class->checkAvailable($translator);
      $this->assertFalse($result->getSuccess());

      // @phpstan-ignore-next-line.
      $this->assertInstanceOf('\Drupal\Core\StringTranslation\TranslatableMarkup', $result->getReason());
      $this->assertEquals('@translator is not available. Make sure it is properly <a href=:configured>configured</a>.', $result->getReason()->getUntranslatedString());
    }
  }

  /**
   * Test requestTranslation method.
   *
   * Tests the method ::requestTranslation.
   */
  public function testRequestTranslation(): void {
    // Create a mock for the job.
    $job = $this->createMock(Job::class);

    // Create mocks two job items.
    $job_item_1 = $this->createMock(JobItemInterface::class);
    $job_item_2 = $this->createMock(JobItemInterface::class);

    // Set up the job items.
    $job->expects($this->once())
      ->method('getItems')
      ->willReturn([$job_item_1, $job_item_2]);

    // Set up expectation for isRejected().
    $job->expects($this->once())
      ->method('isRejected')
      ->willReturn(FALSE);

    $job->expects($this->once())
      ->method('submitted')
      ->with('The translation job has been submitted.');

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock(['requestJobItemsTranslation']);
    $class->expects($this->once())
      ->method('requestJobItemsTranslation')
      ->with([$job_item_1, $job_item_2]);

    $class->requestTranslation($job);
  }

  /**
   * Tests getDefaultRemoteLanguagesMappings method.
   *
   * Tests the method ::getDefaultRemoteLanguagesMappings.
   */
  public function testGetDefaultRemoteLanguagesMappings():void {
    // Define expected mappings.
    $expected_mappings = [
      'ar' => 'AR',
      'bg' => 'BG',
      'cs' => 'CS',
      'da' => 'DA',
      'de' => 'DE',
      'el' => 'EL',
      'en' => 'EN-GB',
      'es' => 'ES',
      'et' => 'ET',
      'fi' => 'FI',
      'fr' => 'FR',
      'hu' => 'HU',
      'id' => 'ID',
      'it' => 'IT',
      'ja' => 'JA',
      'ko' => 'KO',
      'lt' => 'LT',
      'lv' => 'LV',
      'nb' => 'NB',
      'nl' => 'NL',
      'pl' => 'PL',
      'pt-br' => 'PT-BR',
      'pt-pt' => 'PT-PT',
      'ro' => 'RO',
      'ru' => 'RU',
      'sk' => 'SK',
      'sl' => 'SL',
      'sv' => 'SV',
      'tr' => 'TR',
      'uk' => 'UK',
      'zh-hans' => 'ZH',
      'zh-hant' => 'ZH',
    ];

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock();

    $this->assertEquals($expected_mappings, $class->getDefaultRemoteLanguagesMappings());
    $mappings = $class->getDefaultRemoteLanguagesMappings();

    // Assert that all expected keys are present.
    foreach ($expected_mappings as $key => $value) {
      $this->assertArrayHasKey($key, $mappings);
      $this->assertEquals($value, $mappings[$key]);
    }
  }

  /**
   * Tests getSupportedRemoteLanguages method.
   *
   * Tests the method ::getSupportedRemoteLanguages.
   */
  public function testGetSupportedRemoteLanguages():void {
    // Create a mock translator.
    $translator = $this->createMock(Translator::class);

    // Create a mock DeepL API.
    $tmgmt_deepl_api = $this->createMock(DeeplTranslatorApi::class);

    // Mock some languages.
    $available_languages = [
      'EN' => 'English',
      'DE' => 'German',
      'FR' => 'French',
      'ES' => 'Spanish',
    ];

    // Set up expectation for getTargetLanguages().
    $tmgmt_deepl_api->expects($this->once())
      ->method('getTargetLanguages')
      ->willReturn($available_languages);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setDeeplApi($tmgmt_deepl_api);

    // Set the translator for the DeepL API.
    $tmgmt_deepl_api->expects($this->once())
      ->method('setTranslator')
      ->with($translator);

    $result = $class->getSupportedRemoteLanguages($translator);
    $this->assertEquals($available_languages, $result);
  }

  /**
   * Data provider for the testGetSupportedTargetLanguages test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestGetSupportedTargetLanguages(): array {
    return [
      'Source language equals target language' => [
        'DE',
        [
          'EN' => 'English',
          'FR' => 'French',
          'ES' => 'Spanish',
          'IT' => 'Italian',
        ],
      ],
      'Source language is not available' => [
        'ZH',
        [],
      ],
      'Source language is empty' => [
        '',
        [],
      ],
    ];
  }

  /**
   * Tests getSupportedTargetLanguages method.
   *
   * @param string $source_language
   *   The source language.
   * @param array $expected_result
   *   The expected target languages.
   *
   *   Test the method ::getSupportedTargetLanguages.
   */
  #[DataProvider('dataProviderTestGetSupportedTargetLanguages')]
  public function testGetSupportedTargetLanguages(string $source_language, array $expected_result):void {
    // Create a mock translator.
    $translator = $this->createMock(Translator::class);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock(['getSupportedRemoteLanguages']);

    // Mock some languages.
    $available_languages = [
      'EN' => 'English',
      'DE' => 'German',
      'FR' => 'French',
      'ES' => 'Spanish',
      'IT' => 'Italian',
    ];

    // Set up expectation for getSupportedRemoteLanguages().
    $class->expects($this->once())
      ->method('getSupportedRemoteLanguages')
      ->with($translator)
      ->willReturn($available_languages);

    // Source language is available.
    $target_languages = $class->getSupportedTargetLanguages($translator, $source_language);
    $this->assertEquals($expected_result, $target_languages);
  }

  /**
   * Data provider for the testFixSourceLanguageMappings test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestFixSourceLanguageMappings(): array {
    return [
      'EN-GB' => [
        'EN-GB',
        'EN',
      ],
      'EN-US' => [
        'EN-US',
        'EN',
      ],
      'PT-BR' => [
        'PT-BR',
        'PT',
      ],
      'PT-PT' => [
        'PT-PT',
        'PT',
      ],
      'Unmapped language' => [
        'FR',
        'FR',
      ],
    ];
  }

  /**
   * Tests getSupportedTargetLanguages method.
   *
   * @param string $source_language
   *   The source language.
   * @param string $expected_result
   *   The expected language mapping.
   *
   *   Test the method ::fixSourceLanguageMappings.
   */
  #[DataProvider('dataProviderTestFixSourceLanguageMappings')]
  public function testFixSourceLanguageMappings(string $source_language, string $expected_result):void {
    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock();

    $result = $class->fixSourceLanguageMappings($source_language);
    $this->assertEquals($expected_result, $result);
  }

  /**
   * Data provider for the testHasCheckoutSettings test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestHasCheckoutSettings(): array {
    return [
      'Default - should return false' => [
        '',
        FALSE,
        FALSE,
      ],
      'Set alteration to be true' => [
        TRUE,
        TRUE,
        FALSE,
      ],
      'Set alteration and context to be false' => [
        FALSE,
        FALSE,
        FALSE,
      ],
      'Set alteration and context to be true' => [
        TRUE,
        TRUE,
        TRUE,
      ],
      'Set alteration to be false and context to be true' => [
        FALSE,
        TRUE,
        TRUE,
      ],
    ];
  }

  /**
   * Tests hasCheckoutSettings method.
   *
   * @param mixed $altered_value
   *   The value can be true, false or empty.
   * @param bool $expected_result
   *   Whether the method should return true or false.
   * @param bool $enable_context
   *   Whether the context setting is enabled or not.
   *
   *   Test the method ::hasCheckoutSettings.
   */
  #[DataProvider('dataProviderTestHasCheckoutSettings')]
  public function testHasCheckoutSettings(mixed $altered_value, bool $expected_result, bool $enable_context = FALSE): void {
    // Create a mock for the job.
    $job = $this->createMock(Job::class);
    $translator = $this->createMock(TranslatorInterface::class);

    // Mock the getSetting method of the translator.
    $translator->expects($this->once())
      ->method('getSetting')
      ->with('enable_context')
      ->willReturn($enable_context);
    $job->method('getTranslator')->willReturn($translator);

    // Mock the module handler.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);

    // In case of having empty alteration.
    if ($altered_value === '') {
      $module_handler->expects($this->once())
        ->method('alter')
        ->with('tmgmt_deepl_has_checkout_settings', $this->isFalse(), $job);
    }
    else {
      $module_handler->expects($this->once())
        ->method('alter')
        ->willReturnCallback(function ($hook, &$has_checkout_settings, $jobArg) use ($job, $altered_value) {
          $this->assertEquals('tmgmt_deepl_has_checkout_settings', $hook);
          $this->assertSame($job, $jobArg);
          $has_checkout_settings = $altered_value;
        });
    }

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setModuleHandler($module_handler);
    $result = $class->hasCheckoutSettings($job);
    $this->assertEquals($expected_result, $result);
  }

  /**
   * Data provider for the testGetDefaultSettings test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestGetDefaultSettings(): array {
    return [
      'All settings provided' => [
        [
          'model_type' => 'latency_optimized',
          'split_sentences' => '1',
          'preserve_formatting' => '0',
          'formality' => 'default',
          'tag_handling' => 'xml',
          'outline_detection' => '1',
          'splitting_tags' => 'p,br',
          'non_splitting_tags' => 'div,span',
          'ignore_tags' => 'script,style',
        ],
        [
          'model_type' => 'latency_optimized',
          'split_sentences' => '1',
          'preserve_formatting' => '0',
          'formality' => 'default',
          'tag_handling' => 'xml',
          'outline_detection' => '1',
          'splitting_tags' => 'p,br',
          'non_splitting_tags' => 'div,span',
          'ignore_tags' => 'script,style',
        ],
      ],
      'Some settings empty' => [
        [
          'model_type' => 'latency_optimized',
          'split_sentences' => '1',
          'preserve_formatting' => '',
          'formality' => 'default',
          'tag_handling' => '',
          'outline_detection' => '1',
          'splitting_tags' => '',
          'non_splitting_tags' => 'div,span',
          'ignore_tags' => '',
        ],
        [
          'model_type' => 'latency_optimized',
          'split_sentences' => '1',
          'formality' => 'default',
          'outline_detection' => '1',
          'non_splitting_tags' => 'div,span',
        ],
      ],
      'All settings empty' => [
        [
          'model_type' => '',
          'split_sentences' => '',
          'preserve_formatting' => '',
          'formality' => '',
          'tag_handling' => '',
          'outline_detection' => '',
          'splitting_tags' => '',
          'non_splitting_tags' => '',
          'ignore_tags' => '',
        ],
        [],
      ],
    ];
  }

  /**
   * Tests getDefaultSettings method.
   *
   * @param array $settings
   *   The provided settings array.
   * @param array $expected_result
   *   The expected settings array.
   *
   *   Test the method ::getDefaultSettings.
   */
  #[DataProvider('dataProviderTestGetDefaultSettings')]
  public function testGetDefaultSettings(array $settings, array $expected_result): void {
    // Create a mock translator.
    $translator = $this->createMock(Translator::class);
    // Set up the getSettings method to return values based on the input.
    $translator->method('getSettings')->willReturn($settings);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock();
    $result = $class->getDefaultSettings($translator);
    $this->assertEquals($expected_result, $result);
  }

  /**
   * Data provider for the testRequestJobItemsTranslation test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestRequestJobItemsTranslation(): array {
    return [
      'no cron execution, not continuous' => [
        FALSE,
        FALSE,
      ],
      'cron execution, not continuous' => [
        TRUE,
        FALSE,
      ],
      'no cron execution, continuous' => [
        FALSE,
        TRUE,
      ],
      'cron execution, continuous' => [
        TRUE,
        TRUE,
      ],
    ];
  }

  /**
   * Test the method ::requestJobItemsTranslation.
   *
   * @param bool $is_cron
   *   Whether the cron is enabled or not.
   * @param bool $is_continuous
   *   Whether the job is continuous or not.
   */
  #[DataProvider('dataProviderTestRequestJobItemsTranslation')]
  public function testRequestJobItemsTranslation(bool $is_cron, bool $is_continuous = FALSE): void {
    // Create a mock job.
    $job = $this->createMock(Job::class);

    // Create a mock job item.
    $job_item = $this->createMock(JobItemInterface::class);

    // Create a mock job item.
    $tmgmt_data = $this->createMock(Data::class);

    // Create a mock queue.
    $queue = $this->createMock(QueueInterface::class);

    // Mick the DeepL translator batch class.
    $batch = $this->createMock(DeeplTranslatorBatch::class);

    // Set up the job item expectations.
    $job_item->expects($this->once())
      ->method('getJob')
      ->willReturn($job);

    // Set up the job expectations.
    $job->expects($this->once())
      ->method('isContinuous')
      ->willReturn($is_continuous);

    // Set up the job expectations.
    if ($is_continuous) {
      $job_item->expects($this->once())
        ->method('active');
    }
    else {
      $job_item->expects($this->never())
        ->method('active');
    }

    $job_item->method('getData')->willReturn([]);

    // Set up the tmgmt data expectations.
    $tmgmt_data->expects($this->once())
      ->method('filterTranslatable')
      ->willReturn([
        'key1' => ['#text' => 'value1', '#escape' => []],
        'key2' => ['#text' => 'value2', '#escape' => []],
      ]);

    // Mock the class.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock(['isCron', 'escapeText']);
    // Set the tmgmt data property.
    $class->setTmgmtData($tmgmt_data);
    // Set the queue property.
    $class->setQueue($queue);
    // Set the batch property.
    $class->setBatch($batch);

    // Set up isCron expectations.
    $class->expects($this->once())
      ->method('isCron')
      ->willReturn($is_cron);

    // Set up escapeText expectations.
    $class->expects($this->atLeastOnce())
      ->method('escapeText')
      ->willReturnCallback(function (array $data_item) {
        assert(is_string($data_item['#text']));

        return 'escaped__' . $data_item['#text'];
      });

    // In case cron is enabled.
    if ($is_cron) {
      // Set up queue expectation for cron.
      $queue->expects($this->once())
        ->method('createItem')
        ->with($this->callback(function ($item) use ($job, $job_item) {
          assert(is_array($item));

          return $item['job'] === $job &&
            $item['job_item'] === $job_item &&
            $item['q'] === ['escaped__value1', 'escaped__value2'] &&
            $item['keys_sequence'] === ['key1', 'key2'];
        }));

      // Ensure buildBatch is not called during cron.
      $batch->expects($this->never())
        ->method('buildBatch');
    }
    else {
      // Set up the batch expectations.
      $batch->expects($this->once())
        ->method('buildBatch')
        ->with(
          $job,
          $job_item,
          $this->equalTo(['escaped__value1', 'escaped__value2']),
          $this->equalTo(['key1', 'key2']),
        );

      // Ensure createItem is not called during non-cron.
      $queue->expects($this->never())
        ->method('createItem');
    }

    // Call method.
    $class->requestJobItemsTranslation([$job_item]);
  }

  /**
   * Data provider for the testIsCron test.
   *
   * @return array
   *   The test data.
   */
  public static function dataProviderTestIsCron(): array {
    return [
      // Cron execution exists in backtrace.
      'Cron execution' => [
        [
          ['function' => 'some_function'],
          ['function' => 'tmgmt_cron'],
          ['function' => 'another_function'],
        ],
        TRUE,
      ],
      // Cron execution not found in backtrace.
      'No cron execution' => [
        [
          ['function' => 'some_function'],
          ['function' => 'another_function'],
        ],
        FALSE,
      ],
      // Empty backtrace.
      'Empty backtrace' => [
        [],
        FALSE,
      ],
      // Backtrace caching.
      'Cache result' => [
        [
          ['function' => 'tmgmt_cron'],
        ],
        TRUE,
      ],
    ];
  }

  /**
   * Tests the method ::isCron.
   *
   * @param array $backtrace
   *   The array of functions.
   * @param bool $expected_result
   *   Whether cron is running or not.
   */
  #[DataProvider('dataProviderTestIsCron')]
  public function testIsCron(array $backtrace, bool $expected_result): void {
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test&MockObject $class */
    $class = $this->createClassMock(['getDebugBacktrace']);

    // Set up the getDebugBacktrace method to return test backtrace.
    $class->expects($this->once())
      ->method('getDebugBacktrace')
      ->willReturn($backtrace);

    $result = $class->isCron(3);
    $this->assertEquals($expected_result, $result);

    // Force-set the isCron property to simulate a cached value.
    $reflection = new \ReflectionClass($class);
    $is_cron = $reflection->getProperty('isCron');
    $is_cron->setValue($class, $expected_result);

    // Ensure that once `isCron` is called, the cached value is used
    // on subsequent calls.
    $class->expects($this->never())
      ->method('getDebugBacktrace');

    // Call `isCron` again and ensure it uses the cached value.
    $cachedResult = $class->isCron(3);
    $this->assertEquals($expected_result, $cachedResult);

    // Also check for the negative case (when cached is false).
    $is_cron->setValue($class, !$expected_result);

    $negative_cached_result = $class->isCron(3);
    $this->assertEquals(!$expected_result, $negative_cached_result);
  }

  /**
   * Tests the method ::getTranslators.
   */
  public function testGetTranslators(): void {
    // Mock the entity storage.
    $entity_storage = $this->createMock(EntityStorageInterface::class);
    // Mock the entity type manager.
    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);

    // Set up expectations for entity type manager.
    $entity_type_manager->expects($this->once())
      ->method('getStorage')
      ->with('tmgmt_translator')
      ->willReturn($entity_storage);

    // Create mock translators.
    $translator_1 = $this->createMock('Drupal\tmgmt\Entity\Translator');
    $translator_2 = $this->createMock('Drupal\tmgmt\Entity\Translator');
    $translator_3 = $this->createMock('Drupal\tmgmt\Entity\Translator');

    // Set up the expectations for the storage.
    $entity_storage->expects($this->once())
      ->method('loadByProperties')
      ->with(['plugin' => DeeplTranslator::DEEPL_TRANSLATORS])
      ->willReturn([
        'deepl_pro' => $translator_1,
        'deepl_free' => $translator_2,
        'deepl_api' => $translator_3,
      ]);

    // Set up the expectations for the translators.
    $translator_1->expects($this->once())
      ->method('label')
      ->willReturn('DeepL API Pro (deprecated)');
    $translator_2->expects($this->once())
      ->method('label')
      ->willReturn('DeepL API Free (deprecated)');
    $translator_3->expects($this->once())
      ->method('label')
      ->willReturn('DeepL API');

    // Replace global entity type manager service.
    $container = new ContainerBuilder();
    $container->set('entity_type.manager', $entity_type_manager);
    \Drupal::setContainer($container);

    $result = DeeplTranslator::getTranslators();
    $expected = [
      'deepl_pro' => 'DeepL API Pro (deprecated)',
      'deepl_free' => 'DeepL API Free (deprecated)',
      'deepl_api' => 'DeepL API',
    ];
    $this->assertEquals($expected, $result);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl\Unit\Plugin\tmgmt\Translator\DeeplTranslator_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplTranslator_Test|MockObject {
    return $this->getMockBuilder(DeeplTranslator_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart

/**
 * Mocked DeeplTranslator class for tests.
 */
class DeeplTranslator_Test extends DeeplTranslator {

  /**
   * {@inheritdoc}
   */
  public ?bool $isCron = null;

  /**
   * {@inheritdoc}
   */
  public function isCron(int $backtrace_limit = 5): bool {
   return parent::isCron($backtrace_limit);
  }

  /**
   * Reset the isCron property to null.
   */
  public function resetIsCron(): void {
    $this->isCron = null;
  }

  /**
   * Setter for the deeplApi property to allow manual setting in tests.
   *
   * @param \Drupal\tmgmt_deepl\DeeplTranslatorApi $deepl_api
   *  The DeepLTranslatorApi.
   */
  public function setDeeplApi(DeeplTranslatorApi $deepl_api): void {
    $this->deeplTranslatorApi = $deepl_api;
  }

  /**
   * Setter for the module handler property to allow manual setting in tests.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *  The module handler.
   */
  public function setModuleHandler(ModuleHandlerInterface $moduleHandler): void {
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * Setter for the tmgmtData property to allow manual setting in tests.
   *
   * @param \Drupal\tmgmt\Data $tmgmtData
   *  The tmgmt data service.
   */
  public function setTmgmtData(Data $tmgmtData): void {
    $this->tmgmtData = $tmgmtData;
  }

  /**
   * Setter for the queue property to allow manual setting in tests.
   *
   * @param QueueInterface $queue
   *  The queue service.
   */
  public function setQueue(QueueInterface $queue): void {
    $this->queue = $queue;
  }

  /**
   * Setter for the batch property to allow manual setting in tests.
   *
   * @param \Drupal\tmgmt_deepl\DeeplTranslatorBatch $batch
   *  The queue service.
   */
  public function setBatch(DeeplTranslatorBatch $batch): void {
    $this->deeplTranslatorBatch = $batch;
  }
}
// @codingStandardsIgnoreEnd
