<?php

namespace Drupal\tmgmt_deepl_glossary\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl_glossary\DeeplGlossaryInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the DeepL glossary entity.
 *
 * @deprecated in tmgmt_deepl:2.3.0-alpha3 and is removed from tmgmt_deepl:2.3.0-alpha5. Use deepl_ml_glossary entity instead.
 * @see https://www.drupal.org/project/tmgmt_deepl/issues/3528324
 *
 * @ContentEntityType(
 *   id = "deepl_glossary",
 *   label = @Translation("DeepL glossary"),
 *   label_singular = @Translation("DeepL glossary"),
 *   label_plural = @Translation("DeepL glossaries"),
 *   handlers = {
 *     "access" = "Drupal\tmgmt_deepl_glossary\AccessControlHandler",
 *     "list_builder" = "Drupal\tmgmt_deepl_glossary\Controller\DeeplGlossaryListBuilder",
 *     "views_data" = "Drupal\tmgmt_deepl_glossary\Entity\ViewsData\DeeplGlossaryViewsData",
 *     "form" = {
 *       "default" = "Drupal\tmgmt_deepl_glossary\Form\DeeplGlossaryForm",
 *       "add" = "Drupal\tmgmt_deepl_glossary\Form\DeeplGlossaryForm",
 *       "edit" = "Drupal\tmgmt_deepl_glossary\Form\DeeplGlossaryForm",
 *       "delete" = "Drupal\tmgmt_deepl_glossary\Form\DeeplGlossaryDeleteForm",
 *     },
 *     "route_provider" = {
 *        "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "tmgmt_deepl_glossary",
 *   translatable = FALSE,
 *   admin_permission = "administer deepl_glossary entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "ready" = "ready",
 *     "uid" = "uid",
 *   },
 * )
 */
class DeeplGlossary extends ContentEntityBase implements DeeplGlossaryInterface {

  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Name associated with the glossary.
    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the glossary.'))
      ->setRequired(TRUE)
      ->addConstraint('UniqueField')
      ->setDisplayOptions('form', [
        'type' => 'string',
      ])
      ->setDisplayConfigurable('form', TRUE);

    // The language in which the target texts in the glossary are specified.
    $fields['source_lang'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Source language'))
      ->setDescription(t('The language in which the source texts in the glossary are specified.'))
      ->setSetting('allowed_values_function', static::class . '::getAllowedLanguages')
      ->setRequired(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'options_select',
      ])
      ->setDisplayConfigurable('form', TRUE);

    // The language in which the source texts in the glossary are specified.
    $fields['target_lang'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Target language'))
      ->setDescription(t('The language in which the target texts in the glossary are specified.'))
      ->setSetting('allowed_values_function', static::class . '::getAllowedLanguages')
      ->setRequired(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'options_select',
      ])
      ->setDisplayConfigurable('form', TRUE);

    // The user id of the current user.
    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setDescription(t('The author of the glossary entry.'))
      ->setSetting('target_type', 'user')
      ->setReadOnly(TRUE);

    // The machine name of the translator.
    $fields['tmgmt_translator'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Translator'))
      ->setDescription(t('The tmgmt translator.'))
      ->setSetting('allowed_values_function', '\Drupal\tmgmt_deepl\Plugin\tmgmt\Translator\DeeplTranslator::getTranslators')
      ->setRequired(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'options_select',
      ])
      ->setDisplayConfigurable('form', TRUE);

    // A unique ID assigned to the glossary (values is retrieved by DeepL API)
    $fields['glossary_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Glossary Id'))
      ->setDescription(t('The glossary id.'));

    // A boolean that indicates if the newly created glossary can already be
    // used in translate requests.
    $fields['ready'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Ready state'))
      ->setDescription(t('A boolean that indicates if the newly created glossary can already be used in translate requests.'))
      ->setDefaultValue(FALSE);

    // The time that the entity was created.
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    // The number of entries in the glossary.
    $fields['entry_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Entry count'))
      ->setDescription(t('The number of entries in the glossary.'))
      ->setReadOnly(TRUE);

    // The format in which the glossary entries are provided.
    $fields['entries_format'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Entries format'))
      ->setDescription(t('The format in which the glossary entries are provided.'))
      ->setSetting('allowed_values', [['tsv' => 'text/tab-separated-values']])
      ->setReadOnly(TRUE)
      ->setRequired(TRUE)
      ->setDefaultValue('tsv');

    // The entries of the glossary.
    $fields['entries'] = BaseFieldDefinition::create('deepl_glossary_item')
      ->setLabel(t('Entries'))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setDescription(t('The entries of the glossary.'))
      ->setRequired(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'deepl_glossary_item',
      ]);

    // @phpstan-ignore-next-line
    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getGlossaryId(): ?string {
    return is_string($this->get('glossary_id')->value) ? $this->get('glossary_id')->value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getSourceLanguage(): string {
    return is_string($this->get('source_lang')->value) ? $this->get('source_lang')->value : '';
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetLanguage(): string {
    return is_string($this->get('target_lang')->value) ? $this->get('target_lang')->value : '';
  }

  /**
   * {@inheritdoc}
   */
  public function getEntryCount(): ?int {
    /** @var int $entry_count */
    $entry_count = is_int($this->get('entry_count')->value) ? $this->get('entry_count')->value : NULL;
    return $entry_count;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntries(): array {
    $entries = $this->get('entries');
    $entries_ar = [];
    foreach ($entries as $entry) {
      if (isset($entry->subject) && isset($entry->definition)) {
        assert(is_string($entry->subject));
        $entries_ar[$entry->subject] = $entry->definition;
      }
    }

    return $entries_ar;
  }

  /**
   * {@inheritdoc}
   */
  public function getTranslator(): ?TranslatorInterface {
    $tmgmt_translator_storage = $this->entityTypeManager()->getStorage('tmgmt_translator');
    $translator = $this->get('tmgmt_translator')->value;
    assert(is_string($translator));
    /** @var \Drupal\tmgmt\TranslatorInterface $translator */
    $translator = $tmgmt_translator_storage->load($translator);
    return $translator;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    // Set uid to current user.
    $this->set('uid', self::getDefaultEntityOwner());
  }

  /**
   * {@inheritdoc}
   */
  public static function getAllowedLanguages(): array {
    /* @phpstan-ignore-next-line */
    return \Drupal::service('tmgmt_deepl_glossary.helper')->getAllowedLanguages();
  }

}
