<?php

namespace Drupal\Tests\tmgmt_deepl\Unit;

use DeepL\DeepLException;
use DeepL\TextResult;
use DeepL\Translator as DeepLTranslator;
use DeepL\Usage;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_deepl\DeeplTranslatorApi;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the DeeplTranslatorAPI class.
 */
#[CoversClass(DeeplTranslatorApi::class)]
#[Group('tmgmt_deepl')]
class DeeplTranslatorApiTest extends UnitTestCase {

  /**
   * Tests the method ::getTranslator.
   */
  public function testGetTranslator(): void {
    $key_id = 'test_deepl_key_entity_id';
    $api_key_value = 'valid-deepl-api-key-from-mock';

    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with('auth_key_entity')
      ->willReturn($key_id);

    // Mock a key entity and related methods.
    $key_entity_mock = $this->createMock(KeyInterface::class);
    $key_entity_mock->expects($this->once())
      ->method('getKeyValue')
      ->willReturn($api_key_value);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock();

    // Mock key repository and related methods.
    $key_repository_mock = $this->createMock(KeyRepositoryInterface::class);
    $key_repository_mock->expects($this->once())
      ->method('getKey')
      ->with($key_id)
      ->willReturn($key_entity_mock);

    // Set key repository for class.
    $class->setKeyRepository($key_repository_mock);

    $class->translator = $translator;
    $actual = $class->getTranslator();

    // @phpstan-ignore-next-line.
    $this->assertInstanceOf(DeepLTranslator::class, $actual);
  }

  /**
   * Tests the method ::getTranslator.
   */
  public function testGetTranslatorException(): void {
    $translator = $this->createMock(TranslatorInterface::class);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->translator = $translator;

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Translator not set.');
    $class->getTranslator();
  }

  /**
   * Tests the method ::setTranslator.
   */
  public function testSetTranslator(): void {
    $translator = $this->createMock(TranslatorInterface::class);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setTranslator($translator);

    // @phpstan-ignore-next-line.
    $this->assertInstanceOf(TranslatorInterface::class, $class->translator);
  }

  /**
   * Tests the method ::translate.
   */
  public function testTranslate(): void {
    $text = ['Hello World'];
    $source_language = 'DE';
    $target_language = 'EN';
    $options = ['option1' => 'value1'];
    $translation_context = 'Sample context text.';

    // Mock DeepL\TextResult which is returned from translateText.
    $text_result = $this->createMock(TextResult::class);

    // Mock DeepL Translator class.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('translateText')
      ->with($text, $source_language, $target_language, array_merge($options, ['context' => $translation_context]))
      ->willReturn($text_result);

    // Mock only method getTranslator.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    // Mock tmgmt Translator class.
    $translator = $this->createMock(Translator::class);

    $result = $class->translate($translator, $text, $source_language, $target_language, $options, $translation_context);
    $this->assertInstanceOf(TextResult::class, $result);
  }

  /**
   * Tests the method ::translate.
   */
  public function testTranslateException(): void {
    $text = ['Hello World'];
    $source_language = 'DE';
    $target_language = 'EN';

    // Mock DeepL Translator class.
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('translateText')
      ->willThrowException(new DeepLException('Translate error'));

    // Mock only method getTranslator.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    // Mock Messenger Translator class.
    /** @var \Drupal\Core\Messenger\MessengerInterface&MockObject $messenger */
    $messenger = $this->createMock(MessengerInterface::class);
    $messenger->expects($this->once())
      ->method('addMessage')
      ->with('Translate error', 'error');

    $class->setMessenger($messenger);

    // Mock tmgmt Translator class.
    $translator = $this->createMock(Translator::class);

    $result = $class->translate($translator, $text, $source_language, $target_language);
    $this->assertNull($result);
  }

  /**
   * Tests the method ::getUsage.
   */
  public function testGetUsage(): void {
    $usage = $this->createMock(Usage::class);

    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('getUsage')
      ->willReturn($usage);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getUsage();
    $this->assertInstanceOf(Usage::class, $result);
  }

  /**
   * Tests the usage method with exception.
   *
   * Tests the method ::getUsage.
   */
  public function testGetUsageException(): void {
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->method('getUsage')
      ->willThrowException(new DeepLException('Usage error'));

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getUsage();
    $this->assertEquals('Usage error', $result);
  }

  /**
   * Tests the method ::getSourceLanguages.
   */
  public function testGetSourceLanguages(): void {
    // Mock German and English as source languages.
    $source_language_mock_de = new LanguageDefinitionMock('DE', 'German');
    $source_language_mock_en = new LanguageDefinitionMock('EN', 'English');

    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('getSourceLanguages')
      ->willReturn([$source_language_mock_de, $source_language_mock_en]);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getSourceLanguages();
    $expected = [
      'DE' => 'German',
      'EN' => 'English',
    ];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the method ::getSourceLanguages.
   */
  public function testGetSourceLanguagesException(): void {
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->method('getSourceLanguages')
      ->willThrowException(new \RuntimeException('Source languages error'));

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getSourceLanguages();
    $this->assertEmpty($result);
  }

  /**
   * Tests the method ::getTargetLanguages.
   */
  public function testGetTargetLanguages(): void {
    // Mock German and English as source languages.
    $target_language_mock_de = new LanguageDefinitionMock('DE', 'German');
    $target_language_mock_en = new LanguageDefinitionMock('EN', 'English');

    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->expects($this->once())
      ->method('getTargetLanguages')
      ->willReturn([$target_language_mock_de, $target_language_mock_en]);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getTargetLanguages();
    $expected = [
      'DE' => 'German',
      'EN' => 'English',
    ];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the method ::getTargetLanguages.
   */
  public function testGetTargetLanguagesException(): void {
    $deepl_translator = $this->createMock(DeepLTranslator::class);
    $deepl_translator->method('getTargetLanguages')
      ->willThrowException(new \RuntimeException('Target languages error'));

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock(['getTranslator']);
    $class->method('getTranslator')
      ->willReturn($deepl_translator);

    $result = $class->getTargetLanguages();
    $this->assertEmpty($result);
  }

  /**
   * Tests the method ::fixSourceLanguageMappings.
   */
  public function testFixSourceLanguageMappings(): void {
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test&MockObject $class */
    $class = $this->createClassMock();

    $result = $class->fixSourceLanguageMappings('EN-GB');
    $this->assertEquals('EN', $result);

    $result = $class->fixSourceLanguageMappings('PT-BR');
    $this->assertEquals('PT', $result);

    // Test with a language not in the mapping.
    $result = $class->fixSourceLanguageMappings('FR');
    $this->assertEquals('FR', $result);
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorApi_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplTranslatorApi_Test|MockObject {
    return $this->getMockBuilder(DeeplTranslatorApi_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart
/**
 * Mocked DeeplTranslatorApi class for tests.
 */
class DeeplTranslatorApi_Test extends DeeplTranslatorApi {

  /**
   * {@inheritdoc}
   */
  public TranslatorInterface $translator;

  /**
   * {@inheritdoc}
   */
  public MessengerInterface $messenger;

  /**
   * Set Messenger for use in tests.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   */
  public function setMessenger(MessengerInterface $messenger): void {
    $this->messenger = $messenger;
  }

  /**
   * Set Key repository for use in tests.
   *
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   */
  public function setKeyRepository(KeyRepositoryInterface $key_repository): void {
    $this->keyRepository = $key_repository;
  }

}
// @codingStandardsIgnoreEnd

/**
 * Simple mock class to represent a language definition.
 */
class LanguageDefinitionMock {

  /**
   * Define the language code.
   *
   * @var string
   */
  public string $code;

  /**
   * Define the language name.
   *
   * @var string
   */
  public string $name;

  public function __construct(string $code, string $name) {
    $this->code = $code;
    $this->name = $name;
  }

}
