<?php

namespace Drupal\Tests\tmgmt_deepl\Unit;

use DeepL\Usage;
use DeepL\UsageDetail;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt_deepl\DeeplTranslatorApiInterface;
use Drupal\tmgmt_deepl\DeeplTranslatorUi;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests the DeeplTranslatorUi class.
 */
#[CoversClass(DeeplTranslatorUi::class)]
#[Group('tmgmt_deepl')]
class DeeplTranslatorUiTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create/register string translation mock.
    $container = new ContainerBuilder();
    $container->set('string_translation', $this->getStringTranslationStub());
    \Drupal::setContainer($container);
  }

  /**
   * Tests the method ::buildConfigurationForm.
   */
  public function testBuildConfigurationForm(): void {
    // Creating a mock form state object.
    $form_state = $this->createMock(FormStateInterface::class);
    $form_object = $this->createMock(EntityFormInterface::class);
    $form_state->method('getFormObject')->willReturn($form_object);

    $translator = $this->createMock(Translator::class);
    $translator->method('isNew')->willReturn(FALSE);
    $form_object->method('getEntity')->willReturn($translator);

    // Creating a mock DeeplTranslatorApi.
    $deeplApi = $this->createMock(DeeplTranslatorApiInterface::class);
    $usage = $this->createMock(Usage::class);
    $deeplApi->expects($this->once())
      ->method('setTranslator')
      ->with($translator);
    $deeplApi->expects($this->once())->method('getUsage')->willReturn($usage);

    // Mocking the custom class to inject dependencies.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->deeplApi = $deeplApi;

    // Add messenger to class.
    $messenger = $this->createMock(MessengerInterface::class);

    $string_translation_sub = $this->getStringTranslationStub();
    $expected_message = new TranslatableMarkup('Usage information: :usage_count of :usage_limit characters', [
      ':usage_count' => '',
      ':usage_limit' => '',
    ], [], $string_translation_sub);
    $messenger->expects($this->once())
      ->method('addMessage')
      ->with($expected_message, 'status');

    $class->setMessenger($messenger);

    // Add module_handler to class.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $class->setModuleHandler($module_handler);

    // Execute the method.
    $form = [];

    $result = $class->buildConfigurationForm($form, $form_state);

    // Assertions to check the form structure.
    $this->assertArrayHasKey('auth_key_entity', $result);
    $this->assertArrayHasKey('enable_context', $result);

    // Model type.
    $this->assertArrayHasKey('model_type', $result);

    assert(is_array($result['model_type']));
    $this->assertEquals('select', $result['model_type']['#type']);
    $expected_model_type_options = [
      'latency_optimized' => 'latency_optimized (default)',
      'prefer_quality_optimized' => 'prefer_quality_optimized',
    ];
    $this->assertEquals($expected_model_type_options, $result['model_type']['#options']);

    // Split sentences.
    $this->assertArrayHasKey('split_sentences', $result);
    $expected_split_sentences_options = [
      '0' => 'No splitting at all, whole input is treated as one sentence',
      '1' => 'Splits on punctuation and on newlines (default)',
      'nonewlines' => 'Splits on punctuation only, ignoring newlines',
    ];

    assert(is_array($result['split_sentences']));
    $this->assertEquals($expected_split_sentences_options, $result['split_sentences']['#options']);

    // Formality.
    $this->assertArrayHasKey('formality', $result);
    $expected_formality_options = [
      'default' => 'default',
      'more' => 'more - for a more formal language',
      'less' => 'less - for a more informal language',
      'prefer_more' => 'prefer_more - for a more formal language if available, otherwise fallback to default formality',
      'prefer_less' => 'prefer_less - for a more informal language if available, otherwise fallback to default formality',
    ];

    assert(is_array($result['formality']));
    $this->assertEquals($expected_formality_options, $result['formality']['#options']);

    $this->assertArrayHasKey('preserve_formatting', $result);

    // Tag handling.
    $this->assertArrayHasKey('tag_handling', $result);
    $expected_tag_handling_options = [
      '0' => 'off',
      'xml' => 'xml',
      'html' => 'html',
    ];

    assert(is_array($result['tag_handling']));
    $this->assertEquals($expected_tag_handling_options, $result['tag_handling']['#options']);

    $this->assertArrayHasKey('outline_detection', $result);
    $this->assertArrayHasKey('splitting_tags', $result);
    $this->assertArrayHasKey('non_splitting_tags', $result);
    $this->assertArrayHasKey('ignore_tags', $result);
  }

  /**
   * Tests the method ::buildConfigurationForm.
   */
  public function testBuildConfigurationFormInvalidForm(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_object = $this->createMock(FormInterface::class);
    $form_state->method('getFormObject')->willReturn($form_object);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    // Execute the method.
    $form = [];
    $result = $class->buildConfigurationForm($form, $form_state);

    // New form fields should not be available.
    $this->assertArrayNotHasKey('split_sentences', $result);
    $this->assertArrayNotHasKey('formality', $result);
    $this->assertArrayNotHasKey('preserve_formatting', $result);
    $this->assertArrayNotHasKey('tag_handling', $result);
    $this->assertArrayNotHasKey('outline_detection', $result);
    $this->assertArrayNotHasKey('splitting_tags', $result);
    $this->assertArrayNotHasKey('non_splitting_tags', $result);
    $this->assertArrayNotHasKey('ignore_tags', $result);
  }

  /**
   * Tests the method ::validateConfigurationForm.
   */
  public function testValidateConfigurationFormInvalidForm(): void {
    // Creating a mock form state object.
    $form_state = $this->createMock(FormStateInterface::class);
    $form_object = $this->createMock(FormInterface::class);
    $form_state->method('getFormObject')->willReturn($form_object);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();

    // Add module_handler to class.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $class->setModuleHandler($module_handler);

    // Execute the method.
    $form = [];
    $class->validateConfigurationForm($form, $form_state);
    $this->assertFalse($class->isEntityForm($form_state));
    $this->assertEmpty($form);
  }

  /**
   * Data provider for testValidateConfigurationFormTagHandling.
   *
   * @return array
   *   Test scenarios.
   */
  public static function dataProviderTestValidateConfigurationFormTagHandling(): array {
    return [
      'default_tag_handling' => [
        'tag_handling' => '0',
        'actual_values' => [
          'outline_detection' => 1,
          'splitting_tags' => 'div,span',
          'non_splitting_tags' => 'b,i',
          'ignore_tags' => 'code,pre',
        ],
        'expected_values' => [
          'outline_detection' => 0,
          'splitting_tags' => '',
          'non_splitting_tags' => '',
          'ignore_tags' => '',
        ],
      ],
      'html_tag_handling' => [
        'tag_handling' => 'html',
        'actual_values' => [
          'outline_detection' => 1,
          'splitting_tags' => 'div,span',
          'non_splitting_tags' => 'b,i',
          'ignore_tags' => 'code,pre',
        ],
        'expected_values' => [
          'outline_detection' => 0,
          'splitting_tags' => '',
          'non_splitting_tags' => '',
          'ignore_tags' => '',
        ],
      ],
      'xml_tag_handling' => [
        'tag_handling' => 'xml',
        'actual_values' => [
          'outline_detection' => 1,
          'splitting_tags' => 'p,br',
          'non_splitting_tags' => 'a,strong,em',
          'ignore_tags' => 'script,style',
        ],
        'expected_values' => [
          'outline_detection' => 1,
          'splitting_tags' => 'p,br',
          'non_splitting_tags' => 'a,strong,em',
          'ignore_tags' => 'script,style',
        ],
      ],
    ];
  }

  /**
   * Tests the method ::validateConfigurationForm.
   *
   * @param string $tag_handling
   *   The selected tag_handling.
   * @param array $actual_values
   *   The actual values to be used in the form.
   * @param array $expected_values
   *   The expected values.
   */
  #[DataProvider('dataProviderTestValidateConfigurationFormTagHandling')]
  public function testValidateConfigurationFormTagHandling(string $tag_handling, array $actual_values, array $expected_values): void {
    // Creating a mock form state object.
    $form_state = $this->createMock(FormStateInterface::class);
    $form_object = $this->createMock(EntityFormInterface::class);
    $form_state->method('getFormObject')->willReturn($form_object);

    // Add example settings with actual values to simulate filled form fields.
    $settings = [
      'auth_key_entity' => 'test_auth_key',
      'tag_handling' => $tag_handling,
      'outline_detection' => $actual_values['outline_detection'],
      'splitting_tags' => $actual_values['splitting_tags'],
      'non_splitting_tags' => $actual_values['non_splitting_tags'],
      'ignore_tags' => $actual_values['ignore_tags'],
    ];
    $form_state->method('getValue')->with('settings')->willReturn($settings);

    // Use a callback to capture the values being set.
    $cleared_values = [];
    $form_state->method('setValueForElement')
      ->willReturnCallback(function ($element, $value) use (&$cleared_values) {
        assert(is_array($element));
        assert(is_array($element['#parents']));

        $parent_key = end($element['#parents']);
          /* @phpstan-ignore-next-line */
          $cleared_values[$parent_key] = $value;
          return NULL;
      });

    $translator = $this->createMock(Translator::class);
    $translator->method('isNew')->willReturn(FALSE);
    $form_object->method('getEntity')->willReturn($translator);

    // Create form structure with proper Drupal Form API elements.
    $form = [];
    /** @var array{
     *   'plugin_wrapper': array{
     *     'settings': array
     *   }
     * } $form */
    $form['plugin_wrapper']['settings']['outline_detection'] = [
      '#parents' => ['settings', 'outline_detection'],
    ];
    $form['plugin_wrapper']['settings']['splitting_tags'] = [
      '#parents' => ['settings', 'splitting_tags'],
    ];
    $form['plugin_wrapper']['settings']['non_splitting_tags'] = [
      '#parents' => ['settings', 'non_splitting_tags'],
    ];
    $form['plugin_wrapper']['settings']['ignore_tags'] = [
      '#parents' => ['settings', 'ignore_tags'],
    ];

    // Creating a mock DeeplTranslatorApi.
    $deeplApi = $this->createMock(DeeplTranslatorApiInterface::class);

    // Mocking the custom class to inject dependencies.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->deeplApi = $deeplApi;

    $messenger = $this->createMock(MessengerInterface::class);
    $class->setMessenger($messenger);

    // Run the validation.
    $class->validateConfigurationForm($form, $form_state);

    // Verify the values were cleared as expected.
    if ($tag_handling !== 'xml') {
      // For non-xml tag handling, values should be cleared.
      $this->assertCount(4, $cleared_values, 'Expected 4 values to be cleared');
      $this->assertEquals($expected_values['outline_detection'], $cleared_values['outline_detection'], 'outline_detection should be cleared to 0');
      $this->assertEquals($expected_values['splitting_tags'], $cleared_values['splitting_tags'], 'splitting_tags should be cleared to empty string');
      $this->assertEquals($expected_values['non_splitting_tags'], $cleared_values['non_splitting_tags'], 'non_splitting_tags should be cleared to empty string');
      $this->assertEquals($expected_values['ignore_tags'], $cleared_values['ignore_tags'], 'ignore_tags should be cleared to empty string');
    }
    else {
      // For xml tag handling, no values should be cleared.
      $this->assertCount(0, $cleared_values, 'No values should be cleared for XML tag handling');
    }
  }

  /**
   * Data provider for testCheckoutSettingsForm.
   *
   * @return array
   *   Test data for different scenarios.
   */
  public static function dataProviderTestCheckoutSettingsForm(): array {
    return [
      'Default behavior without altering' => [
        'alter_form' => FALSE,
        'enable_context' => FALSE,
        'expected_description' => 'The DeepL translator doesn\'t provide any checkout settings.',
        'expected_context_form' => FALSE,
      ],
      'External altering via tmgmt_deepl_checkout_settings_form' => [
        'alter_form' => TRUE,
        'enable_context' => FALSE,
        'expected_description' => '',
        'expected_context_form' => FALSE,
      ],
      'Translator setting enable_context is active' => [
        'alter_form' => FALSE,
        'enable_context' => TRUE,
        'expected_description' => '',
        'expected_context_form' => TRUE,
      ],
    ];
  }

  /**
   * Tests the method ::checkoutSettingsForm.
   */
  #[DataProvider('dataProviderTestCheckoutSettingsForm')]
  public function testCheckoutSettingsForm(bool $alter_form, bool $enable_context, string $expected_description, bool $expected_context_form): void {
    // Mocking the form state interface.
    $form_state = $this->createMock(FormStateInterface::class);

    // Mocking the JobInterface.
    $job = $this->createMock(JobInterface::class);

    // Mocking the Translator associated with the job.
    $translator = $this->createMock(Translator::class);
    $translator->method('label')->willReturn('DeepL');
    $translator->method('getSetting')
      ->with('enable_context')
      ->willReturn($enable_context);

    $job->method('getTranslator')->willReturn($translator);

    // Mocking the custom class to inject dependencies.
    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();

    // Mocking the module handler.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);

    // Set expectations for altering the form.
    if ($alter_form) {
      $module_handler->expects($this->once())
        ->method('alter')
        ->willReturnCallback(function ($hook, &$form, $job_arg) {
          // Simulate adding a real field to the form.
          $form = [];
          $form['custom_field'] = [
            '#type' => 'textfield',
            '#title' => 'Custom Field',
            '#description' => new TranslatableMarkup('This is a custom field added during form alteration.'),
          ];
        });
    }
    else {
      $module_handler->expects($this->once())
        ->method('alter')
        ->with('tmgmt_deepl_checkout_settings_form', $this->isArray(), $job);
    }

    $class->setModuleHandler($module_handler);

    // Define the initial form structure.
    $form = [];

    // Calling the method under test.
    $result = $class->checkoutSettingsForm($form, $form_state, $job);

    // Assertions for the default description.
    if ($expected_description) {
      $this->assertArrayHasKey('#description', $result);

      assert($result['#description'] instanceof TranslatableMarkup);
      $this->assertEquals($expected_description, $result['#description']->__toString());
    }
    else {
      $this->assertArrayNotHasKey('#description', $result);
    }

    // Assertions for the context form.
    if ($expected_context_form) {
      $this->assertArrayHasKey('context', $result);

      assert(is_array($result['context']));
      $this->assertEquals('textarea', $result['context']['#type']);

      assert($result['context']['#description'] instanceof TranslatableMarkup);
      $this->assertEquals(
        'Provide additional context for translation (e.g., product descriptions, article summaries).',
        $result['context']['#description']->__toString()
      );
    }
    else {
      $this->assertArrayNotHasKey('context', $result);
    }

    // Assertions for external altering.
    if ($alter_form) {
      $this->assertArrayHasKey('custom_field', $result);
      assert(is_array($result['custom_field']));
      $this->assertEquals('textfield', $result['custom_field']['#type']);
      $this->assertEquals('Custom Field', $result['custom_field']['#title']);
      $this->assertEquals(
        'This is a custom field added during form alteration.',
        $result['custom_field']['#description']
      );
    }
  }

  /**
   * Tests the method ::showUsageInfo.
   */
  public function testShowUsageInfoWithLimitsReached(): void {
    // Mocking the Usage object.
    $usage = $this->createMock(Usage::class);
    $usage->method('anyLimitReached')->willReturn(TRUE);

    // Creating a mock DeeplTranslatorApi.
    $deeplApi = $this->createMock(DeeplTranslatorApiInterface::class);
    $deeplApi->expects($this->once())->method('getUsage')->willReturn($usage);

    // Add messenger to class.
    $messenger = $this->createMock(MessengerInterface::class);
    $messenger->expects($this->once())
      ->method('addWarning')
      ->with('The translation limit of your account has been reached.');

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setMessenger($messenger);

    $class->showUsageInfo($deeplApi);
  }

  /**
   * Tests the method ::showUsageInfo.
   */
  public function testShowUsageInfoWithUsageInfo(): void {
    // Mocking the Usage object.
    $usage = $this->createMock(Usage::class);
    $usage->method('anyLimitReached')->willReturn(FALSE);
    $usage->character = new UsageDetail(1000, 10000);

    // Creating a mock DeeplTranslatorApi.
    $deeplApi = $this->createMock(DeeplTranslatorApiInterface::class);
    $deeplApi->expects($this->once())->method('getUsage')->willReturn($usage);

    // Add messenger to class.
    $messenger = $this->createMock(MessengerInterface::class);
    $string_translation_sub = $this->getStringTranslationStub();
    $expected_message = new TranslatableMarkup('Usage information: :usage_count of :usage_limit characters', [
      ':usage_count' => '1000',
      ':usage_limit' => '10000',
    ], [], $string_translation_sub);
    $messenger->expects($this->once())
      ->method('addMessage')
      ->with($expected_message);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    $class->setMessenger($messenger);

    $class->showUsageInfo($deeplApi);
  }

  /**
   * Tests the method ::isEntityForm.
   */
  public function testIsEntityFormValid(): void {
    // Creating a mock form state object.
    $form_state = $this->createMock(FormStateInterface::class);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    // Use EntityFormInterface.
    $form_object_entity_form_interface = $this->createMock(EntityFormInterface::class);
    $form_state->method('getFormObject')->willReturn($form_object_entity_form_interface);
    $this->assertTrue($class->isEntityForm($form_state));
  }

  /**
   * Tests the method ::isEntityForm.
   */
  public function testIsEntityFormInvalid(): void {
    // Creating a mock form state object.
    $form_state = $this->createMock(FormStateInterface::class);

    /** @var \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test&MockObject $class */
    $class = $this->createClassMock();
    // Use default FormInterface instead of EntityFormInterface.
    $form_object_form_interface = $this->createMock(FormInterface::class);
    $form_state->method('getFormObject')->willReturn($form_object_form_interface);
    $this->assertFalse($class->isEntityForm($form_state));
  }

  /**
   * Creates and returns a test class mock.
   *
   * @param list<non-empty-string> $only_methods
   *   An array of names for methods to be configurable.
   *
   * @return \Drupal\Tests\tmgmt_deepl\Unit\DeeplTranslatorUi_Test|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked class.
   */
  protected function createClassMock(array $only_methods = []): DeeplTranslatorUi_Test|MockObject {
    return $this->getMockBuilder(DeeplTranslatorUi_Test::class)
      ->disableOriginalConstructor()
      ->onlyMethods($only_methods)
      ->getMock();
  }

}

// @codingStandardsIgnoreStart

/**
 * Mocked DeeplTranslatorUi class for tests.
 */
class DeeplTranslatorUi_Test extends DeeplTranslatorUi {

  /**
   * {@inheritdoc}
   */
  public DeeplTranslatorApiInterface $deeplApi;

  /**
   * {@inheritdoc}
   */
  public $messenger;

  /**
   * Set Messenger for use in tests.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   */
  public function setMessenger(MessengerInterface $messenger): void {
    $this->messenger = $messenger;
  }

  /**
   * Set ModuleHandler for use in tests.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   */
  public function setModuleHandler(ModuleHandlerInterface $module_handler): void {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public function showUsageInfo(DeeplTranslatorApiInterface $deepl_api): void {
    parent::showUsageInfo($deepl_api);
  }

  /**
   * {@inheritdoc}
   */
  public function isEntityForm(FormStateInterface $form_state): bool {
    return parent::isEntityForm($form_state);
  }

}
// @codingStandardsIgnoreEnd
