<?php

/**
 * @file
 * Post update functions for tmgmt_deepl module.
 */

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Migrate DeepL translator API keys from direct config to Key module reference.
 *
 * @param array $sandbox
 *   Stores information for batch processing (not used in this simple case).
 *
 * @return \Drupal\Core\StringTranslation\TranslatableMarkup|string
 *   Status message for Drush console output.
 */
function tmgmt_deepl_post_update_migrate_api_key_to_key_module(array &$sandbox = []): TranslatableMarkup|string {
  $logger = \Drupal::logger('tmgmt_deepl_update');
  $config_factory = \Drupal::configFactory();

  // Get translator ids by running simple entity query.
  $translator_ids = \Drupal::entityQuery('tmgmt_translator')
    ->condition('plugin', 'deepl_api')
    ->accessCheck(FALSE)
    ->execute();

  $translator_ids = is_array($translator_ids) ? $translator_ids : [];

  if (count($translator_ids) === 0) {
    return t('No DeepL translators (plugin: deepl_api) found to check for API key migration.');
  }

  $translators_requiring_action = [];
  $checked_count = 0;

  foreach ($translator_ids as $translator_id) {
    // Load config entity for translator.
    assert(is_string($translator_id));
    $config = $config_factory->getEditable("tmgmt.translator.$translator_id");

    // Skip if config is missing for some reason.
    if ($config->isNew()) {
      continue;
    }

    $checked_count++;

    // Get raw data from config.
    $settings = $config->get('settings') ?: [];
    $label = $config->get('label');
    $save_needed = FALSE;

    // Case A: Old key exists, New key is missing.
    // Action: Warn user, log, and remove old key for security.
    if (is_array($settings) && (!empty($settings['auth_key']) && empty($settings['auth_key_entity']))) {
      $logger->warning('DeepL translator "%label" (%id) still uses the deprecated direct API key configuration. Please create a key entity (type: DeepL API Key or Authentication) with the API key and re-save the translator settings, selecting the new Key entity. The old key setting has been removed for security.', [
        '%label' => $label,
        '%id' => $translator_id,
      ]);

      // Check for valid types of label, translator_id.
      assert(is_string($label));
      $translators_requiring_action[] = sprintf('%s (%s)', $label, $translator_id);

      // Remove the old setting.
      $settings['auth_key'] = NULL;
      $save_needed = TRUE;
    }
    // Case B: Old key is set but empty/legacy cleanup.
    elseif (is_array($settings) && (array_key_exists('auth_key', $settings) && empty($settings['auth_key']) && $settings['auth_key'] !== NULL)) {
      $settings['auth_key'] = NULL;
      $save_needed = TRUE;
      $logger->info('Removed empty legacy auth_key setting from DeepL translator "%label" (%id).', [
        '%label' => $label,
        '%id' => $translator_id,
      ]);
    }
    // Case C: Both exist (New key is set, so safely remove old one).
    elseif (is_array($settings) && (isset($settings['auth_key']) && !empty($settings['auth_key_entity']))) {
      // If the old key is not already NULL, clean it up.
      if ($settings['auth_key'] != NULL) {
        $settings['auth_key'] = NULL;
        $save_needed = TRUE;
        $logger->info('Removed legacy auth_key setting from DeepL translator "%label" (%id) as new auth_key_entity is set.', [
          '%label' => $label,
          '%id' => $translator_id,
        ]);
      }
    }

    // Save the config entity afterwards, if needed..
    if ($save_needed) {
      $config->set('settings', $settings);
      $config->save();
    }
  }

  // Construct the final message for Drush based on findings.
  if (!empty($translators_requiring_action)) {
    $count = count($translators_requiring_action);
    $drush_message = t('%count DeepL translator(s) require manual API key migration to the Key module:', ['%count' => $count]);
    $drush_message .= "\n - " . implode("\n - ", $translators_requiring_action);
    $drush_message .= "\n" . t('ACTION REQUIRED: For each translator listed above, please create a Key entity (type: DeepL API Key) containing its API key. Then, edit the translator configuration in the TMGMT settings UI and select the corresponding Key entity.');
    $drush_message .= "\n" . t('The old, insecurely stored keys have been removed from the configuration. See Drupal logs (channel: tmgmt_deepl_update) for details.');
    return $drush_message;
  }
  else {
    // If the loop ran but no action was needed.
    return t('Checked @count DeepL translator(s). No configurations required manual API key migration to the Key module.', ['@count' => $checked_count]);
  }
}
