<?php

namespace Drupal\tmgmt_google_v3\Plugin\tmgmt\Translator;

use Drupal\Component\EventDispatcher\ContainerAwareEventDispatcher;
use Drupal\Component\Utility\Html;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Google\ApiCore\ApiException;
use Drupal\tmgmt_google_v3\Event\PostTranslationEvent;
use Google\Cloud\Translate\V3\TranslateTextGlossaryConfig;
use Google\Rpc\Code;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Google\Cloud\Translate\V3\TranslationServiceClient;
use Drupal\tmgmt\Entity\Job;

/**
 * Google V3 translator plugin.
 *
 * @TranslatorPlugin(
 *   id = "google_v3",
 *   label = @Translation("Google V3"),
 *   description = @Translation("Google V3 Translator service."),
 *   ui = "Drupal\tmgmt_google_v3\GoogleV3TranslatorUi",
 *   logo = "icons/google.svg",
 * )
 */
class GoogleV3Translator extends TranslatorPluginBase implements ContainerFactoryPluginInterface, ContinuousTranslatorInterface {

  /**
   * Data help service.
   *
   * @var \Drupal\tmgmt\Data
   */
  protected $dataHelper;

  /**
   * Google\Cloud\Translate\V3\TranslationServiceClient definition.
   *
   * @var \Google\Cloud\Translate\V3\TranslationServiceClient
   */
  protected $translationServiceClient;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\tmgmt\Data $data_helper
   *   Data helper service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   This is the service that allows us to load and save entities.
   * @param \Drupal\Component\EventDispatcher\ContainerAwareEventDispatcher $event_dispatcher
   *   Drupal event dispatcher.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    Data $data_helper,
    EntityTypeManagerInterface $entity_type_manager,
    protected ContainerAwareEventDispatcher $event_dispatcher,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->dataHelper = $data_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('tmgmt.data'),
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * Initializes the Google translation service client for plugin.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator used to retrieve the 'google_credentials' settings.
   */
  protected function initializeClient(TranslatorInterface $translator) {
    $credentials = $translator->getSetting('google_credentials');
    if (isset($credentials['fids'])) {
      $fid = $credentials['fids'][0];
    }
    if (!isset($fid)) {
      $fid = $credentials[0] ?? NULL;
    }
    if ($fid) {
      $credentialsFile = $this->entityTypeManager->getStorage('file')->load($fid);
    }
    if (!isset($credentialsFile)) {
      return FALSE;
    }
    $absolute_path = \Drupal::service('file_system')->realpath($credentialsFile->getFileUri());
    $options = [
      'credentials' => $absolute_path,
    ];
    $this->translationServiceClient = new TranslationServiceClient($options);
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator) {
    if ($translator->getSetting('location') && $translator->getSetting('api_project') && $translator->getSetting('google_credentials')) {
      return AvailableResult::yes();
    }
    return AvailableResult::no($this->t('@translator is not available. Make sure it is properly <a href=:configured>configured</a>.', [
      '@translator' => $translator->label(),
      ':configured' => $translator->toUrl()->toString(),
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job) {
    $this->requestJobItemsTranslation($job->getItems());
    if (!$job->isRejected()) {
      $job->submitted('The translation job has been submitted.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedRemoteLanguages(TranslatorInterface $translator) {
    $remoteLanguages = [];
    if ($this->initializeClient($translator)) {
      $parent = $translator->getSetting('api_project');
      $location = $translator->getSetting('location');
      try {
        $result = $this->translationServiceClient->getSupportedLanguages('projects/' . $parent . '/locations/' . $location);
        /** @var \Google\Cloud\Translate\V3\SupportedLanguages $languages*/
        $languages = $result->getLanguages();
        if (!empty($languages)) {
          $totalLanguages = $languages->count();
          for ($i = 0; $i < $totalLanguages; $i++) {
            /** @var \Google\Cloud\Translate\V3\SupportedLanguage $language*/
            $language = $languages->offsetGet($i);
            $remoteLanguages[$language->getLanguageCode()] = $language->getLanguageCode();
          }
        }
      }
      catch (\Exception $e) {
        \Drupal::messenger()->addMessage($e->getMessage(), 'error');
        return [];
      }
    }
    return $remoteLanguages;
  }

  /**
   * {@inheritdoc}
   */
  public function hasCheckoutSettings(JobInterface $job) {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function requestJobItemsTranslation(array $job_items) {
    /** @var \Drupal\tmgmt\Entity\Job $job */
    $job = reset($job_items)->getJob();
    foreach ($job_items as $job_item) {
      if ($job->isContinuous()) {
        $job_item->active();
      }
      // Pull the source data array through the job and flatten it.
      $data = $this->dataHelper->filterTranslatable($job_item->getData());
      $translation = [];
      try {
        foreach ($data as $key => $value) {
          $result = $this->googleTranslateRequestTranslation($job, $value['#text']);
          if (!empty($result)) {
            // Decode entities before dispatching event to allow subscribers to
            // deal with any edge cases.
            $post_translation_event = new PostTranslationEvent($key, Html::decodeEntities($result));
            $this->event_dispatcher->dispatch($post_translation_event, PostTranslationEvent::POST_TRANSLATION);
            $translation[$key]['#text'] = $post_translation_event->getTranslatedText();
          }
        }
        // Save the translated data through the job.
        // Only reached when all translation requests have succeeded.
        $job_item->addTranslatedData($this->dataHelper
          ->unflatten($translation));
      }
      catch (ApiException $e) {
        // If we reached a quota and this is a continuous job set it back to
        // inactive, so it can be tried again later.
        if ($e->getCode() === Code::RESOURCE_EXHAUSTED && $job->isContinuous()) {
          $job_item->setState(JobItemInterface::STATE_INACTIVE, 'Translation quota exhausted. Job item has been marked inactive.');
        }
        else {
          $job->rejected('Translation has been rejected with following error: @error',
            ['@error' => $e->getMessage()]);
        }
      }
      catch (\Throwable $e) {
        $job->rejected('Translation has been rejected with following error: @error',
          ['@error' => $e->getMessage()], 'error');
      }
    }
  }

  /**
   * Helper method to do translation request.
   *
   * @param \Drupal\tmgmt\Entity\Job $job
   *   TMGMT Job to be used for translation.
   * @param array|string $text
   *   Text/texts to be translated.
   *
   * @return string
   *   The translated text.
   */
  protected function googleTranslateRequestTranslation(Job $job, $text) {
    $translator = $job->getTranslator();
    $this->initializeClient($translator);
    $translation_context = [
      'language_from' => $job->getRemoteSourceLanguage(),
      'language_to_local' => $job->getTargetLangcode(),
      'language_to_remote' => $job->getRemoteTargetLanguage(),
      'project_id' => $translator->getSetting('api_project'),
      'location' => $translator->getSetting('location'),
    ];
    $glossary_mappings = $translator->getSetting('glossary_mappings');
    $translation = $this->executeTranslation($translation_context, $text, $glossary_mappings);
    return $translation;
  }

  /**
   * Translates text using the Google Cloud Translation API.
   *
   * @param array $translation_context
   *   An associative array containing context for the translation. Keys:
   *   - language_from_remote: string
   *     The remote source language code (e.g. "en").
   *   - language_to_remote: string
   *     The remote target language code supported by Google (e.g. "zh-TW").
   *   - language_to_local: string
   *     The local target language code which maps to the Drupal language code
   *     the text is being translated into. This may differ from
   *     `language_to_remote` as some languages e.g. zh-hant-HK don't have an
   *     equivalent model in Google. Therefore, if both zh-hant-hk and
   *     zh-hank-tw use the zh-TW translation model, each can supply its own
   *     unique glossary using this key.
   *   - project_id: string
   *     The Google Cloud project ID.
   *   - location: string
   *     The Google Cloud API location (e.g. "global").
   * @param string $text
   *   The text to be translated.
   * @param array|null $glossary_mappings
   *   Optional glossary mappings keyed by remote target language code.
   *
   * @return string
   *   The translated text.
   */
  protected function executeTranslation(array $translation_context, $text, ?array $glossary_mappings = NULL) {
    if (mb_strlen($text) > 20000) {
      return $this->executeTranslation($translation_context, mb_substr($text, 0, 20000), $glossary_mappings) . $this->executeTranslation($translation_context, mb_substr($text, 20000), $glossary_mappings);
    }
    else {
      try {
        $contents = [$text];
        $formatted_parent = $this->translationServiceClient->locationName($translation_context['project_id'], $translation_context['location']);
        $options = [
          'sourceLanguageCode' => $translation_context['language_from'],
        ];
        if ($glossary_mappings && !empty($glossary_mappings[$translation_context['language_to_local']])) {
          $glossaryConfig = new TranslateTextGlossaryConfig(
            [
              'glossary' => 'projects/' . $translation_context['project_id'] . '/locations/' . $translation_context['location'] . '/glossaries/' . $glossary_mappings[$translation_context['language_to_local']],
              'ignore_case' => TRUE,
            ],
          );
          $options += [
            'glossaryConfig' => $glossaryConfig,
          ];
        }
        $response = $this->translationServiceClient->translateText($contents, $translation_context['language_to_remote'], $formatted_parent, $options);
        // If a glossary definition exists use that, otherwise
        // fallback to default translation.
        if ($response->getGlossaryTranslations()->offsetExists(0)) {
          $translation = $response->getGlossaryTranslations()->offsetGet(0)->getTranslatedText();
        }
        else {
          $translation = $response->getTranslations()->offsetGet(0)->getTranslatedText();
        }
      }
      finally {
        $this->translationServiceClient->close();
      }
    }
    return $translation ?? $text;
  }

}
