<?php

namespace Drupal\tmgmt_ignore_fields\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfo;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Configure fields to ignore for this module.
 */
class IgnoreFieldsSettingsForm extends ConfigFormBase {

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfo
   */
  protected $entityTypeBundleInfo;

  /**
   * The path resolver service.
   *
   * @var \Drupal\Core\Extension\ExtensionPathResolver
   */
  protected $extensionPathResolver;

  /**
   * Constructs an IgnoreFieldsSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $entity_type_bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extension_path_resolver
   *   The extension path resolver service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, EntityFieldManagerInterface $entity_field_manager, EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfo $entity_type_bundle_info, ExtensionPathResolver $extension_path_resolver) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->extensionPathResolver = $extension_path_resolver;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('extension.path.resolver')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['tmgmt_ignore_fields.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'tmgmt_ignore_fields_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('tmgmt_ignore_fields.settings');
    $ignored_fields = $config->get('ignore_fields') ?? [];

    $base_fields = [];
    $custom_fields = [];
    $entity_types = $this->entityTypeManager->getDefinitions();
    foreach ($entity_types as $entity_type_id => $entity_type) {
      if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
        // Retrieve base field definitions.
        $base_field_definitions = $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id);
        foreach ($base_field_definitions as $field_name => $field_definition) {
          $field_key = $field_name;
          $base_fields[$field_key] = [
            'label' => $entity_type->getLabel() . ': ' . $field_definition->getLabel() . ' (' . $field_name . ')',
            'entity_type' => $entity_type_id,
          ];
        }

        // Retrieve custom field definitions.
        $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
        foreach ($bundles as $bundle => $bundle_info) {
          $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
          foreach ($field_definitions as $field_name => $field_definition) {
            if (!isset($base_fields[$field_name])) {
              $field_key = $field_name;
              $custom_fields[$field_key] = [
                'label' => $entity_type->getLabel() . ': ' . $field_definition->getLabel() . ' (' . $field_name . ')',
                'entity_type' => $entity_type_id,
              ];
            }
          }
        }
      }
    }

    // Sort base fields alphabetically by label.
    uasort($base_fields, function ($a, $b) {
      return strcasecmp($a['label'], $b['label']);
    });
    uasort($custom_fields, function ($a, $b) {
      return strcasecmp($a['label'], $b['label']);
    });

    $form['fields'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Base Fields to ignore'),
        $this->t('Custom Fields to ignore'),
      ],
    ];

    // Add the fields to the table.
    $max_rows = max(count($base_fields), count($custom_fields));
    $base_field_keys = array_keys($base_fields);
    $custom_field_keys = array_keys($custom_fields);

    for ($i = 0; $i < $max_rows; $i++) {
      $base_field_key = $base_field_keys[$i] ?? NULL;
      $custom_field_key = $custom_field_keys[$i] ?? NULL;

      $form['fields'][$i] = [];

      // Base field column.
      if ($base_field_key) {
        $form['fields'][$i]['base_field'] = [
          '#type' => 'checkbox',
          '#title' => $base_fields[$base_field_key]['label'],
          '#default_value' => in_array($base_field_key, $ignored_fields, TRUE),
          '#field_name' => $base_field_key,
        ];
      }
      else {
        $form['fields'][$i]['base_field'] = [
          '#type' => 'markup',
          '#markup' => '',
        ];
      }

      // Custom field column.
      if ($custom_field_key) {
        $form['fields'][$i]['custom_field'] = [
          '#type' => 'checkbox',
          '#title' => $custom_fields[$custom_field_key]['label'],
          '#default_value' => in_array($custom_field_key, $ignored_fields, TRUE),
          '#field_name' => $custom_field_key,
        ];
      }
      else {
        $form['fields'][$i]['custom_field'] = [
          '#type' => 'markup',
          '#markup' => '',
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $ignore_fields = [];
    $fields_value = $form_state->getValue('fields');

    // Collect checked fields by directly using row index and field values.
    foreach ($fields_value as $row_index => $row) {
      if (isset($row['base_field']) && $row['base_field']) {
        $ignore_fields[] = $form['fields'][$row_index]['base_field']['#field_name'];
      }
      if (isset($row['custom_field']) && $row['custom_field']) {
        $ignore_fields[] = $form['fields'][$row_index]['custom_field']['#field_name'];
      }
    }

    // Remove duplicates and empty values.
    $ignore_fields = array_filter(array_unique($ignore_fields));

    // Save to configuration.
    $this->configFactory->getEditable('tmgmt_ignore_fields.settings')
      ->set('ignore_fields', $ignore_fields)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
