# Quota Error Handling Implementation

## Overview

Special error handling has been implemented for Lara Translate API quota exceeded errors. When quota is exceeded, job items remain in "Active/In progress" state and are automatically retried rather than being moved to "Needs review" status.

## Problem Statement

### Issue 1: Quota Errors Marked Items as "Needs Review"

When the Lara Translate API returns quota exceeded errors:
```
You have exceeded your "api_translation_chars" quota
```

The original behavior would:
1. Log warnings for each failed segment
2. Mark segments with `TMGMT_DATA_ITEM_STATE_PENDING` status
3. Move job item to "Needs review" state
4. Require manual intervention

This was problematic because quota limits are temporary and reset over time.

### Issue 2: Continuous Translation Blocked Content Save

When using continuous translation (content automatically translated on create/update), the `requestJobItemsTranslation()` method was processing items synchronously. This caused:

1. **EntityStorageException** thrown during content save:
   ```
   Drupal\Core\Entity\EntityStorageException: You have exceeded your "api_translation_chars" quota
   in Drupal\Core\Entity\Sql\SqlContentEntityStorage->save()
   ```

2. **Content save operation failed** - users couldn't save their content
3. **Poor user experience** - editors blocked by translation quota errors
4. **Stack trace showing synchronous processing**:
   ```
   tmgmt_content_entity_update()
   → tmgmt_content_create_continuous_job_items()
   → ContinuousManager->addItem()
   → LaraTranslator->requestJobItemsTranslation()
   → LaraTranslator->processJobItem() [SYNCHRONOUS!]
   → translateData() → quota exception → blocks save!
   ```

This completely defeated the purpose of the queue-based system for continuous translation scenarios.

## Solution

Two complementary solutions were implemented:

1. **Quota errors treated as transient** - trigger automatic retries instead of permanent failure
2. **Continuous translation uses queue** - `requestJobItemsTranslation()` now queues items instead of processing synchronously

## Implementation Details

### 1. QueueWorker Changes

**File**: `src/Plugin/QueueWorker/LaraTranslatorWorker.php`

**Modified**: `isTransientError()` method

**Added quota detection**:
```php
// Quota exceeded errors - should be retried later when quota is available.
if (str_contains($message, 'exceeded your') && str_contains($message, 'quota')) {
  return TRUE;
}
```

**Effect**: When quota errors are detected, the QueueWorker:
- Logs a warning: "Transient error processing job item X: ... Will retry."
- Throws `SuspendQueueException` to re-queue the item
- Job item stays in "Active" state

### 2. Translator Plugin Changes

**File**: `src/Plugin/tmgmt/Translator/LaraTranslator.php`

**Modified**: `translateData()` method

**Added quota error re-throwing**:
```php
catch (LaraException $e) {
  // Check if this is a quota exceeded error - these should be retried.
  $error_message = $e->getMessage();
  if (str_contains($error_message, 'exceeded your') && str_contains($error_message, 'quota')) {
    $this->logger->warning('Failed to translate segment "@text": @error', [
      '@text' => \substr($item['#text'], 0, 100),
      '@error' => $error_message,
    ]);
    // Re-throw to trigger job item retry via queue.
    throw $e;
  }

  // Other errors: log and mark as needs review.
  // ...
}
```

**Effect**: When a quota error occurs during segment translation:
1. Warning is logged for the specific segment
2. Exception is re-thrown (not caught)
3. Entire job item processing stops
4. Exception bubbles up to QueueWorker
5. QueueWorker detects it's transient and re-queues
6. Job item remains "Active" for retry

### 3. Fixed Continuous Translation Blocking

**File**: `src/Plugin/tmgmt/Translator/LaraTranslator.php`

**Modified**: `requestJobItemsTranslation()` method

**Changed from synchronous to queue-based**:

**Before**:
```php
public function requestJobItemsTranslation(array $job_items): void {
  foreach ($job_items as $job_item) {
    try {
      $this->processJobItem($job_item);  // SYNCHRONOUS - blocks!
    }
    catch (TMGMTException $e) {
      // Error during save operation
    }
  }
}
```

**After**:
```php
public function requestJobItemsTranslation(array $job_items): void {
  $queue = $this->queueFactory->get('tmgmt_laratranslate_worker');

  foreach ($job_items as $job_item) {
    $job_item->active();

    $queue->createItem([
      'job_item_id' => $job_item->id(),
      'submitted_time' => time(),
    ]);
  }

  // Returns immediately - doesn't block!
}
```

**Effect**: Continuous translation now:
1. Queues items immediately
2. Returns without blocking
3. Content save operation completes successfully
4. Translation happens in background via cron
5. No quota errors block content editing## Behavior Comparison

### Before (Non-quota errors)

```
Segment translation fails (e.g., invalid data)
→ Log warning
→ Mark segment as PENDING
→ Continue processing other segments
→ Job item moves to "Needs review"
→ Requires manual review/intervention
```

### After (Quota errors)

```
Segment translation fails with quota error
→ Log warning for segment
→ Re-throw exception
→ Stop processing current job item
→ QueueWorker catches exception
→ Detected as transient error
→ Re-queue job item
→ Job item stays "Active"
→ Automatic retry on next cron run
```

### After (Non-quota errors - unchanged)

```
Segment translation fails (other error)
→ Log warning
→ Mark segment as PENDING
→ Continue processing other segments
→ Job item completes with some segments pending
→ Job item moves to "Needs review"
→ Requires manual review/intervention
```

## User Experience

### When Quota is Exceeded - Manual Translation

**What the user sees in TMGMT UI**:
1. Job item appears as "Active/In progress"
2. **Message on job item**: "Translation quota exceeded. Job item will be automatically retried when quota is available."
3. Queue shows pending items (visible via Drush)

**What happens automatically**:
1. Items remain queued
2. Retried on each cron run
3. Once quota available, processing succeeds
4. Job items move to "Translated" state

**No manual intervention needed** - the system handles it automatically.

### When Quota is Exceeded - Continuous Translation

**What the user experiences**:

**Before the fix**:
```
1. User edits content and clicks "Save"
2. ❌ EntityStorageException thrown
3. ❌ Content save fails
4. ❌ Error message displayed
5. ❌ User frustrated - can't save their work!
```

**After the fix**:
```
1. User edits content and clicks "Save"
2. ✅ Content saves successfully
3. ✅ Translation queued in background
4. ✅ User can continue working
5. ✅ Translation happens later via cron
6. ✅ Quota errors don't affect editing workflow
```

**Key benefit**: Content editors are never blocked by translation quota limits!

## Configuration

Quota error retry behavior respects the queue settings:

- **max_retries**: Default 3 attempts
- **timeout**: Processing timeout per cron run
- **items_per_cron**: Number of items processed per run

If max retries are reached, the item will eventually fail permanently.

## Monitoring

### Check for quota errors in logs
```bash
ddev drush watchdog:show --type=tmgmt_laratranslate | grep quota
```

### Expected log output
```
[warning] Failed to translate segment "Example": You have exceeded your "api_translation_chars" quota
[warning] Transient error processing job item 123: You have exceeded your "api_translation_chars" quota. Will retry.
```

### Check queue status
```bash
ddev drush queue:list
```

Items will remain in queue until quota is available or max retries reached.

## Benefits

1. **Automatic recovery**: No manual intervention when quota resets
2. **Better UX**: Users don't need to re-submit jobs
3. **Continuous translation**: Works seamlessly with continuous translation
4. **Graceful degradation**: System handles quota limits elegantly
5. **Clear feedback**: Logs explain what's happening

## Edge Cases

### Quota never increases
If quota never becomes available:
- Items will retry up to `max_retries` times
- After max retries, item will fail permanently
- Admin can increase quota with Lara Translate provider
- Admin can manually re-submit jobs after quota increase

### Mixed errors
If some segments fail due to quota and others due to different errors:
- Quota error triggers immediate re-throw
- Other segments won't be attempted in that run
- On retry, will attempt all segments again
- Once quota available, will process all segments

### Partial processing
If quota runs out mid-job:
- Current segment fails with quota error
- Job item re-queued
- On retry, starts from beginning (TMGMT handles this)
- Already-translated segments may be re-translated (API limitation)

## Testing

### Test 1: Manual Translation with Quota Exceeded
1. Use a Lara Translate account with low quota
2. Submit large translation job via TMGMT UI
3. Monitor logs for quota errors
4. Verify items stay "Active"
5. Wait for quota reset or increase quota
6. Verify automatic processing resumes

### Test 2: Continuous Translation with Quota Exceeded (Critical!)
1. Enable continuous translation for a content type
2. Use account with low/exceeded quota
3. **Edit and save content** - this was the failing scenario
4. ✅ Verify content saves successfully (no EntityStorageException)
5. ✅ Verify translation is queued
6. Check logs show "Job item X queued for translation"
7. Process queue when quota available
8. Verify translation completes

### Verify error detection
```bash
# Check transient error detection is working
ddev drush watchdog:show --type=tmgmt_laratranslate | grep "Will retry"

# Check items are being re-queued
ddev drush queue:list | grep tmgmt_laratranslate_worker

# Verify continuous translation queuing (new!)
ddev drush watchdog:show --type=tmgmt_laratranslate | grep "queued for translation"
```

## Future Enhancements

Potential improvements:
- Exponential backoff for quota errors
- Quota prediction to pause processing before hitting limit
- Real-time quota monitoring
- Admin notifications when quota limits are reached
- Configurable quota error handling strategy

## Related Files

- `src/Plugin/QueueWorker/LaraTranslatorWorker.php` - Queue worker with transient error detection
- `src/Plugin/tmgmt/Translator/LaraTranslator.php` - Translator with quota error re-throwing
- `doc/QUEUE_IMPLEMENTATION_PHASE1_COMPLETE.md` - Phase 1 documentation with quota section

## Conclusion

Quota error handling makes the translation system more resilient and user-friendly by automatically handling temporary quota limitations without requiring manual intervention.
