<?php

declare(strict_types=1);

namespace Drupal\tmgmt_laratranslate\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Queue\SuspendQueueException;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt_laratranslate\Plugin\tmgmt\Translator\LaraTranslator;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Lara Translate translation queue items.
 *
 * @QueueWorker(
 *   id = "tmgmt_laratranslate_worker",
 *   title = @Translation("Lara Translate Worker"),
 *   cron = {"time" = 60}
 * )
 */
final class LaraTranslatorWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new LaraTranslatorWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger channel.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LoggerInterface $logger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $entity_type_manager = $container->get('entity_type.manager');
    \assert($entity_type_manager instanceof EntityTypeManagerInterface);

    $logger = $container->get('logger.channel.tmgmt_laratranslate');
    \assert($logger instanceof LoggerInterface);

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $entity_type_manager,
      $logger,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    // Validate queue item data.
    if (!isset($data['job_item_id'])) {
      $this->logger->error('Queue item missing job_item_id');
      return;
    }

    $job_item_id = $data['job_item_id'];

    try {
      // Load the job item.
      $storage = $this->entityTypeManager->getStorage('tmgmt_job_item');
      $job_item = $storage->load($job_item_id);

      if (!$job_item instanceof JobItemInterface) {
        $this->logger->warning('Job item @id not found or invalid', ['@id' => $job_item_id]);
        return;
      }

      // Check if job item is still active and needs processing.
      if (!$job_item->isActive()) {
        $this->logger->info('Job item @id is no longer active, skipping', ['@id' => $job_item_id]);
        return;
      }

      // Get the translator plugin.
      $job = $job_item->getJob();
      $translator = $job->getTranslator();
      $translator_plugin = $translator->getPlugin();

      if (!$translator_plugin instanceof LaraTranslator) {
        $this->logger->error('Job item @id does not use LaraTranslator plugin', ['@id' => $job_item_id]);
        return;
      }

      // Process the translation.
      $this->logger->info('Processing job item @id from queue', ['@id' => $job_item_id]);
      $translator_plugin->processJobItem($job_item);

      // Check if all job items are complete.
      $this->checkJobCompletion($job);

    }
    catch (\Exception $e) {
      // Check if this is a transient error that should be retried.
      if ($this->isTransientError($e)) {
        $this->logger->warning('Transient error processing job item @id: @error. Will retry.', [
          '@id' => $job_item_id,
          '@error' => $e->getMessage(),
        ]);

        // Re-throw to trigger queue retry mechanism.
        throw new SuspendQueueException($e->getMessage(), $e->getCode(), $e);
      }

      // Permanent error - log and mark as failed.
      $this->logger->error('Permanent error processing job item @id: @error', [
        '@id' => $job_item_id,
        '@error' => $e->getMessage(),
      ]);

      // Try to load job item and mark as failed.
      try {
        $storage = $this->entityTypeManager->getStorage('tmgmt_job_item');
        $job_item = $storage->load($job_item_id);
        if ($job_item instanceof JobItemInterface) {
          $job_item->addMessage('Translation failed: @error', ['@error' => $e->getMessage()], 'error');
        }
      }
      catch (\Exception $inner_exception) {
        $this->logger->error('Failed to mark job item @id as failed: @error', [
          '@id' => $job_item_id,
          '@error' => $inner_exception->getMessage(),
        ]);
      }
    }
  }

  /**
   * Determines if an error is transient and should be retried.
   *
   * @param \Exception $exception
   *   The exception to check.
   *
   * @return bool
   *   TRUE if the error is transient, FALSE otherwise.
   */
  private function isTransientError(\Exception $exception): bool {
    $message = $exception->getMessage();
    $code = $exception->getCode();

    // Quota exceeded errors - should be retried later when quota is available.
    if (str_contains($message, 'exceeded your') && str_contains($message, 'quota')) {
      return TRUE;
    }

    // Network-related errors.
    if (str_contains($message, 'timeout') ||
        str_contains($message, 'connection') ||
        str_contains($message, 'network')) {
      return TRUE;
    }

    // HTTP status codes indicating transient errors.
    $transient_codes = [
    // Too Many Requests.
      429,
    // Internal Server Error.
      500,
    // Bad Gateway.
      502,
    // Service Unavailable.
      503,
    // Gateway Timeout.
      504,
    ];

    if (in_array($code, $transient_codes, TRUE)) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Checks if all items in a job are complete and updates job status.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The translation job.
   */
  private function checkJobCompletion($job): void {
    $all_items_finished = TRUE;
    $has_failed_items = FALSE;

    foreach ($job->getItems() as $item) {
      if ($item->isActive()) {
        $all_items_finished = FALSE;
        break;
      }
      if ($item->isAborted()) {
        $has_failed_items = TRUE;
      }
    }

    if ($all_items_finished) {
      if ($has_failed_items) {
        $this->logger->info('Job @job_id completed with some failed items', ['@job_id' => $job->id()]);
      }
      else {
        $this->logger->info('Job @job_id completed successfully', ['@job_id' => $job->id()]);
      }
    }
  }

}
