<?php

declare(strict_types=1);

namespace Drupal\Tests\tmgmt_laratranslate\Unit\Plugin\tmgmt\Translator;

use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt_laratranslate\Plugin\tmgmt\Translator\LaraTranslator;
use Psr\Log\LoggerInterface;

/**
 * Tests the LaraTranslator plugin.
 *
 * @group tmgmt_laratranslate
 * @coversDefaultClass \Drupal\tmgmt_laratranslate\Plugin\tmgmt\Translator\LaraTranslator
 */
class LaraTranslatorTest extends UnitTestCase {

  /**
   * The LaraTranslator plugin instance.
   */
  protected LaraTranslator $plugin;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * Mock key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $keyRepository;

  /**
   * Mock language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $languageManager;

  /**
   * Mock queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $queueFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->keyRepository = $this->createMock(KeyRepositoryInterface::class);
    $this->languageManager = $this->createMock(LanguageManagerInterface::class);
    $this->queueFactory = $this->createMock(QueueFactory::class);

    // Create the plugin instance.
    $configuration = [];
    $plugin_id = 'lara_translate';
    $plugin_definition = [
      'id' => 'lara_translate',
      'label' => 'Lara Translate',
    ];

    $this->plugin = new LaraTranslator(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->logger,
      $this->keyRepository,
      $this->languageManager,
      $this->queueFactory,
    );
  }

  /**
   * Tests checkAvailable with valid credentials.
   *
   * @covers ::checkAvailable
   */
  public function testCheckAvailableWithValidCredentials(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->exactly(2))
      ->method('getSetting')
      ->willReturnCallback(function ($keys) {
        if ($keys === ['credentials', 'lara_access_key_id']) {
          return 'test_key_id';
        }
        if ($keys === ['credentials', 'lara_access_key_secret']) {
          return 'test_key_secret';
        }
        return NULL;
      });

    $result = $this->plugin->checkAvailable($translator);

    // AvailableResult::yes() returns a truthy result.
    static::assertTrue($result->getSuccess());
  }

  /**
   * Tests getSupportedRemoteLanguages returns empty array on exception.
   *
   * @covers ::getSupportedRemoteLanguages
   */
  public function testGetSupportedRemoteLanguagesHandlesException(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->method('getSetting')
      ->willReturn(NULL);

    $this->logger->expects($this->once())
      ->method('warning')
      ->with(
        static::stringContains('Could not retrieve Lara supported languages'),
        static::isArray(),
      );

    $languages = $this->plugin->getSupportedRemoteLanguages($translator);

    static::assertIsArray($languages);
    static::assertEmpty($languages);
  }

  /**
   * Tests hasCheckoutSettings returns false.
   *
   * @covers ::hasCheckoutSettings
   */
  public function testHasCheckoutSettings(): void {
    $job = $this->createMock(JobInterface::class);

    $result = $this->plugin->hasCheckoutSettings($job);

    static::assertFalse($result);
  }

  /**
   * Tests default language mapping.
   *
   * @covers ::getDefaultLanguageMapping
   */
  public function testGetDefaultLanguageMapping(): void {
    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getDefaultLanguageMapping');
    $method->setAccessible(TRUE);

    $mapping = $method->invoke($this->plugin);

    static::assertIsArray($mapping);
    static::assertArrayHasKey('en', $mapping);
    static::assertEquals('en-US', $mapping['en']);
    static::assertArrayHasKey('it', $mapping);
    static::assertEquals('it-IT', $mapping['it']);
    static::assertArrayHasKey('fr', $mapping);
    static::assertEquals('fr-FR', $mapping['fr']);
    static::assertArrayHasKey('es', $mapping);
    static::assertEquals('es-ES', $mapping['es']);
  }

  /**
   * Tests mapDrupalToLaraLanguage with custom mapping.
   *
   * @covers ::mapDrupalToLaraLanguage
   */
  public function testMapDrupalToLaraLanguageWithCustomMapping(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with(['language_mappings'])
      ->willReturn(['en' => 'en-GB']);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('mapDrupalToLaraLanguage');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->plugin, 'en', $translator);

    static::assertEquals('en-GB', $result);
  }

  /**
   * Tests mapDrupalToLaraLanguage with default mapping.
   *
   * @covers ::mapDrupalToLaraLanguage
   */
  public function testMapDrupalToLaraLanguageWithDefaultMapping(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with(['language_mappings'])
      ->willReturn(NULL);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('mapDrupalToLaraLanguage');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->plugin, 'it', $translator);

    static::assertEquals('it-IT', $result);
  }

  /**
   * Tests mapDrupalToLaraLanguage with unmapped language.
   *
   * @covers ::mapDrupalToLaraLanguage
   */
  public function testMapDrupalToLaraLanguageWithUnmappedLanguage(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with(['language_mappings'])
      ->willReturn(NULL);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('mapDrupalToLaraLanguage');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->plugin, 'unknown', $translator);

    static::assertNull($result);
  }

  /**
   * Tests getKeyValue with valid key.
   *
   * @covers ::getKeyValue
   */
  public function testGetKeyValueWithValidKey(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with(['credentials', 'test_key'])
      ->willReturn('key_id_123');

    $key = $this->createMock(KeyInterface::class);
    $key->expects($this->once())
      ->method('getKeyValue')
      ->willReturn('secret_value');

    $this->keyRepository->expects($this->once())
      ->method('getKey')
      ->with('key_id_123')
      ->willReturn($key);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getKeyValue');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->plugin, $translator, 'test_key');

    static::assertEquals('secret_value', $result);
  }

  /**
   * Tests getKeyValue with missing key entity.
   *
   * @covers ::getKeyValue
   */
  public function testGetKeyValueWithMissingKeyEntity(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with(['credentials', 'test_key'])
      ->willReturn('key_id_123');

    $this->keyRepository->expects($this->once())
      ->method('getKey')
      ->with('key_id_123')
      ->willReturn(NULL);

    $this->logger->expects($this->once())
      ->method('warning')
      ->with(
        static::stringContains('Key entity'),
        static::isArray(),
      );

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getKeyValue');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->plugin, $translator, 'test_key');

    static::assertNull($result);
  }

  /**
   * Tests getKeyValue with NULL setting.
   *
   * @covers ::getKeyValue
   */
  public function testGetKeyValueWithNullSetting(): void {
    $translator = $this->createMock(TranslatorInterface::class);
    $translator->expects($this->once())
      ->method('getSetting')
      ->with(['credentials', 'test_key'])
      ->willReturn(NULL);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getKeyValue');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->plugin, $translator, 'test_key');

    static::assertNull($result);
  }

}
