# Critical Fix: Continuous Translation Blocking Content Save

## Issue

When using **continuous translation** (automatic translation on content create/update), the system was blocking content save operations when quota errors occurred.

### Error Observed

```
Drupal\Core\Entity\EntityStorageException: You have exceeded your "api_translation_chars" quota 
in Drupal\Core\Entity\Sql\SqlContentEntityStorage->save() (line 815)
```

### Stack Trace Analysis

```
SqlContentEntityStorage->save()
↓
EntityStorageBase->doPostSave()
↓
ModuleHandler->invokeAll('entity_update')
↓
tmgmt_content_entity_update()
↓
tmgmt_content_create_continuous_job_items()
↓
ContinuousManager->addItem()
↓
LaraTranslator->requestJobItemsTranslation()  ← CALLED DURING SAVE!
↓
LaraTranslator->processJobItem()               ← PROCESSING SYNCHRONOUSLY!
↓
LaraTranslator->translateData()
↓
LaraTranslator->translateText()
↓
LaraTranslatorSDK->translate()
↓
💥 LaraException: "You have exceeded your quota"
↓
❌ Exception bubbles up and blocks save operation!
```

## Root Cause

The `requestJobItemsTranslation()` method was still processing items **synchronously** even though we implemented a queue system for `requestTranslation()`.

**Why this matters:**
- `requestTranslation()` - Called for **manual** translation jobs (via UI)
- `requestJobItemsTranslation()` - Called for **continuous** translation (during content save)

We fixed the first but not the second!

## The Fix

### Changed Method Implementation

**File**: `src/Plugin/tmgmt/Translator/LaraTranslator.php`

**Before** (Blocking):
```php
public function requestJobItemsTranslation(array $job_items): void {
  foreach ($job_items as $job_item) {
    try {
      $this->processJobItem($job_item);  // ❌ SYNCHRONOUS - BLOCKS!
    }
    catch (TMGMTException $e) {
      $this->logger->error('Job item @item_id translation failed: @error', [
        '@item_id' => $job_item->id(),
        '@error' => $e->getMessage(),
      ]);
      $job_item->addMessage('Translation failed: @error', ['@error' => $e->getMessage()], 'error');
    }
  }
}
```

**After** (Non-blocking):
```php
public function requestJobItemsTranslation(array $job_items): void {
  // Queue all job items for background processing.
  $queue = $this->queueFactory->get('tmgmt_laratranslate_worker');
  $item_count = 0;

  foreach ($job_items as $job_item) {
    // Mark job item as active.
    $job_item->active();

    // Add to queue.
    $queue->createItem([
      'job_item_id' => $job_item->id(),
      'submitted_time' => time(),
    ]);

    $item_count++;

    $this->logger->info('Job item @item_id queued for translation', [
      '@item_id' => $job_item->id(),
    ]);
  }

  $this->logger->info('Queued @count job item(s) for translation', [
    '@count' => $item_count,
  ]);
}
```

## Impact

### Before Fix

```
User workflow:
1. User edits content
2. Clicks "Save"
3. Drupal triggers entity_update hook
4. TMGMT continuous translation triggered
5. requestJobItemsTranslation() called
6. Translation API called immediately
7. Quota exceeded error thrown
8. ❌ EntityStorageException
9. ❌ Content NOT saved
10. ❌ User sees error message
11. ❌ User frustrated - can't save work!
```

### After Fix

```
User workflow:
1. User edits content
2. Clicks "Save"
3. Drupal triggers entity_update hook
4. TMGMT continuous translation triggered
5. requestJobItemsTranslation() called
6. Translation job QUEUED (not processed)
7. ✅ Method returns immediately
8. ✅ Content save completes successfully
9. ✅ User sees "Content saved" message
10. ✅ Translation happens later via cron
11. ✅ User can continue working
```

## Why This Is Critical

Continuous translation is a **core feature** of TMGMT for:
- High-volume multilingual sites
- Automatic translation workflows
- Content editor productivity

**Without this fix:**
- Content editors are blocked when quota is low
- Production workflows break
- Emergency: editors can't publish content
- Poor user experience

**With this fix:**
- Content editing never blocked
- Translation happens asynchronously
- Quota issues transparent to editors
- Professional, reliable system

## Testing

### Reproduce the Original Issue (Before Fix)

1. Enable continuous translation for a content type
2. Use Lara account with exceeded quota
3. Edit content and click Save
4. ❌ Should see EntityStorageException (without fix)
5. ❌ Content should NOT be saved

### Verify the Fix

1. Apply the fix (queue-based requestJobItemsTranslation)
2. Clear cache: `ddev drush cr`
3. Edit content and click Save
4. ✅ Content should save successfully
5. ✅ Check queue: `ddev drush queue:list`
6. ✅ Should see items queued
7. ✅ Check logs: `ddev drush watchdog:show --type=tmgmt_laratranslate`
8. ✅ Should see "Job item X queued for translation"

### Test Flow

```bash
# Clear cache
ddev drush cr

# Enable continuous translation
# (via TMGMT UI: Configuration > Translation Management > Continuous Jobs)

# Edit content with quota exceeded
# Click Save

# Verify content saved
echo "✅ If you see this, content saved successfully!"

# Check queue
ddev drush queue:list
# Should show: tmgmt_laratranslate_worker with items

# Check logs
ddev drush watchdog:show --type=tmgmt_laratranslate | tail -20
# Should show: "Job item X queued for translation"

# Process queue when quota available
ddev drush queue:run tmgmt_laratranslate_worker
```

## Related Documentation

- **QUEUE_IMPLEMENTATION_PROPOSAL.md** - Original design
- **QUEUE_IMPLEMENTATION_PHASE1_COMPLETE.md** - Implementation summary  
- **QUOTA_ERROR_HANDLING.md** - Quota error handling strategy

## Lessons Learned

### Why We Missed This Initially

1. **Two entry points**: Manual (`requestTranslation`) vs Continuous (`requestJobItemsTranslation`)
2. **Different call paths**: UI trigger vs entity hook trigger
3. **Testing focus**: Tested manual jobs but not continuous translation flow
4. **Stack depth**: Error occurs deep in Drupal's entity save process

### Best Practices Going Forward

1. ✅ Test **both** manual and continuous translation workflows
2. ✅ Always test with quota errors/limits
3. ✅ Check ALL methods that might be called during content save
4. ✅ Never do synchronous API calls in entity hooks
5. ✅ Queue-based processing should be used everywhere, not just UI-triggered jobs

## Conclusion

This fix ensures that continuous translation **never blocks content editing operations**, even when API quota limits are exceeded. This is essential for production multilingual websites where content editors need reliable, non-blocking workflows.

The fix aligns both entry points (manual and continuous) to use the same queue-based architecture, ensuring consistent behavior regardless of how translation is triggered.
