# Language Code Mapping Implementation

## Overview

This module implements a **Hybrid Approach** for mapping Drupal's ISO 639-1 (2-letter) language codes to Lara Translate's full locale codes (e.g., `en-US`, `it-IT`).

## Problem

- **Drupal** uses ISO 639-1 codes: `en`, `it`, `de`, `fr`, etc.
- **Lara Translate** uses full locale codes: `en-US`, `it-IT`, `de-DE`, `fr-FR`, etc.

Without proper mapping, translation requests fail with errors like:
```
Translation job failed: The language pair en to it is not supported by Lara Translate.
```

## Solution: Hybrid Mapping Approach

The implementation provides two layers of language code mapping:

### 1. Default Static Mappings

Located in `LaraTranslator::getDefaultLanguageMapping()`, this method provides sensible defaults for all languages supported by Lara Translate:

- Single-variant languages map directly (e.g., `it` → `it-IT`)
- Multi-variant languages use common defaults:
  - `en` → `en-US` (US English as default)
  - `es` → `es-ES` (European Spanish as default)
  - `fr` → `fr-FR` (French French as default)
  - `pt` → `pt-PT` (European Portuguese as default)
- Regional variants are also supported (e.g., `en-gb` → `en-GB`, `pt-br` → `pt-BR`)
- Chinese variants use Drupal's standard codes:
  - `zh-hans` → `zh-CN` (Simplified Chinese)
  - `zh-hant` → `zh-TW` (Traditional Chinese)

### 2. Custom Configuration Mappings

Site administrators can override default mappings via the translator configuration. This is useful when:

- A site prefers a different variant (e.g., `en` → `en-GB` instead of `en-US`)
- Custom language codes need special handling
- Regional variations need specific mappings

Configuration is stored in the `language_mappings` setting as an array:

```php
[
  'en' => 'en-GB',  // Override: Use British English instead of US
  'zh' => 'zh-CN',  // Custom: Map generic Chinese to Simplified
]
```

## Implementation Details

### Key Methods

#### `mapDrupalToLaraLanguage(string $drupal_langcode, TranslatorInterface $translator): ?string`

The main mapping method that:
1. Checks for custom mappings in translator settings first
2. Falls back to default static mappings
3. Returns `NULL` if no mapping exists

#### `getDefaultLanguageMapping(): array`

Returns the complete default mapping array covering all 42 Lara-supported languages.

#### `getSupportedRemoteLanguages(TranslatorInterface $translator): array`

Filters site languages to only those that:
1. Can be mapped to Lara language codes
2. Are supported by Lara Translate API

#### `isLanguagePairSupported(string $source_langcode, string $target_langcode, TranslatorInterface $translator): bool`

Validates language pairs by:
1. Mapping both source and target codes
2. Checking if both exist in Lara's supported languages
3. Returning `FALSE` if either mapping fails

#### `translateText(...)`

Uses mapped language codes when calling the Lara SDK:
```php
$lara_source = $this->mapDrupalToLaraLanguage($source_langcode, $translator);
$lara_target = $this->mapDrupalToLaraLanguage($target_langcode, $translator);
```

## Supported Language Mappings

| Drupal Code | Lara Code | Language |
|-------------|-----------|----------|
| `ar` | `ar-SA` | Arabic (Saudi Arabia) |
| `bg` | `bg-BG` | Bulgarian |
| `ca` | `ca-ES` | Catalan |
| `cs` | `cs-CZ` | Czech |
| `da` | `da-DK` | Danish |
| `de` | `de-DE` | German |
| `el` | `el-GR` | Greek |
| `en` | `en-US` | English (US) ⭐ |
| `en-au` | `en-AU` | English (Australia) |
| `en-ca` | `en-CA` | English (Canada) |
| `en-gb` | `en-GB` | English (UK) |
| `en-ie` | `en-IE` | English (Ireland) |
| `es` | `es-ES` | Spanish (Spain) ⭐ |
| `es-ar` | `es-AR` | Spanish (Argentina) |
| `es-mx` | `es-MX` | Spanish (Mexico) |
| `fi` | `fi-FI` | Finnish |
| `fr` | `fr-FR` | French (France) ⭐ |
| `fr-ca` | `fr-CA` | French (Canada) |
| `he` | `he-IL` | Hebrew |
| `hr` | `hr-HR` | Croatian |
| `hu` | `hu-HU` | Hungarian |
| `id` | `id-ID` | Indonesian |
| `it` | `it-IT` | Italian |
| `ja` | `ja-JP` | Japanese |
| `ko` | `ko-KR` | Korean |
| `ms` | `ms-MY` | Malay |
| `nb` | `nb-NO` | Norwegian Bokmål |
| `nl` | `nl-NL` | Dutch (Netherlands) ⭐ |
| `nl-be` | `nl-BE` | Dutch (Belgium) |
| `pl` | `pl-PL` | Polish |
| `pt` | `pt-PT` | Portuguese (Portugal) ⭐ |
| `pt-br` | `pt-BR` | Portuguese (Brazil) |
| `ru` | `ru-RU` | Russian |
| `sk` | `sk-SK` | Slovak |
| `sv` | `sv-SE` | Swedish |
| `th` | `th-TH` | Thai |
| `tr` | `tr-TR` | Turkish |
| `uk` | `uk-UA` | Ukrainian |
| `zh-hans` | `zh-CN` | Chinese Simplified |
| `zh-hant` | `zh-TW` | Chinese Traditional |
| `zh-hk` | `zh-HK` | Chinese (Hong Kong) |

⭐ = Default variant for multi-variant languages

## Configuration

### Via Configuration Schema

The `language_mappings` setting is defined in `config/schema/tmgmt_laratranslate.schema.yml`:

```yaml
language_mappings:
  type: sequence
  label: 'Custom Language Code Mappings'
  sequence:
    type: mapping
    mapping:
      drupal_code:
        type: string
        label: 'Drupal Language Code'
      lara_code:
        type: string
        label: 'Lara Language Code'
```

### Programmatic Configuration

Custom mappings can be set programmatically:

```php
$translator->setSetting(['language_mappings'], [
  'en' => 'en-GB',
  'es' => 'es-MX',
  'pt' => 'pt-BR',
]);
$translator->save();
```

## Testing

To verify the implementation works:

1. **Check Supported Languages**:
   - Visit the translator configuration page
   - The "Supported Languages" field should show only languages that can be mapped

2. **Test Translation**:
   - Create a translation job from `en` to `it`
   - Should now successfully translate instead of failing

3. **View Logs**:
   - Check `admin/reports/dblog` for messages
   - Should see successful translation messages
   - No more "language pair not supported" errors

## Benefits

✅ **Works out-of-the-box**: No configuration required for standard language pairs  
✅ **Flexible**: Site admins can override defaults for specific needs  
✅ **Maintainable**: Centralized mapping logic in one place  
✅ **Extensible**: Easy to add new language mappings  
✅ **Fail-safe**: Returns NULL for unsupported languages instead of crashing  
✅ **Logged**: All mapping failures are logged for debugging

## Future Enhancements

Possible improvements:

- Add a UI form to manage custom language mappings
- Auto-detect best variant based on site default country
- Support for additional Lara language codes as they're added
- Validation to ensure custom mappings use valid Lara codes
