# Queue-Based Translation Processing Proposal

## Overview

This document proposes implementing a Drupal Queue-based system for the LaraTranslate TMGMT plugin, inspired by the `ai_tmgmt` module's approach. This will allow translation jobs to be processed asynchronously during cron runs instead of immediately when submitted.

## Current Implementation

Currently, the `LaraTranslator` plugin processes translations synchronously:

1. Translation is triggered by:
   - User manually submitting a translation job, OR
   - Continuous translation when content is created/updated (via `ContinuousTranslatorInterface`)
2. `requestTranslation()` is called immediately
3. `requestJobItemsTranslation()` processes all job items synchronously
4. Each job item is translated via API calls in `processJobItem()`
5. The job is marked as submitted only after all items are processed

**Problems with this approach:**
- **Performance**: Large jobs with many items can timeout
- **User Experience**: User must wait for entire job to complete before UI responds (manual jobs)
- **Content Operations Blocking**: Content save operations are blocked until translation completes (continuous translation)
- **Reliability**: If one API call fails mid-process, the entire job may be left in an inconsistent state
- **Resource Management**: No control over API rate limiting or concurrent requests

## Proposed Solution: Queue-Based Processing

### Architecture Overview

The proposed solution follows Drupal's Queue API pattern:

```mermaid
flowchart TD
    A[Translation Triggered] --> B{Trigger Type}
    B -->|Manual| C[User submits translation job]
    B -->|Continuous| D[Content created/updated]

    C --> E[requestTranslation]
    D --> E

    E --> F[Validate job configuration]
    F --> G[Queue all job items]
    G --> H[Mark job as 'submitted']

    H --> I[Drupal Queue System]
    I -->|Stores job item IDs| J[Wait for cron]

    J --> K[Cron runs]
    K --> L[Process N items per run]

    L --> M[LaraTranslatorWorker::processItem]
    M --> N[Load job item by ID]
    N --> O[Validate job item is active]
    O --> P[Translate via Lara API]
    P --> Q[Update job item state]
    Q --> R[Check if all items complete]
    R -->|Yes| S[Mark job as finished]
    R -->|No| T[Continue processing on next cron]

    M -.->|On Error| U[Handle error gracefully]
    U -->|Transient| V[Retry on next cron]
    U -->|Permanent| W[Mark item as failed]
```

### Components to Implement

#### 1. Queue Worker Plugin: `LaraTranslatorWorker`

**File**: `src/Plugin/QueueWorker/LaraTranslatorWorker.php`

```php
/**
 * Processes Lara Translate translation queue items.
 *
 * @QueueWorker(
 *   id = "tmgmt_laratranslate_worker",
 *   title = @Translation("Lara Translate Worker"),
 *   cron = {"time" = 60}
 * )
 */
class LaraTranslatorWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  public function processItem($data) {
    // Load job item by ID
    // Validate job item is still active
    // Process translation
    // Handle success/failure
    // Update job status if all items complete
  }
}
```

**Key features:**
- `cron = {"time" = 60}`: Process items for max 60 seconds per cron run
- Implements `QueueWorkerBase` for automatic queue integration
- Handles individual job item processing
- Graceful error handling with logging
- Automatic retry logic via Drupal's queue system

#### 2. Modified Translator Plugin

**File**: `src/Plugin/tmgmt/Translator/LaraTranslator.php`

**Changes needed:**

1. **Inject Queue Factory service** in constructor:
```php
private readonly QueueFactoryInterface $queueFactory;
```

2. **Modify `requestTranslation()` method**:
```php
public function requestTranslation(JobInterface $job): void {
  // Validate translator configuration
  // Validate language pair support

  // Instead of processing immediately, queue all items
  $queue = $this->queueFactory->get('tmgmt_laratranslate_worker');

  foreach ($job->getItems() as $job_item) {
    // Mark job item as active/in-progress
    $job_item->active();

    // Add to queue
    $queue->createItem([
      'job_item_id' => $job_item->id(),
      'submitted_time' => time(),
    ]);
  }

  // Mark job as submitted immediately
  $job->submitted('The translation job has been queued for processing.');

  $this->logger->info('Translation job @job_id queued with @count items', [
    '@job_id' => $job->id(),
    '@count' => count($job->getItems()),
  ]);
}
```

3. **Keep existing `processJobItem()` method**:
   - This will be called by the QueueWorker
   - No changes needed to actual translation logic
   - Already handles errors appropriately

4. **Add helper method for checking job completion**:
```php
private function checkJobCompletion(JobInterface $job): void {
  $all_items_finished = TRUE;
  foreach ($job->getItems() as $item) {
    if ($item->isActive()) {
      $all_items_finished = FALSE;
      break;
    }
  }

  if ($all_items_finished) {
    // Job can be marked as finished/needs_review
    // Existing logic handles this
  }
}
```

#### 3. Service Configuration

**File**: `tmgmt_laratranslate.services.yml`

No changes required. The `QueueFactory` service is already available in the Drupal container and will be injected via the `create()` method in both the `LaraTranslator` plugin and the `LaraTranslatorWorker`.

### Queue Behavior

- **Queue name**: `tmgmt_laratranslate_worker`
- **Processing**: Automatic during cron runs
- **Time limit**: 60 seconds per cron run (defined in QueueWorker annotation)
- **Concurrency**: Single-threaded per cron run (Drupal default)
- **Persistence**: Database-backed (survives restarts)
- **Retry logic**: Automatic via Drupal's queue system

### Error Handling Strategy

#### Transient Errors (Retry)
- Network timeouts
- Rate limiting (HTTP 429)
- Temporary API unavailability (HTTP 5xx)

**Action**: Return item to queue for retry (Drupal handles automatic retry)

#### Permanent Errors (Mark as Failed)
- Invalid credentials (HTTP 401/403)
- Unsupported language pair (HTTP 400)
- Malformed request data
- Job item no longer exists

**Action**: Mark job item as failed, log error, remove from queue

#### Implementation in Worker:
```php
public function processItem($data) {
  try {
    // Load and process job item
    $this->processJobItem($job_item);
  }
  catch (LaraException $e) {
    // Check if retryable
    if ($this->isTransientError($e)) {
      // Re-throw to trigger queue retry
      throw new SuspendQueueException($e->getMessage(), $e->getCode(), $e);
    }
    else {
      // Mark as failed permanently
      $job_item->addMessage('Translation failed: @error',
        ['@error' => $e->getMessage()], 'error');
      // Don't re-throw - removes from queue
    }
  }
}
```

### Implementation Strategy

#### Phase 1: Implement Core Queue System
1. Create `LaraTranslatorWorker` plugin
2. Add `QueueFactory` to `LaraTranslator` constructor
3. Modify `requestTranslation()` to use queue

#### Phase 2: Testing
1. Unit tests for queue worker
2. Kernel tests for queue integration
3. Manual testing with various job sizes
4. Performance benchmarking

#### Phase 3: Documentation
1. Update README with queue information
2. Add queue troubleshooting guide
3. Document cron configuration recommendations

### Implementation Approach

Since this is a new module not yet in production use, we can implement queue processing as the default and only behavior:

1. **Queue processing**: Always enabled
2. **No synchronous fallback**: Simplifies implementation and testing
3. **Clean architecture**: No legacy code to maintain
4. **API compatibility**: Public method signatures remain unchanged
5. **Continuous translation**: Full support via `ContinuousTranslatorInterface`

```php
public function requestTranslation(JobInterface $job): void {
  // Validate translator configuration
  // Validate language pair support

  // Queue all job items
  $this->queueJobItems($job);

  // Mark job as submitted immediately
  $job->submitted('The translation job has been queued for processing.');
}
```

This approach provides a cleaner, more maintainable codebase without the complexity of supporting multiple processing modes.

### Benefits of Queue-Based Approach

#### Performance Benefits
- **No timeouts**: Large jobs won't timeout the web request
- **Better resource utilization**: Spread processing across multiple cron runs
- **Rate limiting friendly**: Can throttle API requests automatically
- **Batch processing**: Can optimize API calls by batching segments

#### Reliability Benefits
- **Fault tolerance**: Failed items don't block entire job
- **Automatic retries**: Transient errors handled automatically
- **Consistent state**: Job items independently tracked
- **Resumable**: System restarts don't lose progress

#### User Experience Benefits
- **Immediate feedback**: UI responds instantly after submission
- **Progress tracking**: Can monitor individual item status
- **Non-blocking**: Users can continue working while translation runs
- **Continuous translation**: Content save operations complete quickly without waiting for API calls
- **Transparent**: Clear visibility into processing state

### Monitoring and Administration

#### Queue Status Information

Administrators can check queue status:

1. **Via Drush**:
```bash
drush queue:list
drush queue:run tmgmt_laratranslate_worker
```

2. **Via Admin UI** (if Queue UI module installed):
   - View pending items count
   - Manually trigger queue processing
   - Clear queue if needed

3. **Via Logs**:
   - Monitor processing in `tmgmt_laratranslate` log channel
   - Track success/failure rates
   - Identify problematic job items

#### Metrics to Track

- Items processed per cron run
- Average processing time per item
- Success/failure rates
- Queue depth over time
- API response times

### Comparison with ai_tmgmt Module

#### Similarities
- Queue-based processing architecture
- Cron-triggered workers
- Individual job item processing
- Error handling with retries
- Logging and monitoring

#### Differences for LaraTranslate
- **Simpler authentication**: Key-based vs AI provider diversity
- **No model selection**: Single API endpoint
- **Different options**: Lara-specific settings (adaptTo, glossaries, etc.)
- **Language mapping**: Custom Drupal → Lara code mapping
- **HTML content**: Lara handles HTML natively (multiline support)

### Potential Enhancements (Future)

1. **Priority Queue**: Process urgent translations first
2. **Batch API calls**: Group multiple segments in single API request
3. **Progress notifications**: Email/webhook when job completes
4. **Queue dashboard**: Custom admin page showing queue status
5. **Dynamic throttling**: Adjust processing rate based on API response
6. **Multi-language batching**: Optimize for same-source translations
7. **Parallel processing**: Use Advanced Queue module for concurrency

## Implementation Checklist

- [ ] Create `src/Plugin/QueueWorker/LaraTranslatorWorker.php`
- [ ] Add `QueueFactory` injection to `LaraTranslator`
- [ ] Modify `requestTranslation()` to queue items
- [ ] Add error handling for queue processing
- [ ] Create unit tests for queue worker
- [ ] Create kernel tests for queue integration
- [ ] Update module documentation
- [ ] Add troubleshooting guide for queue issues
- [ ] Performance testing with large jobs
- [ ] Update CHANGELOG.md with queue feature

## Conclusion

Implementing a queue-based translation system for the LaraTranslate module will significantly improve performance, reliability, and user experience. The proposed architecture follows Drupal best practices and is inspired by the proven approach in the `ai_tmgmt` module.

Since this is a new module not yet in production, we can implement a clean, queue-only architecture without the burden of maintaining backwards compatibility. This simplifies the implementation and provides a robust foundation for future enhancements.

The queue-based approach is particularly beneficial for continuous translation scenarios where content editors should not be blocked waiting for API calls to complete when saving content.

**Recommended approach**: Implement the core queue system (Phase 1) and test thoroughly. The queue uses Drupal's default processing behavior (60 seconds per cron run) which is sufficient for most use cases.
