<?php

declare(strict_types=1);

namespace Drupal\tmgmt_laratranslate\Plugin\tmgmt\Translator;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt_laratranslate\Service\RecursiveCharacterTextSplitter;
use Lara\LaraCredentials;
use Lara\LaraException;
use Lara\TextBlock;
use Lara\TranslateOptions;
use Lara\Translator as LaraTranslatorSDK;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Lara Translator plugin.
 *
 * @TranslatorPlugin(
 *   id = "lara_translate",
 *   label = @Translation("Lara Translate"),
 *   description = @Translation("Lara Translate integration"),
 *   default_settings = {
 *     "lara_access_key_id" = "",
 *     "lara_access_key_secret" = "",
 *     "adaptTo" = {},
 *     "glossaries" = {},
 *     "instructions" = {},
 *     "style" = "fluid",
 *   },
 *   logo = "icons/lara-translate.svg",
 *   ui = "Drupal\tmgmt_laratranslate\LaraTranslatorUi",
 * )
 */
final class LaraTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface, ContinuousTranslatorInterface {

  /**
   * Maximum allowed length for HTML content.
   *
   * Lara Translate API has a 10,000 character limit. We use 9,900 character
   * limit to provide 100-char safety buffer.
   *
   * @var int
   */
  private const MAX_LENGTH = 9900;

  /**
   * Constructs a new LaraTranslator object.
   *
   * @param array<array-key, mixed> $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger channel factory.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   The queue factory service.
   * @param \Drupal\tmgmt_laratranslate\Service\RecursiveCharacterTextSplitter $textChunker
   *   The text chunker service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    private readonly LoggerInterface $logger,
    private readonly KeyRepositoryInterface $keyRepository,
    private readonly LanguageManagerInterface $languageManager,
    private readonly QueueFactory $queueFactory,
    private readonly RecursiveCharacterTextSplitter $textChunker,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   *
   * @phpstan-param array<array-key, mixed> $configuration
   * @phpstan-param string $plugin_id
   * @phpstan-param mixed $plugin_definition
   * @phpstan-return static
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $logger = $container->get('logger.channel.tmgmt_laratranslate');
    \assert($logger instanceof LoggerInterface);

    $key_repository = $container->get('key.repository');
    \assert($key_repository instanceof KeyRepositoryInterface);

    $language_manager = $container->get('language_manager');
    \assert($language_manager instanceof LanguageManagerInterface);

    $queue_factory = $container->get('queue');
    \assert($queue_factory instanceof QueueFactory);

    $text_chunker = $container->get(RecursiveCharacterTextSplitter::class);
    \assert($text_chunker instanceof RecursiveCharacterTextSplitter);

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $logger,
      $key_repository,
      $language_manager,
      $queue_factory,
      $text_chunker,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator): AvailableResult {
    if (
      $translator->getSetting(['credentials', 'lara_access_key_id']) !== '' &&
      $translator->getSetting(['credentials', 'lara_access_key_secret']) !== ''
    ) {
      return AvailableResult::yes();
    }

    return AvailableResult::no($this->t('@translator is not available. Make sure it is properly <a href=":configured">configured</a>.', [
      '@translator' => $translator->label(),
      ':configured' => $translator->toUrl()->toString(),
    ]));
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator plugin.
   *
   * @phpstan-return array<string, string>
   *   An associative array of langcode => language name.
   */
  public function getSupportedRemoteLanguages(TranslatorInterface $translator): array {
    try {
      $lara_translator = $this->createLaraTranslator($translator);
      $lara_languages = $lara_translator->getLanguages();

      $languages = [];
      $site_languages = $this->languageManager->getLanguages();

      // Only return site languages that can be mapped to Lara languages.
      foreach ($site_languages as $langcode => $language) {
        $lara_code = $this->mapDrupalToLaraLanguage((string) $langcode, $translator);
        if ($lara_code !== NULL && \in_array($lara_code, $lara_languages, TRUE)) {
          $languages[$langcode] = $language->getName();
        }
      }

      return $languages;
    }
    catch (\Exception $e) {
      $this->logger->warning('Could not retrieve Lara supported languages: @error', ['@error' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator plugin.
   * @phpstan-param mixed $source_language
   *   The source language.
   *
   * @phpstan-return array<string, string>
   *   An associative array of langcode => language name.
   */
  public function getSupportedTargetLanguages(TranslatorInterface $translator, $source_language): array {
    $languages = $this->getSupportedRemoteLanguages($translator);
    // There are no language pairs, any supported language can be translated
    // into the others. If the source language is part of the languages,
    // then return them all, just remove the source language.
    if (\array_key_exists($source_language, $languages)) {
      unset($languages[$source_language]);
      return $languages;
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function hasCheckoutSettings(JobInterface $job): bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job): void {
    try {
      // Validate translator configuration.
      $translator = $job->getTranslator();

      // Get credentials from translator settings.
      $access_key_id = $translator->getSetting(['credentials', 'lara_access_key_id']);
      $access_key_secret = $translator->getSetting(['credentials', 'lara_access_key_secret']);

      if ($access_key_id === NULL || $access_key_secret === NULL) {
        throw new TMGMTException('Lara Translate credentials are not properly configured.');
      }

      // Validate language pair support.
      $source_langcode = $job->getSourceLangcode();
      $target_langcode = $job->getTargetLangcode();

      if (!$this->isLanguagePairSupported($source_langcode, $target_langcode, $translator)) {
        throw new TMGMTException('The language pair @source_langcode to @target_langcode is not supported by Lara Translate.', [
          '@source_langcode' => $source_langcode,
          '@target_langcode' => $target_langcode,
        ]);
      }

      $this->logger->info('Starting translation job @job_id from @source to @target', [
        '@job_id' => $job->id(),
        '@source' => $source_langcode,
        '@target' => $target_langcode,
      ]);

      // Queue all job items for processing.
      $queue = $this->queueFactory->get('tmgmt_laratranslate_worker');
      $item_count = 0;

      foreach ($job->getItems() as $job_item) {
        // Mark job item as active.
        $job_item->active();

        // Add to queue.
        $queue->createItem([
          'job_item_id' => $job_item->id(),
          'submitted_time' => time(),
        ]);

        $item_count++;
      }

      // Mark job as submitted.
      $job->submitted('The translation job has been queued for processing. @count item(s) will be processed during cron runs.', [
        '@count' => $item_count,
      ]);

      $this->logger->info('Translation job @job_id queued with @count item(s)', [
        '@job_id' => $job->id(),
        '@count' => $item_count,
      ]);
    }
    catch (TMGMTException $e) {
      $this->logger->error('Translation job @job_id failed: @error', [
        '@job_id' => $job->id(),
        '@error' => $e->getMessage(),
      ]);
      $job->rejected('Translation job failed: @error', ['@error' => $e->getMessage()]);
    }
    catch (\Exception $e) {
      $this->logger->error('Unexpected error in translation job @job_id: @error', [
        '@job_id' => $job->id(),
        '@error' => $e->getMessage(),
      ]);
      $job->rejected('An unexpected error occurred: @error', ['@error' => $e->getMessage()]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function requestJobItemsTranslation(array $job_items): void {
    // Queue all job items for background processing.
    $queue = $this->queueFactory->get('tmgmt_laratranslate_worker');
    $item_count = 0;

    foreach ($job_items as $job_item) {
      // Mark job item as active.
      $job_item->active();

      // Add to queue.
      $queue->createItem([
        'job_item_id' => $job_item->id(),
        'submitted_time' => time(),
      ]);

      $item_count++;

      $this->logger->info('Job item @item_id queued for translation', [
        '@item_id' => $job_item->id(),
      ]);
    }

    $this->logger->info('Queued @count job item(s) for translation', [
      '@count' => $item_count,
    ]);
  }

  /**
   * Processes a single job item for translation.
   *
   * This method is called by the queue worker to translate individual
   * job items.
   *
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item to process.
   *
   * @throws \Drupal\tmgmt\TMGMTException
   *   If translation fails.
   * @throws \Lara\LaraException
   *   If a quota error or other Lara API error occurs.
   */
  public function processJobItem(JobItemInterface $job_item): void {
    $job = $job_item->getJob();
    $translator = $job->getTranslator();

    // Initialize Lara SDK.
    $lara_translator = $this->createLaraTranslator($translator);

    // Get translatable data from the job item.
    $data = $job_item->getData();

    if ($data === NULL) {
      throw new TMGMTException('No translatable data found in job item.');
    }
    if (!\is_array($data)) {
      throw new TMGMTException('Invalid data format in job item, expected array but got ' . gettype($data) . '.');
    }

    try {
      // Extract translatable segments and translate them.
      $translated_data = $this->translateData($data, $job, $lara_translator);

      // Add the translated data back to the job item.
      $job_item->addTranslatedData($translated_data);

      // Mark the job item as translated.
      $job_item->addMessage('Translation completed successfully via Lara Translate.');

      $this->logger->info('Job item @item_id successfully translated', ['@item_id' => $job_item->id()]);
    }
    catch (LaraException $e) {
      // Check if this is a quota error - add message to job item and log.
      $error_message = $e->getMessage();
      if (str_contains($error_message, 'exceeded your') && str_contains($error_message, 'quota')) {
        // Add message visible in TMGMT UI.
        $job_item->addMessage('Translation quota exceeded. Job item will be automatically retried when quota is available. Error: @error', [
          '@error' => $error_message,
        ], 'warning');

        $this->logger->warning('Job item @item_id translation failed due to quota exceeded. Item will be retried. Error: @error', [
          '@item_id' => $job_item->id(),
          '@error' => $error_message,
        ]);
      }
      // Re-throw the exception so the queue worker can handle it.
      throw $e;
    }
  }

  /**
   * Creates and configures a Lara translator instance.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The TMGMT translator entity.
   *
   * @return \Lara\Translator
   *   The configured Lara translator instance.
   *
   * @throws \Drupal\tmgmt\TMGMTException
   *   If credentials are missing or invalid.
   */
  private function createLaraTranslator(TranslatorInterface $translator): LaraTranslatorSDK {
    // Get actual key values using KeyRepository.
    $access_key_id = $this->getKeyValue($translator, 'lara_access_key_id');
    $access_key_secret = $this->getKeyValue($translator, 'lara_access_key_secret');

    if ($access_key_id === NULL || $access_key_secret === NULL) {
      throw new TMGMTException('Lara Translate credentials are not configured.');
    }

    try {
      $credentials = new LaraCredentials($access_key_id, $access_key_secret);

      return new LaraTranslatorSDK($credentials);
    }
    catch (\Exception $e) {
      throw new TMGMTException('Failed to initialize Lara Translate client: @error', ['@error' => $e->getMessage()]);
    }
  }

  /**
   * Gets the actual key value from the Key module.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator entity.
   * @param string $key_setting_name
   *   The name of the key setting (e.g., 'lara_access_key_id').
   *
   * @return string|null
   *   The actual key value, or NULL if not found.
   */
  private function getKeyValue(TranslatorInterface $translator, string $key_setting_name): ?string {
    $key_id = $translator->getSetting(['credentials', $key_setting_name]);

    if ($key_id === NULL) {
      return NULL;
    }

    $key = $this->keyRepository->getKey($key_id);
    if ($key === NULL) {
      $this->logger->warning('Key entity @key_id not found for setting @setting', [
        '@key_id' => $key_id,
        '@setting' => $key_setting_name,
      ]);

      return NULL;
    }

    return $key->getKeyValue();
  }

  /**
   * Recursively translates data structure using Lara SDK.
   *
   * @param array<mixed> $data
   *   The data structure to translate.
   * @param \Drupal\tmgmt\JobInterface $job
   *   The translation job.
   * @param \Lara\Translator $lara_translator
   *   The Lara translator instance.
   *
   * @return array<mixed>
   *   The translated data structure.
   *
   * @throws \Drupal\tmgmt\TMGMTException
   *   If translation fails.
   */
  private function translateData(array $data, JobInterface $job, LaraTranslatorSDK $lara_translator): array {
    $translated_data = [];
    $tmgmt_translator = $job->getTranslator();

    // Map Drupal language codes to Lara language codes.
    $source_langcode = $this->mapDrupalToLaraLanguage(
      $job->getSourceLangcode(),
      $tmgmt_translator,
    );
    $target_langcode = $this->mapDrupalToLaraLanguage(
      $job->getTargetLangcode(),
      $tmgmt_translator,
    );

    if ($source_langcode === NULL || $target_langcode === NULL) {
      throw new LaraException('Unable to map language codes: ' . $source_langcode . ' -> ' . $target_langcode);
    }

    foreach ($data as $key => $item) {
      if (is_array($item) && isset($item['#text']) && $item['#text'] !== '') {
        // This is a translatable text segment.
        try {
          $translated_text = $this->translateText(
            $item['#text'],
            $source_langcode,
            $target_langcode,
            $lara_translator,
            $tmgmt_translator,
          );

          $translated_data[$key] = [
            '#text' => $translated_text,
            '#status' => TMGMT_DATA_ITEM_STATE_TRANSLATED,
          ];

          // Preserve any additional metadata.
          foreach ($item as $meta_key => $meta_value) {
            if ($meta_key !== '#text' && $meta_key !== '#status') {
              $translated_data[$key][$meta_key] = $meta_value;
            }
          }
        }
        catch (LaraException $e) {
          $error_message = $e->getMessage();

          // Check if this is a quota exceeded error - these should be retried.
          if (str_contains($error_message, 'exceeded your') && str_contains($error_message, 'quota')) {
            // Re-throw immediately to trigger job item retry via queue.
            // The job item level catch will add a user-visible message.
            throw $e;
          }

          // Other errors: log and mark segment as needs review.
          $this->logger->warning('Failed to translate segment "@text": @error', [
            '@text' => \substr($item['#text'], 0, 100),
            '@error' => $error_message,
          ]);

          // Mark as needs review if translation fails.
          $translated_data[$key] = [
            '#text' => $item['#text'],
            '#status' => TMGMT_DATA_ITEM_STATE_PENDING,
          ];
        }
      }
      elseif (\is_array($item)) {
        // Recursively process nested arrays.
        $translated_data[$key] = $this->translateData($item, $job, $lara_translator);
      }
      else {
        // Copy non-translatable data as-is.
        $translated_data[$key] = $item;
      }
    }

    return $translated_data;
  }

  /**
   * Translates a single text segment using Lara SDK.
   *
   * Checks the total text length against the API's 10,000 character limit.
   * Uses TextChunker for content exceeding 9,900 characters to provide
   * a safety buffer.
   *
   * @param string $text
   *   The text to translate (may contain HTML).
   * @param string $source_langcode
   *   The source language code.
   * @param string $target_langcode
   *   The target language code.
   * @param \Lara\Translator $lara_translator
   *   The Lara translator instance.
   * @param \Drupal\tmgmt\TranslatorInterface $tmgmt_translator
   *   The TMGMT translator entity for settings.
   *
   * @return string
   *   The translated text.
   *
   * @throws \Lara\LaraException
   *   If translation fails.
   */
  private function translateText(
    string $text,
    string $source_langcode,
    string $target_langcode,
    LaraTranslatorSDK $lara_translator,
    TranslatorInterface $tmgmt_translator,
  ): string {
    // CRITICAL: Check TOTAL HTML length (including tags), not just text.
    // The API limit applies to the full HTML string sent to the service.
    $total_length = $this->textChunker->getTextLength($text);

    if ($total_length > self::MAX_LENGTH) {
      // Content exceeds limit - use chunking.
      $this->logger->info('Text exceeds API limit (@total chars), using chunking', [
        '@total' => $total_length,
      ]);

      return $this->translateWithChunking(
        $text,
        $source_langcode,
        $target_langcode,
        $lara_translator,
        $tmgmt_translator
      );
    }

    // Content is within limit - translate directly.
    $this->logger->debug('Translating text segment: @total total chars', [
      '@total' => $total_length,
    ]);

    return $this->performSingleTranslation(
      $text,
      $source_langcode,
      $target_langcode,
      $lara_translator,
      $tmgmt_translator,
    );
  }

  /**
   * Translates content that exceeds the API limit using chunking.
   *
   * @param string $text
   *   The text to translate.
   * @param string $source_langcode
   *   The source language code.
   * @param string $target_langcode
   *   The target language code.
   * @param \Lara\Translator $lara_translator
   *   The Lara translator instance.
   * @param \Drupal\tmgmt\TranslatorInterface $tmgmt_translator
   *   The TMGMT translator entity for settings.
   *
   * @return string
   *   The translated text.
   *
   * @throws \Lara\LaraException
   *   If translation fails.
   */
  private function translateWithChunking(
    string $text,
    string $source_langcode,
    string $target_langcode,
    LaraTranslatorSDK $lara_translator,
    TranslatorInterface $tmgmt_translator,
  ): string {
    // Split text into chunks.
    // Custom configuration.
    $html_chunker = $this->textChunker->forLanguage('html', [
      'chunk_size' => self::MAX_LENGTH,
      'chunk_overlap' => 0,
    ]);
    $chunks = $html_chunker->splitText($text);

    $this->logger->info('Split content into @count chunks for translation', [
      '@count' => count($chunks),
    ]);

    // Translate each chunk.
    $translated_chunks = [];
    foreach ($chunks as $index => $chunk) {
      $chunk_length = $this->textChunker->getTextLength($chunk);

      $this->logger->debug('Translating chunk @num of @total (@length chars)', [
        '@num' => $index + 1,
        '@total' => count($chunks),
        '@length' => $chunk_length,
      ]);

      $translated_chunks[] = $this->performSingleTranslation(
        $chunk,
        $source_langcode,
        $target_langcode,
        $lara_translator,
        $tmgmt_translator,
      );
    }

    // Reassemble translated chunks.
    $result = $this->textChunker->reassembleChunks($translated_chunks);

    $this->logger->info('Successfully reassembled @count chunks into @length chars', [
      '@count' => count($translated_chunks),
      '@length' => $this->textChunker->getTextLength($result),
    ]);

    return $result;
  }

  /**
   * Performs a single translation API call.
   *
   * @param string $text
   *   The text to translate.
   * @param string $source_langcode
   *   The source language code.
   * @param string $target_langcode
   *   The target language code.
   * @param \Lara\Translator $lara_translator
   *   The Lara translator instance.
   * @param \Drupal\tmgmt\TranslatorInterface $tmgmt_translator
   *   The TMGMT translator entity for settings.
   *
   * @return string
   *   The translated text.
   *
   * @throws \Lara\LaraException
   *   If translation fails.
   */
  private function performSingleTranslation(
    string $text,
    string $source_langcode,
    string $target_langcode,
    LaraTranslatorSDK $lara_translator,
    TranslatorInterface $tmgmt_translator,
  ): string {
    // Prepare translation options from translator settings.
    $options = new TranslateOptions();

    // Set adapt_to option if configured.
    $adapt_to = $tmgmt_translator->getSetting(['translation_options', 'adaptTo']);
    if ($adapt_to !== NULL && is_array($adapt_to)) {
      $options->setAdaptTo($adapt_to);
    }

    // Set instructions if configured.
    $instructions = $tmgmt_translator->getSetting(['translation_options', 'instructions']);
    if ($instructions !== NULL && is_array($instructions)) {
      $options->setInstructions($instructions);
    }

    // Set content type (assume HTML for Drupal content).
    $options->setContentType('text/html');
    $options->setMultiline(TRUE);

    // Perform translation.
    $result = $lara_translator->translate($text, $source_langcode, $target_langcode, $options);

    $translation = $result->getTranslation();

    // Handle case where getTranslation() returns an array of TextBlocks
    // or strings.
    if (is_array($translation)) {
      $translation_parts = [];
      foreach ($translation as $part) {
        if (is_string($part)) {
          $translation_parts[] = $part;
        }
        elseif ($part instanceof TextBlock) {
          $translation_parts[] = $part->getText();
        }
        else {
          // Fallback: convert to string.
          $translation_parts[] = (string) $part;
        }
      }
      return implode('', $translation_parts);
    }

    return (string) $translation;
  }

  /**
   * Checks if a language pair is supported by Lara Translate.
   *
   * @param string $source_langcode
   *   The source language code.
   * @param string $target_langcode
   *   The target language code.
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator entity.
   *
   * @return bool
   *   TRUE if the language pair is supported, FALSE otherwise.
   */
  private function isLanguagePairSupported(string $source_langcode, string $target_langcode, TranslatorInterface $translator): bool {
    try {
      $lara_translator = $this->createLaraTranslator($translator);
      $supported_languages = $lara_translator->getLanguages();

      // Map Drupal codes to Lara codes.
      $lara_source = $this->mapDrupalToLaraLanguage($source_langcode, $translator);
      $lara_target = $this->mapDrupalToLaraLanguage($target_langcode, $translator);

      if ($lara_source === NULL || $lara_target === NULL) {
        return FALSE;
      }

      return \in_array($lara_source, $supported_languages, TRUE) &&
             \in_array($lara_target, $supported_languages, TRUE);
    }
    catch (\Exception $e) {
      $this->logger->warning('Could not check language support: @error', ['@error' => $e->getMessage()]);

      return FALSE;
    }
  }

  /**
   * Maps Drupal language codes to Lara Translate language codes.
   *
   * Uses a hybrid approach: first checks for custom mappings in translator
   * settings, then falls back to default static mappings.
   *
   * @param string $drupal_langcode
   *   The Drupal language code (ISO 639-1, e.g., 'en', 'it').
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator entity.
   *
   * @return string|null
   *   The Lara language code (e.g., 'en-US', 'it-IT'), or NULL if not found.
   */
  private function mapDrupalToLaraLanguage(string $drupal_langcode, TranslatorInterface $translator): ?string {
    // Check for custom mapping in translator settings.
    $custom_mappings = $translator->getSetting(['language_mappings']) ?? [];
    if (isset($custom_mappings[$drupal_langcode])) {
      return $custom_mappings[$drupal_langcode];
    }

    // Fall back to default static mapping.
    return $this->getDefaultLanguageMapping()[$drupal_langcode] ?? NULL;
  }

  /**
   * Gets the default language code mappings.
   *
   * Maps Drupal's ISO 639-1 (2-letter) codes to Lara's full locale codes.
   * For languages with multiple variants, sensible defaults are chosen.
   *
   * @return array<string, string>
   *   An associative array of Drupal langcode => Lara langcode mappings.
   */
  private function getDefaultLanguageMapping(): array {
    return [
      'ar' => 'ar-SA',
      'bg' => 'bg-BG',
      'ca' => 'ca-ES',
      'cs' => 'cs-CZ',
      'da' => 'da-DK',
      'de' => 'de-DE',
      'el' => 'el-GR',
      'en' => 'en-US',
      'en-au' => 'en-AU',
      'en-ca' => 'en-CA',
      'en-gb' => 'en-GB',
      'en-ie' => 'en-IE',
      'es' => 'es-ES',
      'es-ar' => 'es-AR',
      'es-mx' => 'es-MX',
      'fi' => 'fi-FI',
      'fr' => 'fr-FR',
      'fr-ca' => 'fr-CA',
      'he' => 'he-IL',
      'hr' => 'hr-HR',
      'hu' => 'hu-HU',
      'id' => 'id-ID',
      'it' => 'it-IT',
      'ja' => 'ja-JP',
      'ko' => 'ko-KR',
      'ms' => 'ms-MY',
      'nb' => 'nb-NO',
      'nl' => 'nl-NL',
      'nl-be' => 'nl-BE',
      'pl' => 'pl-PL',
      'pt' => 'pt-PT',
      'pt-br' => 'pt-BR',
      'ru' => 'ru-RU',
      'sk' => 'sk-SK',
      'sv' => 'sv-SE',
      'th' => 'th-TH',
      'tr' => 'tr-TR',
      'uk' => 'uk-UA',
      'zh-hans' => 'zh-CN',
      'zh-hant' => 'zh-TW',
      'zh-hk' => 'zh-HK',
    ];
  }

}
