<?php

declare(strict_types=1);

namespace Drupal\tmgmt_laratranslate\Service;

use Psr\Log\LoggerInterface;

/**
 * Service for HTML validation.
 */
class TextSplitterValidator {

  public function __construct(
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * Validates that HTML is well-formed.
   *
   * NOTE: This is a utility method for optional validation.
   * Used to validate the final reassembled result.
   *
   * Use cases:
   * - Validate final translation output (optional)
   * - Debug HTML structure issues
   * - Quality assurance logging
   *
   * @param string $html
   *   The HTML to validate.
   *
   * @return bool
   *   TRUE if HTML is well-formed, FALSE otherwise.
   */
  public function validateHtml(string $html): bool {
    if ($html === '') {
      $this->logger->warning('Empty HTML provided for validation.');

      return TRUE;
    }

    $dom = new \DOMDocument();
    libxml_use_internal_errors(TRUE);

    // Load as HTML fragment (no doctype/html/body).
    $result = $dom->loadHTML(
      '<?xml encoding="UTF-8">' . $html,
      LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD
    );

    $errors = libxml_get_errors();
    libxml_clear_errors();
    libxml_use_internal_errors(FALSE);

    return $result && $errors === [];
  }

}
