<?php

declare(strict_types=1);

namespace Drupal\tmgmt_laratranslate\Enum;

/**
 * Enum for separator retention strategies.
 *
 * Defines how separators should be handled when splitting text:
 * - Yes: Keep separators as separate elements in the output
 * - No: Discard separators completely
 * - Start: Attach separators to the start of the following text chunk
 * - End: Attach separators to the end of the preceding text chunk.
 */
enum KeepSeparator: string {

  // Keep separators as separate elements.
  case Yes = 'yes';

  // Discard separators completely.
  case No = 'no';

  // Attach separators to the start of following text.
  case Start = 'start';

  // Attach separators to the end of preceding text.
  case End = 'end';

  /**
   * Create enum from legacy boolean or string value.
   *
   * @param bool|string|\Drupal\tmgmt_laratranslate\Enum\KeepSeparator $value
   *   The value to convert: TRUE, FALSE, "yes", "no", "start", "end",
   *   or a KeepSeparator instance.
   *
   * @return self
   *   The corresponding KeepSeparator enum case.
   *
   * @throws \InvalidArgumentException
   *   If the value cannot be converted to a valid enum case.
   */
  public static function fromValue(bool|string|self $value): self {
    // Already an enum instance.
    if ($value instanceof self) {
      return $value;
    }

    // Handle boolean values for backward compatibility.
    if (is_bool($value)) {
      return $value ? self::Yes : self::No;
    }

    // Handle string values.
    return match (strtolower($value)) {
      'yes', 'true', '1' => self::Yes,
      'no', 'false', '0' => self::No,
      'start' => self::Start,
      'end' => self::End,
      default => throw new \InvalidArgumentException(
        sprintf('Invalid keep_separator value: "%s". Must be yes, no, start, or end.', $value)
      ),
    };
  }

}
