<?php

declare(strict_types=1);

namespace Drupal\tmgmt_laratranslate;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\key\KeyRepositoryInterface;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginUiBase;
use Drupal\tmgmt_laratranslate\Trait\FormMessageTrait;
use Lara\LaraCredentials;
use Lara\Translator as LaraTranslator;

/**
 * Default ui controller class for Lara Translate translator plugin.
 */
class LaraTranslatorUi extends TranslatorPluginUiBase {

  use FormMessageTrait;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface|null
   */
  protected ?KeyRepositoryInterface $keyRepository = NULL;

  /**
   * The Lara translator instance for API calls.
   *
   * @var \Lara\Translator|null
   */
  protected ?LaraTranslator $laraTranslator = NULL;

  /**
   * Logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<array-key, mixed> $configuration
   * @phpstan-param string $plugin_id
   * @phpstan-param array<array-key, mixed> $plugin_definition
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->keyRepository = $this->getKeyRepository();
    // @todo Currently we cannot use the Dependency Injection in this class,
    // because TranslatorPluginUiBase doesn't support it; so we have to load
    // the path service directly.
    // @phpstan-ignore-next-line
    $this->logger = \Drupal::logger('tmgmt_laratranslate');
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<array-key, mixed> $form
   * @phpstan-param \Drupal\Core\Form\FormStateInterface $form_state
   * @phpstan-return array<array-key, mixed>
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $doc_link = Link::fromTextAndUrl(
      $this->t('Documentation'),
      Url::fromUri(
        'https://developers.laratranslate.com/docs/translate-text',
        [
          'absolute' => TRUE,
          'attributes' => ['target' => '_blank'],
        ],
      ),
    );
    $form['intro'] = [
      '#type' => 'markup',
      '#markup' => $this->t('For more information on setting up and using Lara Translate, please refer to the Lara Translate @doc_link.', [
        '@doc_link' => $doc_link->toString(),
      ]),
    ];

    // @phpstan-ignore-next-line
    $translator = $form_state->getFormObject()->getEntity();
    assert($translator instanceof TranslatorInterface);

    // Create a link to the key creation page with destination parameter.
    // @todo Currently we cannot use the Dependency Injection in this class,
    // because TranslatorPluginUiBase doesn't support it; so we have to load
    // the path service directly.
    // @phpstan-ignore-next-line
    $current_path = \Drupal::service('path.current')->getPath();
    $key_add_url = Url::fromRoute('entity.key.add_form', [], [
      'query' => ['destination' => $current_path],
    ]);
    $key_add_link = Link::fromTextAndUrl($this->t('create keys'), $key_add_url);

    // Credentials fieldset.
    $form['credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('API Credentials'),
      '#description' => $this->t('Configure your Lara Translate API credentials. You must create keys using the @key_add_link before you can select them here.', [
        '@key_add_link' => $key_add_link->toString(),
      ]),
      '#open' => TRUE,
    ];

    // Check if Key module is available.
    if ($this->keyRepository === NULL) {
      $form['credentials']['key_module_warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('The Key module is required but not available. Please install and enable the Key module.') . '</div>',
      ];

      return $form;
    }

    // Get available keys for selection.
    $key_options = $this->getKeyOptions();
    // Get current credentials for API calls.
    $current_access_key_id = $translator->getSetting(['credentials', 'lara_access_key_id']);
    $current_access_key_secret = $translator->getSetting(['credentials', 'lara_access_key_secret']);

    if ($key_options === []) {
      $form['credentials']['no_keys_warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--warning">' . $this->t('No keys are configured. Please @key_add_link before configuring this translator.', [
          '@key_add_link' => $key_add_link->toString(),
        ]) . '</div>',
      ];
    }

    $form['credentials']['lara_access_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Key ID'),
      '#description' => $this->t('Select the key containing your Lara Translate Access Key ID.'),
      '#options' => $key_options,
      '#empty_option' => $this->t('- Select a key -'),
      '#default_value' => $current_access_key_id,
      '#required' => TRUE,
      '#disabled' => $key_options === [],
    ];

    $form['credentials']['lara_access_key_secret'] = [
      '#type' => 'select',
      '#title' => $this->t('Access Key Secret'),
      '#description' => $this->t('Select the key containing your Lara Translate Access Key Secret.'),
      '#options' => $key_options,
      '#empty_option' => $this->t('- Select a key -'),
      '#default_value' => $current_access_key_secret,
      '#required' => TRUE,
      '#disabled' => $key_options === [],
    ];

    // Add advanced options if user has permission.
    $this->addAdvancedOptionElements($form, $translator);

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<array-key, mixed> $form
   * @phpstan-param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $settings = $form_state->getValue('settings');

    // Always validate credentials as they are visible to all users.
    if ($settings['credentials']['lara_access_key_id'] === $settings['credentials']['lara_access_key_secret']) {
      $form_state->setErrorByName('credentials][lara_access_key_id', $this->t('Access Key ID and Access Key Secret must be different keys.'));
    }
  }

  /**
   * Get available key options for selection.
   *
   * @return array<int|string, string>
   *   An array of key options keyed by key ID.
   */
  protected function getKeyOptions(): array {
    $options = [];

    if ($this->keyRepository === NULL) {
      return $options;
    }

    try {
      $keys = $this->keyRepository->getKeys();
      foreach ($keys as $key) {
        $id = $key->id();
        if ($id !== NULL && $id !== '') {
          $label = $key->label();
          $options[$id] = ($label instanceof TranslatableMarkup) ? $label->render() : (string) $label;
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error loading keys: @message', ['@message' => $e->getMessage()]);
    }

    return $options;
  }

  /**
   * Get the key repository service.
   *
   * Since TranslatorPluginUiBase doesn't support dependency injection,
   * we load the service manually.
   *
   * @return \Drupal\key\KeyRepositoryInterface|null
   *   The key repository service or null if not available.
   */
  protected function getKeyRepository(): ?KeyRepositoryInterface {
    if ($this->keyRepository === NULL) {
      try {
        // @todo Currently we cannot use the Dependency Injection in this
        // class, because TranslatorPluginUiBase doesn't support it; so we
        // have to load the path service directly.
        // @phpstan-ignore-next-line
        $this->keyRepository = \Drupal::service('key.repository');
      }
      catch (\Exception $e) {
        $this->logger->error(
          'Could not load key repository service: @message',
          ['@message' => $e->getMessage()],
        );

        return NULL;
      }
    }

    return $this->keyRepository;
  }

  /**
   * Get Lara SDK translator instance for API calls.
   *
   * @param string|null $access_key_id
   *   The access key ID, or null to get from saved settings.
   * @param string|null $access_key_secret
   *   The access key secret, or null to get from saved settings.
   *
   * @return \Lara\Translator|null
   *   The Lara translator instance or null if credentials are not available.
   */
  protected function getLaraTranslator(?string $access_key_id = NULL, ?string $access_key_secret = NULL): ?LaraTranslator {
    if ($this->keyRepository === NULL) {
      return NULL;
    }

    try {
      // If not provided, try to get from current form values or saved settings.
      if ($access_key_id === NULL || $access_key_secret === NULL) {
        // Can't fetch without credentials.
        return NULL;
      }

      // Get the actual key values from the key repository.
      $access_key_id_key = $this->keyRepository->getKey($access_key_id);
      $access_key_secret_key = $this->keyRepository->getKey($access_key_secret);

      if ($access_key_id_key === NULL || $access_key_secret_key === NULL) {
        return NULL;
      }

      $access_key_id_value = $access_key_id_key->getKeyValue();
      $access_key_secret_value = $access_key_secret_key->getKeyValue();

      if ($access_key_id_value === '' || $access_key_secret_value === '') {
        return NULL;
      }

      // Create Lara credentials and translator.
      $credentials = new LaraCredentials($access_key_id_value, $access_key_secret_value);

      return new LaraTranslator($credentials);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to initialize Lara Translator for API call: @message', ['@message' => $e->getMessage()]);

      return NULL;
    }
  }

  /**
   * Get available resources from Lara API.
   *
   * @param string $resource_type
   *   The type of resource to fetch ('memories' or 'glossaries').
   * @param string|null $access_key_id
   *   The access key ID.
   * @param string|null $access_key_secret
   *   The access key secret.
   *
   * @return array<string, string>
   *   Array of resource options keyed by resource ID.
   */
  protected function getLaraResourceOptions(string $resource_type, ?string $access_key_id = NULL, ?string $access_key_secret = NULL): array {
    $options = [];

    if ($access_key_id === NULL || $access_key_secret === NULL) {
      return $options;
    }

    $lara_translator = $this->getLaraTranslator($access_key_id, $access_key_secret);
    if ($lara_translator === NULL) {
      return $options;
    }

    try {
      $resources = match ($resource_type) {
        'memories' => $lara_translator->memories->getAll(),
        'glossaries' => $lara_translator->glossaries->getAll(),
        default => throw new \InvalidArgumentException("Unsupported resource type: {$resource_type}"),
      };

      foreach ($resources as $resource) {
        // Adjust these property names based on actual Lara API response.
        $resource_id = $resource->getId();
        $resource_name = $resource->getName();

        if ($resource_id !== '') {
          $options[$resource_id] = $resource_name;
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching @resource_type from Lara API: @message', [
        '@resource_type' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
    }

    return $options;
  }

  /**
   * Get available memories from Lara API.
   *
   * @param string|null $access_key_id
   *   The access key ID.
   * @param string|null $access_key_secret
   *   The access key secret.
   *
   * @return array<string, string>
   *   Array of memory options keyed by memory ID.
   */
  protected function getMemoryOptions(?string $access_key_id = NULL, ?string $access_key_secret = NULL): array {
    return $this->getLaraResourceOptions('memories', $access_key_id, $access_key_secret);
  }

  /**
   * Get available glossaries from Lara API.
   *
   * @param string|null $access_key_id
   *   The access key ID.
   * @param string|null $access_key_secret
   *   The access key secret.
   *
   * @return array<string, string>
   *   Array of glossary options keyed by glossary ID.
   */
  protected function getGlossaryOptions(?string $access_key_id = NULL, ?string $access_key_secret = NULL): array {
    return $this->getLaraResourceOptions('glossaries', $access_key_id, $access_key_secret);
  }

  /**
   * Check if the current user has permission to configure advanced options.
   *
   * @return bool
   *   TRUE if the user has permission to configure advanced translation
   *   options, FALSE otherwise.
   */
  protected function hasAdvancedOptionsAccess(): bool {
    // @todo Currently we cannot use the Dependency Injection in this class,
    // because TranslatorPluginUiBase doesn't support it; so we have to load
    // the path service directly.
    // @phpstan-ignore-next-line
    return \Drupal::currentUser()->hasPermission('configure advanced lara translate options');
  }

  /**
   * Add advanced option elements to the form.
   *
   * @param array<array-key, mixed> $form
   *   The form array to modify.
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator entity.
   */
  private function addAdvancedOptionElements(array &$form, TranslatorInterface $translator): void {
    // Check if user has permission to configure advanced options.
    $has_advanced_access = $this->hasAdvancedOptionsAccess();

    $no_admin_message = $this->t('Your account has <b>no permissions to edit</b> the Lara Translate advanced options, please contact the Administrator.');

    // Translation options fieldset - always show but disable for users without
    // permissions.
    $form['translation_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Translation Options'),
      '#description' => $has_advanced_access
        ? $this->t('Configure additional options for Lara Translate processing.')
        : $no_admin_message,
    ];

    $form['translation_options']['instructions'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Translation Instructions'),
      '#description' => $this->t('Enter specific translation instructions, one per line. These will be provided to the translation service.'),
      '#default_value' => $translator->getSetting(['translation_options', 'instructions']),
      '#rows' => 3,
      '#disabled' => !$has_advanced_access,
    ];

    $form['translation_options']['style'] = [
      '#type' => 'select',
      '#title' => $this->t('Translation Style'),
      '#description' => $this->t('Select the translation style to be used.'),
      '#options' => [
        'fluid' => $this->t('Fluid'),
        'faithful' => $this->t('Faithful'),
        'creative' => $this->t('Creative'),
      ],
      '#default_value' => $translator->getSetting(['translation_options', 'style']),
      '#disabled' => !$has_advanced_access,
    ];

    // Get current credentials for API calls.
    $current_access_key_id = $translator->getSetting(['credentials', 'lara_access_key_id']);
    $current_access_key_secret = $translator->getSetting(['credentials', 'lara_access_key_secret']);
    // Translation memories and glossaries - show if credentials are configured.
    if ($current_access_key_id !== NULL && $current_access_key_secret !== NULL) {
      $form['memories'] = [
        '#type' => 'details',
        '#title' => $this->t('Translation Memories'),
        '#description' => $has_advanced_access
          ? $this->t('Select which translation memories to use during translation. These will be used for the "adaptTo" option.')
          : $no_admin_message,
        '#open' => FALSE,
      ];

      $memory_options = $this->getMemoryOptions($current_access_key_id, $current_access_key_secret);
      if ($memory_options !== []) {
        $form['memories']['adaptTo'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Available Memories'),
          '#description' => $this->t('Select the translation memories you want to use. Multiple memories can be selected.'),
          '#options' => $memory_options,
          '#default_value' => $translator->getSetting(['memories', 'adaptTo']),
          '#disabled' => !$has_advanced_access,
        ];
      }
      else {
        $this->addMessage(
          $form,
          ['memories', 'warning'],
          $this->t('No translation memories are available for your account, or there was an error connecting to the Lara API.'),
          'warning',
        );
      }

      // Glossaries fieldset.
      $form['glossaries_section'] = [
        '#type' => 'details',
        '#title' => $this->t('Glossaries'),
        '#description' => $has_advanced_access
          ? $this->t('Select which glossaries to use during translation.')
          : $no_admin_message,
        '#open' => FALSE,
      ];

      $glossary_options = $this->getGlossaryOptions($current_access_key_id, $current_access_key_secret);
      if ($glossary_options !== []) {
        $form['glossaries_section']['glossaries'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Available Glossaries'),
          '#description' => $has_advanced_access
            ? $this->t('Select the glossaries you want to use. Multiple glossaries can be selected.')
            : $this->t('Your account has no permissions to edit the Lara Translate advanced options, please contact the Administrator.'),
          '#options' => $glossary_options,
          '#default_value' => $translator->getSetting(['glossaries_section', 'glossaries']),
          '#disabled' => !$has_advanced_access,
        ];
      }
      else {
        $this->addMessage(
          $form,
          ['glossaries_section', 'no_glossaries_warning'],
          $this->t('No glossaries are available for your account, or there was an error connecting to the Lara API.'),
          'warning',
        );
      }
    }
    else {
      $this->addMessage(
        $form,
        ['api_warning'],
        $this->t('Configure your API credentials first to load available translation memories and glossaries.'),
        'warning',
      );
    }
  }

}
