<?php

namespace Drupal\tmgmt_reverso\Plugin\tmgmt\Translator;

use Drupal\Component\Utility\Html;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\Translator;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\TranslatorPluginBase;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Psr7\Request;
use http\Exception\InvalidArgumentException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Reverso translator plugin.
 *
 * @TranslatorPlugin(
 *   id = "reverso",
 *   label = @Translation("Reverso"),
 *   description = @Translation("Reverso Translator service."),
 *   ui = "Drupal\tmgmt_reverso\ReversoTranslatorUi",
 * )
 */
class ReversoTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface, ContinuousTranslatorInterface {

  /**
   * Max number of text queries for translation sent in one request.
   *
   * @var int
   */
  const qChunkSize = 5;

  /**
   * Name of parameter that contains source string to be translated.
   *
   * @var string
   */
  const qParamName = 'q';

  /**
   * Mapping language.
   *
   * @var array
   */
  protected $mappingLanguages = [
    'fr' => 'fra', 'en' => 'eng', 'it' => 'ita', 'es' => 'spa',
    'de' => 'deu', 'pt' => 'por', 'nl' => 'nld', 'ru' => 'rus',
    'pl' => 'pol', 'cs' => 'ces', 'sk' => 'slk', 'hu' => 'hun',
    'ro' => 'ron', 'bg' => 'bul', 'hr' => 'hrv', 'sl' => 'slv',
    'et' => 'est', 'lv' => 'lav', 'lt' => 'lit', 'el' => 'ell',
    'sv' => 'swe', 'da' => 'dan', 'no' => 'nor', 'fi' => 'fin',
    'is' => 'isl', 'ga' => 'gle', 'mt' => 'mlt', 'cy' => 'cym',
    'eu' => 'eus', 'ca' => 'cat', 'gl' => 'glg', 'zh' => 'zho',
    'ja' => 'jpn', 'ko' => 'kor', 'th' => 'tha', 'vi' => 'vie',
    'hi' => 'hin', 'bn' => 'ben', 'ur' => 'urd', 'ar' => 'ara',
    'he' => 'heb', 'fa' => 'fas', 'tr' => 'tur', 'uk' => 'ukr',
    'be' => 'bel', 'mk' => 'mkd', 'sr' => 'srp', 'bs' => 'bos',
    'sq' => 'sqi', 'af' => 'afr', 'sw' => 'swa', 'zu' => 'zul',
  ];

  /**
   * Guzzle HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;


  /**
   * @param \GuzzleHttp\ClientInterface $client
   * @param array $configuration
   * @param $plugin_id
   * @param $plugin_definition
   */
  public function __construct(ClientInterface $client, array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->client = $client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('http_client'),
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * Request Translation method
   *
   * @param \Drupal\tmgmt\JobInterface $job
   * @return void
   */
  public function requestTranslation(JobInterface $job) {
    $this->requestJobItemsTranslation($job->getItems());
    if (!$job->isRejected()) {
      $job->finished();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function requestJobItemsTranslation(array $jobItems) {
    $job = reset($jobItems)->getJob();
    $dataService = \Drupal::service('tmgmt.data');
    foreach ($jobItems as $jobItem) {
      if ($job->isContinuous()) {
        $jobItem->active();
      }
      $data = $dataService->filterTranslatable($jobItem->getData());
      $translation = [];
      $q = [];
      $keys_sequence = [];
      $i = 0;
      foreach ($data as $key => $value) {
        $q[] = $value['#text'];
        $keys_sequence[] = $key;
      }
      try {
        foreach (array_chunk($q, self::qChunkSize) as $_q) {
          $result = $this->reversoRequestTranslation($job, $_q);
          // Collect translated texts with use of initial keys.
          foreach ($result['data']['translations'] as $translated) {
            $translation[$keys_sequence[$i]]['#text'] = Html::decodeEntities($translated["TranslatedText"]);
            $i++;
          }
        }
        $jobItem->addTranslatedData($dataService->unflatten($translation));

      }catch (TMGMTException $e) {
        $job->rejected('Translation has been rejected with following error: @error',
          array('@error' => $e->getMessage()), 'error');
      }
    }
  }

  /**
   * Overrides TMGMTDefaultTranslatorPluginController::hasCheckoutSettings().
   */
  public function hasCheckoutSettings(JobInterface $job) {
    return FALSE;
  }

  /**
   * Helper method to do translation request.
   *
   * @param Job $job
   *   TMGMT Job to be used for translation.
   * @param array|string $q
   *   Text/texts to be translated.
   *
   * @return array
   *   Userialized JSON containing translated texts.
   */
  protected function reversoRequestTranslation(Job $job, $q): array {
    $translator = $job->getTranslator();
    return $this->doRequest($translator, [
      'source' => $job->getRemoteSourceLanguage(),
      'target' => $job->getRemoteTargetLanguage(),
      self::qParamName => $q,
    ], [
      'headers' => array(
        'Content-Type' => 'application/json',
      ),
    ]);
  }

  /**
   * Local method to do request to Reverso Translate service.
   *
   * @param Translator $translator
   *   The translator entity to get the settings from.
   * @param string $action
   *   Action to be performed [translate, languages, detect]
   * @param array $request_query
   *   (Optional) Additional query params to be passed into the request.
   * @param array $options
   *   (Optional) Additional options that will be passed into drupal_http_request().
   *
   * @return array object
   *   Unserialized JSON response from Reverso.
   *
   * @throws TMGMTException
   *   - Invalid action provided
   *   - Unable to connect to the Reverso Service
   *   - Error returned by the Reverso Service
   */
  protected function doRequest(Translator $translator, array $requestQuery = array(), array $options = array()) {
    $this->validateRequestQuery($requestQuery);

    $headers = $this->buildAuthHeaders($translator);
    $source = $this->getLanguageCode($requestQuery['source']);
    $target = $this->getLanguageCode($requestQuery['target']);

    $result = ['data' => ['translations' => []]];

    foreach ($requestQuery['q'] as $query) {
      try {
        $translatedText = $this->translateSingleQuery($translator, $query, $source, $target, $headers);
        $result['data']['translations'][] = ['TranslatedText' => $translatedText];
      } catch (BadResponseException $e) {
        throw new TMGMTException('Reverso service returned following error: @error', ['@error' => $e->getMessage()]);
      }
    }
    return $result;
  }

  /**
   * Validate request params
   * @param array $requestQuery
   * @return void
   */
  private function validateRequestQuery(array $requestQuery): void {
    $requiredFields = ['source', 'target', 'q'];

    foreach ($requiredFields as $field) {
      if (empty($requestQuery[$field])) {
        throw new InvalidArgumentException("Missing required field: {$field}");
      }
    }

    if (!is_array($requestQuery['q'])) {
      throw new InvalidArgumentException("Field 'q' must be an array");
    }
  }

  /**
   * Get mapped langue code
   */
  private function getLanguageCode(string $language): string {
    if (!isset($this->mappingLanguages[$language])) {
      throw new InvalidArgumentException("Unsupported language: {$language}");
    }
    return $this->mappingLanguages[$language];
  }

  /**
   * Build headers authentication
   */
  private function buildAuthHeaders(Translator $translator): array
  {
    $date = date("D, d M Y H:i:s O");
    $username = $translator->getSetting('username');
    $password = $translator->getSetting('password');

    if (empty($username) || empty($password)) {
      throw new InvalidArgumentException("Username and password are required");
    }

    $signature = hash_hmac('sha1', $username . $date, $password);

    return [
      'Created' => $date,
      'Username' => $username,
      'Signature' => $signature,
    ];
  }

  /**
   * Translate single query
   *
   * @param Translator $translator
   * @param string $query
   * @param string $source
   * @param string $target
   * @param array $headers
   * @return string
   */
  private function translateSingleQuery(Translator $translator, string $query, string $source, string $target, array $headers): string {
    $url = $this->buildTranslationUrl($translator, $query, $source, $target);
    $request = new Request('POST', $url, $headers);

    $response = $this->client->send($request, ['body' => $query]);
    $output = json_decode($response->getBody()->getContents(), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
      throw new TMGMTException('Invalid JSON response from Reverso service');
    }

    if (isset($output['TranslatedStream'])) {
      return base64_decode($output['TranslatedStream']);
    }

    if (isset($output['TranslatedText'])) {
      return $output['TranslatedText'];
    }

    throw new TMGMTException('No translation found in response');
  }

  /**
   * Build the translation Url
   *
   * @param Translator $translator
   * @param string $query
   * @param string $source
   * @param string $target
   * @return string
   */
  private function buildTranslationUrl(Translator $translator, string $query, string $source, string $target): string {
    $baseUrl = $translator->getSetting('url');
    $direction = "{$source}-{$target}";
    $isHtml = $query !== strip_tags($query);

    if ($isHtml) {
      return "{$baseUrl}/v1/TranslateStream/direction={$direction}/inputExtension=html";
    }

    return "{$baseUrl}/v1/TranslateText/direction={$direction}";
  }
}
