<?php

/**
 * @file
 * Contains \Drupal\tmgmt_smartcat\Plugin\tmgmt\Translator\SmartcatTranslator.
 */

namespace Drupal\tmgmt_smartcat\Plugin\tmgmt\Translator;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt_content\Plugin\tmgmt\Source\ContentEntitySource;
use Drupal\tmgmt_smartcat\API\API;
use Drupal\tmgmt_smartcat\API\Clients\IHub;
use Drupal\tmgmt_smartcat\API\Clients\Smartcat;
use Drupal\tmgmt_smartcat\Data\Project;
use Drupal\tmgmt_smartcat\Data\ProjectMT;
use Drupal\tmgmt_smartcat\Data\TranslatableItem;
use Drupal\tmgmt_smartcat\Data\TranslatableItemSegment;
use Drupal\tmgmt_smartcat\Database\Migrations;
use Drupal\tmgmt_smartcat\Services\LocaleMapper;
use Drupal\tmgmt_smartcat\Services\SmartcatDocument;
use Drupal\tmgmt_smartcat\Services\SmartcatProjectStatus;
use GuzzleHttp\Exception\RequestException;

/**
 * Smartcat translation plugin controller.
 *
 * @TranslatorPlugin(
 *   id = "smartcat",
 *   label = @Translation("Smartcat"),
 *   description = @Translation("..."),
 *   ui = "Drupal\tmgmt_smartcat\SmartcatTranslatorUi",
 *   logo = "icons/smartcat.png",
 * )
 */
class SmartcatTranslator extends TranslatorPluginBase implements ContinuousTranslatorInterface
{
    private Smartcat $sc;

    private IHub $ihub;

    /**
     * Sending documents to Smartcat
     *
     * @return JobInterface
     *
     * @throws TMGMTException
     * @throws EntityStorageException
     */
    public function requestTranslation(JobInterface $job)
    {
        // FIXME: Replace with hook_update_N()
        Migrations::run();

        return $this->requestJobItemsTranslation($job->getItems());
    }

    /**
     * Create a Smartcat project and submit items from a job
     *
     * @param  JobItemInterface[]  $job_items
     * @return JobInterface
     *
     * @throws EntityStorageException
     * @throws TMGMTException
     */
    public function requestJobItemsTranslation(array $job_items)
    {
        /** @var Job $job */
        $job = reset($job_items)->getJob();
        $translator = $job->getTranslator();
        $importExistingTranslations = (bool)$translator->getSetting('import_existing_translations');

        $this->api($job);

        $project = $this->getOrCreateProject($job);

        if (! $job->getReference()) {
            $job->addMessage('Smartcat project has been initialized. Project ID: @id', ['@id' => $project->getId()], 'debug');
            $job->reference = $project->getId();
            $job->save();
        }

        if (is_null($project)) {
            return $job;
        }

        $translatableItems = $this->prepareDataForSending($job_items, $importExistingTranslations);

        foreach ($translatableItems as $translatableItem) {
            try {
                $response = $this->ihub->import(
                    $translatableItem,
                    $project->getId(),
                    LocaleMapper::getProviderLocale($job->getSourceLangcode(), $job->getTranslator()),
                    LocaleMapper::getProviderLocale($job->getTargetLangcode(), $job->getTranslator())
                );

                $jobItem = $translatableItem->getJobItem();

                if (! is_null($response)) {
                    $jobItem->active();
                    $jobItem->save();

                    if ($job->isContinuous()) {
                        $document = (new SmartcatDocument())->findByJobItemId($jobItem->id());

                        if (is_null($document)) {
                            $document = new SmartcatDocument($jobItem->id(), $response->getDocumentId());
                            $document->store();
                        }
                    }
                }
            } catch (RequestException $e) {
                \Drupal::logger('tmgmt_smartcat')->error('@message | Job ID: @job_id | Job Item ID: @job_item_id | Project ID: @project_id | Message: @message | Response: @response', [
                    '@message' => 'An error occurred while importing a document into Smartcat',
                    '@job_id' => $job->id(),
                    '@job_item_id' => $translatableItem->getJobItem()->id(),
                    '@project_id' => $project->getId(),
                    '@response' => $e->getResponse()->getBody()->getContents(),
                ]);
            }
        }

        if (! $job->isContinuous()) {
            $job->submitted('The job has been submitted to Smartcat. Project ID: @id', ['@id' => $project->getId()]);
        }

        return $job;
    }

    /**
     * Checking the connection with Smartcat
     */
    public function checkAvailable(TranslatorInterface $translator): AvailableResult
    {
        if ($translator->getSetting('account_id') && $translator->getSetting('api_key')) {
            return AvailableResult::yes();
        }

        return AvailableResult::no(t('@translator is not available. Make sure it is properly <a href=:configured>configured</a>.', [
            '@translator' => $translator->label(),
            // ':configured' => $translator->url(),
        ]));
    }

    private function getTemplates()
    {
      return $this->ihub->getTemplates();
    }

    /**
     * Prepare data for sending
     *
     * @param  JobItemInterface[]  $jobItems
     * @return array<TranslatableItem>
     */
    protected function prepareDataForSending(array $jobItems, bool $importExistingTranslations): array
    {
        $translatableItems = [];

        foreach ($jobItems as $jobItem) {
            $translatableItem = new TranslatableItem();

            $translatableItem->setId($jobItem->id());
            $translatableItem->setName($jobItem->getSourceLabel());
            $translatableItem->setJobItem($jobItem);

            $items = $this->prepareSingleDataForSending($jobItem->getJob(), 'json', $jobItem->id());
            
            // Get existing translations for target language
            $existingTranslations = $importExistingTranslations ? $this->getExistingTranslations($jobItem) : '';

            $segments = [];

            foreach ($items as $itemKey => $value) {
                $existingTranslation = $existingTranslations[$itemKey] ?? '';
                $segments[] = new TranslatableItemSegment($itemKey, $value, $existingTranslation);
            }

            $translatableItem->setSegments($segments);

            $translatableItems[] = $translatableItem;
        }

        return $translatableItems;
    }

    /**
     * Prepare single data for sending
     *
     * @return array|string
     */
    protected function prepareSingleDataForSending(JobInterface $job, $type = 'json', $keepOnlyThisIndex = null)
    {
        $data_service = \Drupal::service('tmgmt.data');

        $data = array_filter($data_service->flatten($job->getData()), function ($value) {
            return ! (empty($value['#text']) || (isset($value['#translate']) && $value['#translate'] === false));
        });

        if ($type === 'json') {
            $items = [];
        } else {
            $items = '';
        }

        foreach ($data as $key => $value) {
            // when we are in multiple source files mode,
            // for proper translation return,
            // we need to keep keys like this: "13][title][0][value"
            // they need to start with the job item ID.
            // for this, for each item, we flatten the whole job (instead of job item)
            // and then remove the other job items from the flattened data, so that the file will contain
            // only the current job item.
            if ($keepOnlyThisIndex !== null) {
                if (strpos($key, $keepOnlyThisIndex.$data_service::TMGMT_ARRAY_DELIMITER) !== 0) {
                    continue;
                }
            }

            if ($type === 'json') {
                $items[$key] = $value['#text'];
            } else {
                $items .= str_replace(
                    ['@key', '@text'],
                    [$key, $value['#text']],
                    '<item key="@key"><text type="text/html"><![CDATA[@text]]></text></item>'
                );
            }
        }

        if ($type === 'json') {
            return $items;
        } else {
            return '<items>'.$items.'</items>';
        }
    }

    /**
     * @throws TMGMTException
     */
    private function getOrCreateProject(JobInterface $job): ?Project
    {
        if ($job->getReference()) {
            /** @var SmartcatProjectStatus $projectStatus */
            $projectStatus = \Drupal::service('tmgmt_smartcat.project_status');

            if (! $projectStatus->check($job)) {
                return null;
            }

            return $this->sc->getProject($job->getReference());
        } else {
            $template = $job->getSetting('project_template_id');

            $workflowStage = $job->getSetting('workflow_stage');
            $workflowStages = $template ? null : $this->getWorkflowStage($workflowStage);

            $project = $this->ihub->getOrCreateProject(
                null, "Drupal TMGMT {$job->label()}",
                LocaleMapper::getProviderLocale($job->getSourceLangcode(), $job->getTranslator()),
                [LocaleMapper::getProviderLocale($job->getTargetLangcode(), $job->getTranslator())],
                $workflowStages,
                $template,
            );

            if ($workflowStage !== 'manual-translation') {
                $this->setupMtEngineToSmartcatProject($project);
            }
        }

        return $project;
    }

    private function setupMtEngineToSmartcatProject(Project $project)
    {
        $mts = $this->sc->getAvailableProjectMT($project->getId());

        $selectedMT = null;

        $intelligentRouting = array_filter($mts, function ($mt) {
            return $mt->isIntelligentRouting();
        });

        /** @var ProjectMT|null $intelligentRouting */
        $intelligentRouting = $intelligentRouting[0] ?? null;

        if (
            ! is_null($intelligentRouting) &&
            $intelligentRouting->languagesCount() === $project->targetLocalesCount()
        ) {
            $selectedMT = $intelligentRouting;
        }

        if (is_null($selectedMT)) {
            $google = array_filter($mts, function ($mt) {
                return $mt->isGoogle();
            });

            /** @var ProjectMT|null $google */
            $google = $google[0] ?? null;

            if (
                ! is_null($google) &&
                $google->languagesCount() === $project->targetLocalesCount()
            ) {
                $selectedMT = $google;
            }
        }

        if (is_null($selectedMT)) {
            foreach ($mts as $mt) {
                if ($mt->languagesCount() === $project->targetLocalesCount()) {
                    $selectedMT = $mt;
                    break;
                }
            }
        }

        try {
            $this->sc->setupMtEngine($project->getId(), [$selectedMT->toArray()]);
        } catch (RequestException $exception) {
            \Drupal::logger('tmgmt_smartcat')->error("Failed to setup MT engine to project {$project->getId()}. Response: @response", [
                '@response' => $exception->getResponse()->getBody()->getContents(),
            ]);
        }

        $rule = ['ruleType' => 'MT', 'order' => 1];

        $translationStage = array_filter($project->getWorkflowStages(), function ($stage) {
            return $stage->isTranslation();
        });

        $translationStage = $translationStage[0] ?? null;

        if (! is_null($translationStage)) {
            $rule['confirmAtWorkflowStep'] = $translationStage->getId();
        }

        try {
            $this->sc->setupPreTranslationRules($project->getId(), [$rule]);
        } catch (RequestException $exception) {
            \Drupal::logger('tmgmt_smartcat')->error('Failed finding translation workflow step. Pretransaltion rule will be without automatic confirmation. Response: @response', [
                '@response' => $exception->getResponse()->getBody()->getContents(),
            ]);
        }
    }

    private function getWorkflowStage(string $name): array
    {
        $stages = [
            'mt' => [1],
            'mt-postediting' => [1, 7],
            'manual-translation' => [],
        ];

        return $stages[$name] ?? [];
    }

    /**
     * Get existing translations for target language
     *
     * @param JobItemInterface $jobItem
     * @return array
     */
    protected function getExistingTranslations(JobItemInterface $jobItem): array
    {
        $existingTranslations = [];
        $job = $jobItem->getJob();
        $targetLangcode = $job->getTargetLangcode();
        
        // Check if target language exists in existing translations
        $existingLangCodes = $jobItem->getExistingLangCodes();
        
        if (!in_array($targetLangcode, $existingLangCodes)) {
            return $existingTranslations;
        }
        
        // Get source plugin to access the entity
        $sourcePlugin = $jobItem->getSourcePlugin();
        
        // For content entities, get the translated entity using public methods
        if ($sourcePlugin instanceof ContentEntitySource) {
            // Use entity type manager to load the entity directly
            $entityTypeManager = \Drupal::entityTypeManager();
            
            try {
                $entity = $entityTypeManager
                    ->getStorage($jobItem->getItemType())
                    ->load($jobItem->getItemId());
                
                if ($entity && $entity->hasTranslation($targetLangcode)) {
                    $translatedEntity = $entity->getTranslation($targetLangcode);
                    
                    // Get the same data structure as source but from translated entity
                    $translatedData = $this->getTranslatedDataFromEntity($translatedEntity, $jobItem);
                    
                    $existingTranslations = $translatedData;
                }
            } catch (\Exception $e) {
                \Drupal::logger('tmgmt_smartcat')->warning('Could not load entity for existing translations: @message', ['@message' => $e->getMessage()]);
            }
        }
        
        return $existingTranslations;
    }
    
    /**
     * Extract translated data from entity in the same format as source data
     *
     * @param \Drupal\Core\Entity\ContentEntityInterface $translatedEntity
     * @param JobItemInterface $jobItem
     * @return array
     */
    protected function getTranslatedDataFromEntity($translatedEntity, JobItemInterface $jobItem): array
    {
        $translatedData = [];
        
        try {
            // Get the data service to work with flattened data
            $data_service = \Drupal::service('tmgmt.data');
            
            // Get translated field values directly from the translated entity
            foreach ($translatedEntity->getFields() as $fieldName => $field) {
                // Skip empty fields
                if ($field->isEmpty()) {
                    continue;
                }
                
                // Check if field is translatable through field definition
                $fieldDefinition = $field->getFieldDefinition();
                if (!$fieldDefinition->isTranslatable()) {
                    continue;
                }
                
                $fieldValue = $field->getValue();
                
                // Build keys similar to TMGMT structure
                foreach ($fieldValue as $delta => $item) {
                    if (is_array($item)) {
                        foreach ($item as $property => $value) {
                            if (is_string($value) && !empty($value)) {
                                $key = $jobItem->id() . $data_service::TMGMT_ARRAY_DELIMITER . $fieldName . $data_service::TMGMT_ARRAY_DELIMITER . $delta . $data_service::TMGMT_ARRAY_DELIMITER . $property;
                                $translatedData[$key] = $value;
                            }
                        }
                    }
                }
            }
            
        } catch (\Exception $e) {
            // If we can't get structured data, fall back to empty array
            \Drupal::logger('tmgmt_smartcat')->warning('Could not extract structured translated data: @message', ['@message' => $e->getMessage()]);
        }
        
        return $translatedData;
    }

    /**
     * @throws TMGMTException
     */
    private function api(JobInterface $job)
    {
        $translator = $job->getTranslator();

        $this->sc = API::sc($translator);
        $this->ihub = API::ihub($translator);
    }
}
