<?php

namespace Drupal\tmgmt_smartling\Context;

use Drupal\Core\Entity\EntityTypeManager;
use Drupal\tmgmt_smartling\Exceptions\WrongUsernameException;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Psr\Log\LoggerInterface;

class ContextUserAuth {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * The user storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $userStorage;

  /**
   * The session manager service.
   *
   * @var \Drupal\Core\Session\SessionManagerInterface
   */
  protected $sessionManager;

  /**
   * The session.
   *
   * @var \Symfony\Component\HttpFoundation\Session\Session
   */
  protected $session;

  /**
   * @var LoggerInterface
   */
  protected $logger;

  /**
   * @var ModuleHandler
   */
  protected $moduleHandler;

  /**
   * Constructs a new ContextUserAuth object
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManager $entity_manager
   *   The user storage.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The user storage.
   * @param \Drupal\Core\Session\SessionManagerInterface $session_manager
   *   The session manager service.
   * @param \Symfony\Component\HttpFoundation\Session\Session $session
   *   The session.
   * @param \Psr\Log\LoggerInterface $logger
   */
  public function __construct(
    AccountProxyInterface $account,
    EntityTypeManager $entity_manager,
    ModuleHandlerInterface $module_handler,
    SessionManagerInterface $session_manager,
    Session $session,
    LoggerInterface $logger
  ) {
    $this->account = $account;
    $this->userStorage = $entity_manager->getStorage('user');
    $this->moduleHandler = $module_handler;
    $this->sessionManager = $session_manager;
    $this->session = $session;
    $this->logger = $logger;
  }

  /**
   * Generates a one-time login URL for the specified user.
   *
   * This creates a temporary login link that can be used once to authenticate
   * the context user. This is more reliable than session manipulation.
   *
   * @param string $name
   *   The username to create a login URL for.
   *
   * @return string
   *   One-time login URL
   *
   * @throws \Drupal\tmgmt_smartling\Exceptions\WrongUsernameException
   */
  public function generateOneTimeLoginUrl($name) {
    if (empty($name) || !($accounts = $this->userStorage->loadByProperties(['name' => $name]))) {
      throw new WrongUsernameException(\Drupal::translation()->translate('User with username "@username" was not found', ['@username' => $name]));
    }
    /** @var \Drupal\user\Entity\User $account */
    $account = reset($accounts);

    // Generate one-time login URL using Drupal's user module function
    // @phpstan-ignore-next-line - user_pass_reset_url is a global function from user.module
    $url = \user_pass_reset_url($account, ['absolute' => TRUE]) . '/login';

    $this->logger->info('Generated one-time login URL for user "@user": @url', [
      '@user' => $name,
      '@url' => $url,
    ]);

    return $url;
  }

  public function getCurrentAccount() {
    return $this->account;
  }
}
