<?php

namespace Drupal\Tests\tmgmt_smartling\Kernel;

use Drupal\tmgmt_smartling\Context\ContextUploader;
use Drupal\tmgmt_smartling\Context\ContextUserAuth;
use Drupal\tmgmt_smartling\Context\HtmlAssetInliner;
use Drupal\tmgmt_smartling\Context\TranslationJobToUrl;
use Drupal\tmgmt_smartling\Smartling\SmartlingApiWrapper;
use Psr\Log\LoggerInterface;

/**
 * Tests for ContextUploader URL host functionality.
 *
 * @group tmgmt_smartling_kernel
 */
class ContextUploaderTest extends SmartlingTestBase {

  /**
   * Test the actual applyContextUrlHost method from ContextUploader.
   */
  public function testApplyContextUrlHostMethod() {
    $contextUploader = $this->createContextUploader();

    // Test case 1: Empty context_url_host should not modify URL
    $originalUrl = 'https://example.com/node/123?arg=value';
    $settings = ['context_url_host' => ''];
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($originalUrl, $modifiedUrl, 'Empty context_url_host should not modify URL');

    // Test case 2: Valid context_url_host should replace host
    $originalUrl = 'https://example.com/node/123?arg=value';
    $expectedUrl = 'https://custom-domain.com/node/123?arg=value';
    $settings = ['context_url_host' => 'custom-domain.com'];
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($expectedUrl, $modifiedUrl, 'context_url_host should replace the host');

    // Test case 3: context_url_host with protocol should strip protocol
    $originalUrl = 'http://example.com/node/123';
    $expectedUrl = 'http://custom-domain.com/node/123';
    $settings = ['context_url_host' => 'https://custom-domain.com'];
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($expectedUrl, $modifiedUrl, 'Protocol should be stripped from context_url_host');

    // Test case 4: Should preserve query parameters and fragments
    $originalUrl = 'https://example.com/node/123?arg1=value1&arg2=value2#section1';
    $expectedUrl = 'https://custom-domain.com/node/123?arg1=value1&arg2=value2#section1';
    $settings = ['context_url_host' => 'custom-domain.com'];
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($expectedUrl, $modifiedUrl, 'Query parameters and fragments should be preserved');

    // Test case 5: HTTP protocol should be preserved
    $originalUrl = 'http://example.com/node/123';
    $expectedUrl = 'http://custom-domain.com/node/123';
    $settings = ['context_url_host' => 'custom-domain.com'];
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($expectedUrl, $modifiedUrl, 'Original protocol should be preserved');

    // Test case 6: Malformed URL should return original
    $originalUrl = 'not-a-valid-url';
    $settings = ['context_url_host' => 'custom-domain.com'];
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($originalUrl, $modifiedUrl, 'Malformed URL should return original');

    // Test case 7: Missing context_url_host key should not modify URL
    $originalUrl = 'https://example.com/node/123';
    $settings = []; // No context_url_host key
    $modifiedUrl = $contextUploader->applyContextUrlHost($originalUrl, $settings);
    $this->assertEquals($originalUrl, $modifiedUrl, 'Missing context_url_host key should not modify URL');
  }

  /**
   * Create a ContextUploader instance for testing.
   */
  protected function createContextUploader() {
    // Create minimal mocks for dependencies
    $apiWrapper = $this->getMockBuilder(SmartlingApiWrapper::class)
      ->disableOriginalConstructor()
      ->getMock();

    $urlConverter = $this->getMockBuilder(TranslationJobToUrl::class)
      ->disableOriginalConstructor()
      ->getMock();

    $authenticator = $this->getMockBuilder(ContextUserAuth::class)
      ->disableOriginalConstructor()
      ->getMock();

    $assetInliner = $this->getMockBuilder(HtmlAssetInliner::class)
      ->disableOriginalConstructor()
      ->getMock();

    $logger = $this->getMockBuilder(LoggerInterface::class)
      ->getMock();

    // Create the ContextUploader with mocked dependencies
    return new ContextUploader(
      $apiWrapper,
      $urlConverter,
      $authenticator,
      $assetInliner,
      $logger
    );
  }

}
