<?php

namespace Drupal\Tests\tmgmt_smartling\Kernel;

use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;
use Drupal\Tests\tmgmt\Kernel\TMGMTKernelTestBase;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt_smartling\Smartling\Submission\TranslationRequestManager;
use Exception;
use Smartling\Batch\BatchApi;
use Smartling\File\FileApi;

/**
 * Smartling kernel test base class.
 */
abstract class SmartlingTestBase extends TMGMTKernelTestBase {

  protected static $modules = ['tmgmt_smartling', 'tmgmt_extension_suit', 'tmgmt_file', 'file'];

  protected $apiWrapperMock;
  protected $batchApiMock;
  protected $fileApiMock;
  protected $translationRequestManagerMock;
  protected $pluginMock;
  protected $loggerMock;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->installConfig(['tmgmt_smartling', 'tmgmt_extension_suit', 'tmgmt_file', 'file']);
    $this->installSchema('file', ['file_usage']);
    $this->installEntitySchema('file');
    $this->installEntitySchema('tmgmt_remote');

    $this->pluginMock = $this->getMockBuilder('\Drupal\tmgmt_smartling\Plugin\tmgmt_file\Format\Xml')
      ->onlyMethods(['validateImport', 'import'])
      ->getMock();

    $format_manager_mock = $this->getMockBuilder('\Drupal\tmgmt_file\Format\FormatManager')
      ->disableOriginalConstructor()
      ->onlyMethods(['createInstance'])
      ->getMock();

    $format_manager_mock->expects($this->any())
      ->method('createInstance')
      ->willReturn($this->pluginMock);

    $api_factory_mock = $this->getMockBuilder('\Drupal\tmgmt_smartling\Smartling\SmartlingApiFactory')
      ->onlyMethods([])
      ->getMock();

    $this->loggerMock = $this->getMockBuilder('\Drupal\Core\Logger\LoggerChannel')
      ->onlyMethods([])
      ->disableOriginalConstructor()
      ->getMock();

    $this->batchApiMock = $this->getMockBuilder(BatchApi::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['uploadBatchFile'])
      ->getMock();

    $this->fileApiMock = $this->createMock(FileApi::class);

    $this->translationRequestManagerMock = $this->getMockBuilder(TranslationRequestManager::class)
      ->disableOriginalConstructor()
      ->onlyMethods([
        'getSubmitterName',
        'getBucketName',
        'upsertTranslationRequest',
        'getTranslationRequest',
        'commitSuccessfulDownload',
        'commitSuccessfulUpload',
        'commitError'
      ])
      ->getMock();

    $this->translationRequestManagerMock->expects($this->any())
      ->method('getSubmitterName')
      ->willReturn('test_submitter_name');

    $this->translationRequestManagerMock->expects($this->any())
      ->method('getBucketName')
      ->willReturn('test_bucket_name');

    $this->apiWrapperMock = $this->getMockBuilder('\Drupal\tmgmt_smartling\Smartling\SmartlingApiWrapper')
      ->onlyMethods([
        'createFirebaseRecord',
        'getApi',
        'executeBatch',
        'searchTranslationRequest',
        'searchTranslationSubmissions',
        'createAuditLogRecord'
      ])
      ->setConstructorArgs([$api_factory_mock, $this->loggerMock])
      ->getMock();

    \Drupal::getContainer()->set('tmgmt_smartling.smartling_api_wrapper', $this->apiWrapperMock);
    \Drupal::getContainer()->set('tmgmt_smartling.translation_request_manager', $this->translationRequestManagerMock);
    \Drupal::getContainer()->set('plugin.manager.tmgmt_file.format', $format_manager_mock);
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
  }

  /**
   * Creates a job.
   *
   * @param array $settings
   * @param string $translator
   * @return \Drupal\tmgmt\JobInterface
   */
  protected function createJobWithItems(array $settings, $translator = 'smartling') {
    $job = parent::createJob('en', 'de');

    for ($i = 1; $i < 3; $i++) {
      $job->addItem('test_source', 'test', $i);
    }

    $job->settings = $settings;
    $job->translator = $translator;

    return $job;
  }

  /**
   * Create file attachment TMGMT Job Item.
   *
   * @param JobInterface $job
   * @param FileInterface $tfile
   */
  protected function createAttachmentFileJobItem($job, $file) {
    $data = array(
      'dummy' => array(
        'deep_nesting' => array(
          '#file' => $file->id(),
          '#translate' => TRUE,
        ),
      ),
      'one_more_dummy' => array(
        'deep_nesting' => array(
          '#text' => 'Text for job item with type @type and id @id.',
          '#label' => 'Label for job item with type @type and id @id.',
          '#translate' => TRUE,
        ),
      ),
    );

    $source = \Drupal::state()->get('tmgmt.test_source_data');

    \Drupal::state()->set('tmgmt.test_source_data', array($data));

    $job->addItem('test_source', 'test', 100500 + $file->id(), array($data));

    \Drupal::state()->set('tmgmt.test_source_data', $source);
  }

  /**
   * Creates a file.
   *
   * @param string $filename
   * @return FileInterface
   */
  protected function createFile($filename) {
    $path = 'public://tmgmt_sources/' . $filename;
    $dirname = dirname($path);

    if (\Drupal::service('file_system')->prepareDirectory($dirname, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
      return \Drupal::service('file.repository')->writeData("test", 'public://tmgmt_sources/' . $filename);
    }
    else {
      throw new Exception("Failed to prepare " . $dirname);
    }
  }

  /**
   * Invokes private or protected method.
   *
   * @param $object
   * @param $methodName
   * @param array $parameters
   *
   * @return mixed
   */
  protected function invokeMethod($object, $methodName, array $parameters = []) {
    $reflection = new \ReflectionClass(get_class($object));
    $method = $reflection->getMethod($methodName);
    $method->setAccessible(true);

    return $method->invokeArgs($object, $parameters);
  }
}
